/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.drill.exec.vector;

import static org.apache.drill.shaded.guava.com.google.common.base.Preconditions.checkArgument;
import static org.apache.drill.shaded.guava.com.google.common.base.Preconditions.checkState;

import org.apache.drill.shaded.guava.com.google.common.collect.Lists;
import org.apache.drill.shaded.guava.com.google.common.collect.ObjectArrays;
import org.apache.drill.shaded.guava.com.google.common.base.Charsets;
import org.apache.drill.shaded.guava.com.google.common.collect.ObjectArrays;

import org.apache.drill.shaded.guava.com.google.common.base.Preconditions;
import io.netty.buffer.*;

import org.apache.commons.lang3.ArrayUtils;

import org.apache.drill.common.exceptions.UserException;
import org.apache.drill.exec.memory.*;
import org.apache.drill.exec.proto.SchemaDefProtos;
import org.apache.drill.exec.proto.UserBitShared;
import org.apache.drill.exec.proto.UserBitShared.DrillPBError;
import org.apache.drill.exec.proto.UserBitShared.SerializedField;
import org.apache.drill.exec.record.*;
import org.apache.drill.exec.vector.*;
import org.apache.drill.common.exceptions.*;
import org.apache.drill.exec.exception.*;
import org.apache.drill.exec.expr.holders.*;
import org.apache.drill.common.types.TypeProtos.*;
import org.apache.drill.common.types.Types;
import org.apache.drill.common.util.DrillStringUtils;
import org.apache.drill.exec.vector.complex.*;
import org.apache.drill.exec.vector.complex.reader.*;
import org.apache.drill.exec.vector.complex.impl.*;
import org.apache.drill.exec.vector.complex.writer.*;
import org.apache.drill.exec.vector.complex.writer.BaseWriter.MapWriter;
import org.apache.drill.exec.vector.complex.writer.BaseWriter.DictWriter;
import org.apache.drill.exec.vector.complex.writer.BaseWriter.ListWriter;
import org.apache.drill.exec.util.JsonStringArrayList;
import org.apache.drill.exec.memory.AllocationManager.BufferLedger;

import org.apache.drill.exec.exception.OutOfMemoryException;

import java.util.Arrays;
import java.util.Random;
import java.util.List;
import java.util.Set;

import java.io.Closeable;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.ByteBuffer;

import java.time.LocalDate;
import java.time.LocalTime;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.time.Instant;
import java.math.BigDecimal;
import java.math.BigInteger;

import org.joda.time.DateTime;
import org.joda.time.Period;

import org.apache.drill.exec.util.Text;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;

import org.apache.drill.exec.util.DecimalUtility;

/**
 * UInt4 implements a vector of fixed width values. Elements in the
 * vector are accessed by position, starting from the logical start of the
 * vector. Values should be pushed onto the vector sequentially, but may be
 * accessed randomly.
 * <ul>
 * <li>The width of each element is {@link #VALUE_WIDTH} (= 4)
 * bytes.</li>
 * <li>The equivalent Java primitive is 'int'.</li>
 * </ul>
 *
 * NB: this class is automatically generated from FixedValueVectors.java and
 * ValueVectorTypes.tdd using FreeMarker.
 */
public final class UInt4Vector extends BaseDataValueVector implements FixedWidthVector {
  private static final org.slf4j.Logger logger = org.slf4j.LoggerFactory.getLogger(UInt4Vector.class);

  /**
   * Width of each fixed-width value.
   */

  public static final int VALUE_WIDTH = 4;

  /**
   * Maximum number of values that this fixed-width vector can hold
   * and stay below the maximum vector size limit. This is the limit
   * enforced when the vector is used to hold values in a repeated
   * vector.
   */

  public static final int MAX_VALUE_COUNT = MAX_BUFFER_SIZE / VALUE_WIDTH;

  /**
   * Maximum number of values that this fixed-width vector can hold
   * and stay below the maximum vector size limit and/or stay below
   * the maximum row count. This is the limit enforced when the
   * vector is used to hold scalar (required or nullable) values.
   * <p>
   * Note: <tt>MAX_ROW_COUNT</tt> is defined in the parent <tt>ValueVector</tt>
   * class as the maximum number of rows in a record batch (64K). Use this
   * in place of the <tt>Character.MAX_SIZE</tt> value previously used.
   */

  public static final int MAX_SCALAR_COUNT = Math.min(MAX_ROW_COUNT, MAX_VALUE_COUNT);

  /**
   * Actual maximum vector size, in bytes, given the number of fixed-width
   * values that either fit in the maximum overall vector size, or that
   * is no larger than the maximum vector item count.
   */

  public static final int NET_MAX_SCALAR_SIZE = VALUE_WIDTH * MAX_SCALAR_COUNT;

  private final FieldReader reader = new UInt4ReaderImpl(UInt4Vector.this);
  private final Accessor accessor = new Accessor();
  private final Mutator mutator = new Mutator();

  private int allocationSizeInBytes = Math.min(INITIAL_VALUE_ALLOCATION * VALUE_WIDTH, MAX_BUFFER_SIZE);
  private int allocationMonitor = 0;

  public UInt4Vector(MaterializedField field, BufferAllocator allocator) {
    super(field, allocator);
  }

  @Override
  public FieldReader getReader() { return reader; }

  @Override
  public int getBufferSizeFor(int valueCount) {
    if (valueCount == 0) {
      return 0;
    }
    return valueCount * VALUE_WIDTH;
  }

  @Override
  public int getValueCapacity() {
    return data.capacity() / VALUE_WIDTH;
  }

  @Override
  public Accessor getAccessor() { return accessor; }

  @Override
  public Mutator getMutator() { return mutator; }

  @Override
  public void setInitialCapacity(int valueCount) {
    long size = (long) valueCount * VALUE_WIDTH;
    // TODO: Replace this with MAX_BUFFER_SIZE once all
    // code is aware of the maximum vector size.
    if (size > MAX_ALLOCATION_SIZE) {
      throw new OversizedAllocationException("Requested amount of memory is more than max allowed allocation size");
    }
    allocationSizeInBytes = (int) size;
  }

  @Override
  public void allocateNew() {
    if (!allocateNewSafe()) {
      throw new OutOfMemoryException("Failure while allocating buffer.");
    }
  }

  @Override
  public boolean allocateNewSafe() {
    long curAllocationSize = allocationSizeInBytes;
    if (allocationMonitor > 10) {
      curAllocationSize = Math.max(8, curAllocationSize / 2);
      allocationMonitor = 0;
    } else if (allocationMonitor < -2) {
      curAllocationSize = allocationSizeInBytes * 2L;
      allocationMonitor = 0;
    }

    try{
      allocateBytes(curAllocationSize);
    } catch (DrillRuntimeException ex) {
      return false;
    }
    return true;
  }

  /**
   * Allocate a new buffer that supports setting at least the provided number of
   * values. May actually be sized bigger depending on underlying buffer
   * rounding size. Must be called prior to using the ValueVector.
   *
   * Note that the maximum number of values a vector can allocate is
   * Integer.MAX_VALUE / value width.
   *
   * @param valueCount
   * @throws OutOfMemoryException
   *           if it can't allocate the new buffer
   */
  @Override
  public void allocateNew(int valueCount) {
    allocateBytes(valueCount * VALUE_WIDTH);
  }

  @Override
  public void reset() {
    allocationSizeInBytes = INITIAL_VALUE_ALLOCATION;
    allocationMonitor = 0;
    zeroVector();
    super.reset();
  }

  private void allocateBytes(long size) {
    // TODO: Replace this with MAX_BUFFER_SIZE once all
    // code is aware of the maximum vector size.
    if (size > MAX_ALLOCATION_SIZE) {
      throw new OversizedAllocationException("Requested amount of memory is more than max allowed allocation size");
    }

    int curSize = (int)size;
    clear();
    data = allocator.buffer(curSize);
    data.readerIndex(0);
    allocationSizeInBytes = curSize;
  }

  /**
   * Allocate new buffer with double capacity, and copy data into the new
   * buffer. Replace vector's buffer with new buffer, and release old one
   *
   * @throws org.apache.drill.exec.memory.OutOfMemoryException
   *           if it can't allocate the new buffer
   */

  public void reAlloc() {

    // Avoid an infinite loop if we try to double the size of
    // a zero-length buffer. Instead, just allocate a 256 byte
    // buffer if we start at 0.

    long newAllocationSize = allocationSizeInBytes == 0
        ? 256
        : allocationSizeInBytes * 2L;

    int currentCapacity = data.capacity();
    // Some operations, such as Value Vector#exchange, can be change DrillBuf data field without corresponding allocation size changes.
    // Check that the size of the allocation is sufficient to copy the old buffer.
    while (newAllocationSize < currentCapacity) {
      newAllocationSize *= 2L;
    }

    // TODO: Replace this with MAX_BUFFER_SIZE once all
    // code is aware of the maximum vector size.

    if (newAllocationSize > MAX_ALLOCATION_SIZE)  {
      throw new OversizedAllocationException("Unable to expand the buffer. Max allowed buffer size is reached.");
    }

    reallocRaw((int) newAllocationSize);
    data.setZero(currentCapacity, data.capacity() - currentCapacity);
  }

  @Override
  public DrillBuf reallocRaw(int newAllocationSize) {
    logger.debug("Reallocating vector [{}]. # of bytes: [{}] -> [{}]", field, allocationSizeInBytes, newAllocationSize);
    if (newAllocationSize == 0) {
      throw new IllegalStateException("Attempt to reAlloc a zero-sized vector");
    }
    DrillBuf newBuf = allocator.buffer(newAllocationSize);
    newBuf.setBytes(0, data, 0, data.capacity());
    newBuf.writerIndex(data.writerIndex());
    data.release(1);
    data = newBuf;
    allocationSizeInBytes = newAllocationSize;
    return newBuf;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void zeroVector() {
    data.setZero(0, data.capacity());
  }

  @Override
  public void load(SerializedField metadata, DrillBuf buffer) {
    Preconditions.checkArgument(this.field.getName().equals(metadata.getNamePart().getName()),
                                "The field %s doesn't match the provided metadata %s.", this.field, metadata);
    int actualLength = metadata.getBufferLength();
    int valueCount = metadata.getValueCount();
    int expectedLength = valueCount * VALUE_WIDTH;
    assert actualLength == expectedLength : String.format("Expected to load %d bytes but actually loaded %d bytes", expectedLength, actualLength);

    clear();
    if (data != null) {
      data.release(1);
    }
    data = buffer.slice(0, actualLength);
    data.retain(1);
    data.writerIndex(actualLength);
  }

  @Override
  public TransferPair getTransferPair(BufferAllocator allocator) {
    return new TransferImpl(getField(), allocator);
  }

  @Override
  public TransferPair getTransferPair(String ref, BufferAllocator allocator) {
    return new TransferImpl(getField().withPath(ref), allocator);
  }

  @Override
  public TransferPair makeTransferPair(ValueVector to) {
    return new TransferImpl((UInt4Vector) to);
  }

  public void transferTo(UInt4Vector target) {
    target.clear();
    target.data = data.transferOwnership(target.allocator).buffer;
    target.data.writerIndex(data.writerIndex());
    clear();
  }

  public void splitAndTransferTo(int startIndex, int length, UInt4Vector target) {
    int startPoint = startIndex * VALUE_WIDTH;
    int sliceLength = length * VALUE_WIDTH;
    target.clear();
    target.data = data.slice(startPoint, sliceLength).transferOwnership(target.allocator).buffer;
    target.data.writerIndex(sliceLength);
  }

  @Override
  public int getPayloadByteCount(int valueCount) {
    return valueCount * 4;
  }

  @Override
  public int getValueWidth() {
    return 4;
  }

  private class TransferImpl implements TransferPair {
    private UInt4Vector to;

    public TransferImpl(MaterializedField field, BufferAllocator allocator) {
      to = new UInt4Vector(field, allocator);
    }

    public TransferImpl(UInt4Vector to) {
      this.to = to;
    }

    @Override
    public UInt4Vector getTo() {
      return to;
    }

    @Override
    public void transfer() {
      transferTo(to);
    }

    @Override
    public void splitAndTransfer(int startIndex, int length) {
      splitAndTransferTo(startIndex, length, to);
    }

    @Override
    public void copyValueSafe(int fromIndex, int toIndex) {
      to.copyFromSafe(fromIndex, toIndex, UInt4Vector.this);
    }
  }

  public void copyFrom(int fromIndex, int thisIndex, UInt4Vector from) {
 
    data.setInt(thisIndex * VALUE_WIDTH,
        from.data.getInt(fromIndex * VALUE_WIDTH)
    );
     
  }

  public void copyFromSafe(int fromIndex, int thisIndex, UInt4Vector from) {
    while (thisIndex >= getValueCapacity()) {
        reAlloc();
    }
    copyFrom(fromIndex, thisIndex, from);
  }

  @Override
  public void copyEntry(int toIndex, ValueVector from, int fromIndex) {
    copyFromSafe(fromIndex, toIndex, (UInt4Vector) from);
  }

  public void decrementAllocationMonitor() {
    if (allocationMonitor > 0) {
      allocationMonitor = 0;
    }
    --allocationMonitor;
  }

  private void incrementAllocationMonitor() {
    ++allocationMonitor;
  }

  @Override
  public void toNullable(ValueVector nullableVector) {
    NullableUInt4Vector dest = (NullableUInt4Vector) nullableVector;
    dest.getMutator().fromNotNullable(this);
  }

  public final class Accessor extends BaseDataValueVector.BaseAccessor {
    @Override
    public int getValueCount() {
      return data.writerIndex() / VALUE_WIDTH;
    }

    @Override
    public boolean isNull(int index) {
      return false;
    }
 

    public int get(int index) {
      return data.getInt(index * VALUE_WIDTH);
    }

    public long getTwoAsLong(int index) {
      return data.getLong(index * VALUE_WIDTH);
    }

    @Override
    public Integer getObject(int index) {
      return get(index);
    }

    public int getPrimitiveObject(int index) {
      return get(index);
    }

    public void get(int index, UInt4Holder holder) {

      holder.value = data.getInt(index * VALUE_WIDTH);
    }

    public void get(int index, NullableUInt4Holder holder) {
      holder.isSet = 1;
      holder.value = data.getInt(index * VALUE_WIDTH);
    }
     
  }

  /**
   * UInt4.Mutator implements a mutable vector of fixed width values.
   * Elements in the vector are accessed by position from the logical start of
   * the vector. Values should be pushed onto the vector sequentially, but may
   * be randomly accessed.
   * <ul>
   * <li>The width of each element is {@link #VALUE_WIDTH} (= 4)
   * byte(s).</li>
   * <li>The equivalent Java primitive is 'int'</li>
   * </ul>
   *
   * NB: this class is automatically generated from ValueVectorTypes.tdd using
   * FreeMarker.
   */
   public final class Mutator extends BaseDataValueVector.BaseMutator {

    private Mutator() {};

    /**
     * Set the element at the given index to the given value. Note that widths
     * smaller than 32 bits are handled by the DrillBuf interface.
     *
     * @param index
     *          position of the bit to set
     * @param value
     *          value to set
     */
 

    public void set(int index, int value) {
      data.setInt(index * VALUE_WIDTH, value);
    }


    /**
     * Set the value of a required or nullable vector. Grows the vector as needed.
     * Does not enforce size limits; scalar fixed-width types can never overflow
     * a vector.
     * @param index item to write
     */
    public void setSafe(int index, int value) {
      while (index >= getValueCapacity()) {
        reAlloc();
      }
      set(index, value);
    }

    protected void set(int index, UInt4Holder holder) {
      data.setInt(index * VALUE_WIDTH, holder.value);
    }

    public void setSafe(int index, UInt4Holder holder) {
      while (index >= getValueCapacity()) {
        reAlloc();
      }
      set(index, holder);
    }

    protected void set(int index, NullableUInt4Holder holder) {
      data.setInt(index * VALUE_WIDTH, holder.value);
    }

    public void setSafe(int index, NullableUInt4Holder holder) {
      while (index >= getValueCapacity()) {
        reAlloc();
      }
      set(index, holder);
    }

    @Override
    public void generateTestData(int size) {
      setValueCount(size);
      boolean even = true;
      final int valueCount = getAccessor().getValueCount();
      for(int i = 0; i < valueCount; i++, even = !even) {
        if(even) {
          set(i, Integer.MIN_VALUE);
        } else {
          set(i, Integer.MAX_VALUE);
        }
      }
    }

    public void generateTestDataAlt(int size) {
      setValueCount(size);
      boolean even = true;
      final int valueCount = getAccessor().getValueCount();
      for(int i = 0; i < valueCount; i++, even = !even) {
        if(even) {
          set(i, (int) 1);
        } else {
          set(i, (int) 0);
        }
      }
    }
   

    @Override
    public void setValueCount(int valueCount) {
      final int currentValueCapacity = getValueCapacity();
      final int idx = VALUE_WIDTH * valueCount;
      while (valueCount > getValueCapacity()) {
        reAlloc();
      }
      if (valueCount > 0 && currentValueCapacity > valueCount * 2) {
        incrementAllocationMonitor();
      } else if (allocationMonitor > 0) {
        allocationMonitor = 0;
      }
      data.writerIndex(idx);
    }
  }

  /**
   * Helper class to buffer container mutation as a means to optimize native memory copy operations.
   *
   * NB: this class is automatically generated from ValueVectorTypes.tdd using FreeMarker.
   */
  public static final class BufferedMutator {
    /** The default buffer size */
    private static final int DEFAULT_BUFF_SZ = 1024 << 2;
    /** Byte buffer */
    private final ByteBuffer buffer;

    /** Tracks the index where to copy future values */
    private int currentIdx;
    /** Parent conatiner object */
    private final UInt4Vector parent;

    /** @see {@link #BufferedMutator(int startIdx, int buffSz, UInt4Vector parent)} */
    public BufferedMutator(int startIdx, UInt4Vector parent) {
      this(startIdx, DEFAULT_BUFF_SZ, parent);
    }

    /**
     * Buffered mutator to optimize bulk access to the underlying vector container
     * @param startIdx start idex of the first value to be copied
     * @param buffSz buffer length to us
     * @param parent parent container object
     */
    public BufferedMutator(int startIdx, int buffSz, UInt4Vector parent) {
      this.buffer = ByteBuffer.allocate(buffSz);

      // set the buffer to the native byte order
      buffer.order(ByteOrder.nativeOrder());

      this.currentIdx = startIdx;
      this.parent = parent;
    }

    public void setSafe(int value) {
      if (buffer.remaining() < 4) {
        flush();
      }

      int tgtPos = buffer.position();
      byte[] bufferArray = buffer.array();

      writeInt(value, bufferArray, tgtPos);
      buffer.position(tgtPos + 4);
    }

    public void setSafe(int[] values, int numValues) {
      int remaining = numValues;
      byte[] bufferArray = buffer.array();
      int srcPos = 0;

      do {
        if (buffer.remaining() < 4) {
            flush();
        }

        int toCopy = Math.min(remaining, buffer.remaining() / 4);
        int tgtPos = buffer.position();

        for (int idx = 0; idx < toCopy; idx++, tgtPos += 4, srcPos++) {
          writeInt(values[srcPos], bufferArray, tgtPos);
        }

        // Update counters
        buffer.position(tgtPos);
        remaining -= toCopy;

      } while (remaining > 0);
    }

    public static final void writeInt(int val, byte[] buffer, int pos) {
      DrillBuf.putInt(buffer, pos, val);
    }
     
 

    /**
     * @return the backing byte buffer; this is useful when the caller can infer the values to write but
     *         wants to avoid having to use yet another intermediary byte array; caller is responsible for
     *         flushing the buffer
    */
    public ByteBuffer getByteBuffer() {
      return buffer;
    }

    public void flush() {
      int numElements = buffer.position() / UInt4Vector.VALUE_WIDTH;

      if (numElements == 0) {
        return; // NOOP
      }

      while ((currentIdx + numElements -1) >= parent.getValueCapacity()) {
        parent.reAlloc();
      }

      parent.data.setBytes(currentIdx * UInt4Vector.VALUE_WIDTH, buffer.array(), 0, buffer.position());

      // Update the start index
      currentIdx += numElements;

      // Reset the byte buffer
      buffer.clear();
    }
  }
   
}
 

