/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.drill.exec.vector;

import static org.apache.drill.shaded.guava.com.google.common.base.Preconditions.checkArgument;
import static org.apache.drill.shaded.guava.com.google.common.base.Preconditions.checkState;

import org.apache.drill.shaded.guava.com.google.common.collect.Lists;
import org.apache.drill.shaded.guava.com.google.common.collect.ObjectArrays;
import org.apache.drill.shaded.guava.com.google.common.base.Charsets;
import org.apache.drill.shaded.guava.com.google.common.collect.ObjectArrays;

import org.apache.drill.shaded.guava.com.google.common.base.Preconditions;
import io.netty.buffer.*;

import org.apache.commons.lang3.ArrayUtils;

import org.apache.drill.common.exceptions.UserException;
import org.apache.drill.exec.memory.*;
import org.apache.drill.exec.proto.SchemaDefProtos;
import org.apache.drill.exec.proto.UserBitShared;
import org.apache.drill.exec.proto.UserBitShared.DrillPBError;
import org.apache.drill.exec.proto.UserBitShared.SerializedField;
import org.apache.drill.exec.record.*;
import org.apache.drill.exec.vector.*;
import org.apache.drill.common.exceptions.*;
import org.apache.drill.exec.exception.*;
import org.apache.drill.exec.expr.holders.*;
import org.apache.drill.common.types.TypeProtos.*;
import org.apache.drill.common.types.Types;
import org.apache.drill.common.util.DrillStringUtils;
import org.apache.drill.exec.vector.complex.*;
import org.apache.drill.exec.vector.complex.reader.*;
import org.apache.drill.exec.vector.complex.impl.*;
import org.apache.drill.exec.vector.complex.writer.*;
import org.apache.drill.exec.vector.complex.writer.BaseWriter.MapWriter;
import org.apache.drill.exec.vector.complex.writer.BaseWriter.DictWriter;
import org.apache.drill.exec.vector.complex.writer.BaseWriter.ListWriter;
import org.apache.drill.exec.util.JsonStringArrayList;
import org.apache.drill.exec.memory.AllocationManager.BufferLedger;

import org.apache.drill.exec.exception.OutOfMemoryException;

import java.util.Arrays;
import java.util.Random;
import java.util.List;
import java.util.Set;

import java.io.Closeable;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.ByteBuffer;

import java.time.LocalDate;
import java.time.LocalTime;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.time.Instant;
import java.math.BigDecimal;
import java.math.BigInteger;

import org.joda.time.DateTime;
import org.joda.time.Period;

import org.apache.drill.exec.util.Text;

import org.apache.drill.exec.util.DecimalUtility;

/**
 * Date implements a vector of fixed width values. Elements in the
 * vector are accessed by position, starting from the logical start of the
 * vector. Values should be pushed onto the vector sequentially, but may be
 * accessed randomly.
 * <ul>
 * <li>The width of each element is {@link #VALUE_WIDTH} (= 8)
 * bytes.</li>
 * <li>The equivalent Java primitive is 'long'.</li>
 * </ul>
 *
 * NB: this class is automatically generated from FixedValueVectors.java and
 * ValueVectorTypes.tdd using FreeMarker.
 */
public final class DateVector extends BaseDataValueVector implements FixedWidthVector {
  private static final org.slf4j.Logger logger = org.slf4j.LoggerFactory.getLogger(DateVector.class);

  /**
   * Width of each fixed-width value.
   */

  public static final int VALUE_WIDTH = 8;

  /**
   * Maximum number of values that this fixed-width vector can hold
   * and stay below the maximum vector size limit. This is the limit
   * enforced when the vector is used to hold values in a repeated
   * vector.
   */

  public static final int MAX_VALUE_COUNT = MAX_BUFFER_SIZE / VALUE_WIDTH;

  /**
   * Maximum number of values that this fixed-width vector can hold
   * and stay below the maximum vector size limit and/or stay below
   * the maximum row count. This is the limit enforced when the
   * vector is used to hold scalar (required or nullable) values.
   * <p>
   * Note: <tt>MAX_ROW_COUNT</tt> is defined in the parent <tt>ValueVector</tt>
   * class as the maximum number of rows in a record batch (64K). Use this
   * in place of the <tt>Character.MAX_SIZE</tt> value previously used.
   */

  public static final int MAX_SCALAR_COUNT = Math.min(MAX_ROW_COUNT, MAX_VALUE_COUNT);

  /**
   * Actual maximum vector size, in bytes, given the number of fixed-width
   * values that either fit in the maximum overall vector size, or that
   * is no larger than the maximum vector item count.
   */

  public static final int NET_MAX_SCALAR_SIZE = VALUE_WIDTH * MAX_SCALAR_COUNT;

  private final FieldReader reader = new DateReaderImpl(DateVector.this);
  private final Accessor accessor = new Accessor();
  private final Mutator mutator = new Mutator();

  private int allocationSizeInBytes = Math.min(INITIAL_VALUE_ALLOCATION * VALUE_WIDTH, MAX_BUFFER_SIZE);
  private int allocationMonitor = 0;

  public DateVector(MaterializedField field, BufferAllocator allocator) {
    super(field, allocator);
  }

  @Override
  public FieldReader getReader() { return reader; }

  @Override
  public int getBufferSizeFor(int valueCount) {
    if (valueCount == 0) {
      return 0;
    }
    return valueCount * VALUE_WIDTH;
  }

  @Override
  public int getValueCapacity() {
    return data.capacity() / VALUE_WIDTH;
  }

  @Override
  public Accessor getAccessor() { return accessor; }

  @Override
  public Mutator getMutator() { return mutator; }

  @Override
  public void setInitialCapacity(int valueCount) {
    long size = (long) valueCount * VALUE_WIDTH;
    // TODO: Replace this with MAX_BUFFER_SIZE once all
    // code is aware of the maximum vector size.
    if (size > MAX_ALLOCATION_SIZE) {
      throw new OversizedAllocationException("Requested amount of memory is more than max allowed allocation size");
    }
    allocationSizeInBytes = (int) size;
  }

  @Override
  public void allocateNew() {
    if (!allocateNewSafe()) {
      throw new OutOfMemoryException("Failure while allocating buffer.");
    }
  }

  @Override
  public boolean allocateNewSafe() {
    long curAllocationSize = allocationSizeInBytes;
    if (allocationMonitor > 10) {
      curAllocationSize = Math.max(8, curAllocationSize / 2);
      allocationMonitor = 0;
    } else if (allocationMonitor < -2) {
      curAllocationSize = allocationSizeInBytes * 2L;
      allocationMonitor = 0;
    }

    try{
      allocateBytes(curAllocationSize);
    } catch (DrillRuntimeException ex) {
      return false;
    }
    return true;
  }

  /**
   * Allocate a new buffer that supports setting at least the provided number of
   * values. May actually be sized bigger depending on underlying buffer
   * rounding size. Must be called prior to using the ValueVector.
   *
   * Note that the maximum number of values a vector can allocate is
   * Integer.MAX_VALUE / value width.
   *
   * @param valueCount
   * @throws OutOfMemoryException
   *           if it can't allocate the new buffer
   */
  @Override
  public void allocateNew(int valueCount) {
    allocateBytes(valueCount * VALUE_WIDTH);
  }

  @Override
  public void reset() {
    allocationSizeInBytes = INITIAL_VALUE_ALLOCATION;
    allocationMonitor = 0;
    zeroVector();
    super.reset();
  }

  private void allocateBytes(long size) {
    // TODO: Replace this with MAX_BUFFER_SIZE once all
    // code is aware of the maximum vector size.
    if (size > MAX_ALLOCATION_SIZE) {
      throw new OversizedAllocationException("Requested amount of memory is more than max allowed allocation size");
    }

    int curSize = (int)size;
    clear();
    data = allocator.buffer(curSize);
    data.readerIndex(0);
    allocationSizeInBytes = curSize;
  }

  /**
   * Allocate new buffer with double capacity, and copy data into the new
   * buffer. Replace vector's buffer with new buffer, and release old one
   *
   * @throws org.apache.drill.exec.memory.OutOfMemoryException
   *           if it can't allocate the new buffer
   */

  public void reAlloc() {

    // Avoid an infinite loop if we try to double the size of
    // a zero-length buffer. Instead, just allocate a 256 byte
    // buffer if we start at 0.

    long newAllocationSize = allocationSizeInBytes == 0
        ? 256
        : allocationSizeInBytes * 2L;

    int currentCapacity = data.capacity();
    // Some operations, such as Value Vector#exchange, can be change DrillBuf data field without corresponding allocation size changes.
    // Check that the size of the allocation is sufficient to copy the old buffer.
    while (newAllocationSize < currentCapacity) {
      newAllocationSize *= 2L;
    }

    // TODO: Replace this with MAX_BUFFER_SIZE once all
    // code is aware of the maximum vector size.

    if (newAllocationSize > MAX_ALLOCATION_SIZE)  {
      throw new OversizedAllocationException("Unable to expand the buffer. Max allowed buffer size is reached.");
    }

    reallocRaw((int) newAllocationSize);
    data.setZero(currentCapacity, data.capacity() - currentCapacity);
  }

  @Override
  public DrillBuf reallocRaw(int newAllocationSize) {
    logger.debug("Reallocating vector [{}]. # of bytes: [{}] -> [{}]", field, allocationSizeInBytes, newAllocationSize);
    if (newAllocationSize == 0) {
      throw new IllegalStateException("Attempt to reAlloc a zero-sized vector");
    }
    DrillBuf newBuf = allocator.buffer(newAllocationSize);
    newBuf.setBytes(0, data, 0, data.capacity());
    newBuf.writerIndex(data.writerIndex());
    data.release(1);
    data = newBuf;
    allocationSizeInBytes = newAllocationSize;
    return newBuf;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void zeroVector() {
    data.setZero(0, data.capacity());
  }

  @Override
  public void load(SerializedField metadata, DrillBuf buffer) {
    Preconditions.checkArgument(this.field.getName().equals(metadata.getNamePart().getName()),
                                "The field %s doesn't match the provided metadata %s.", this.field, metadata);
    int actualLength = metadata.getBufferLength();
    int valueCount = metadata.getValueCount();
    int expectedLength = valueCount * VALUE_WIDTH;
    assert actualLength == expectedLength : String.format("Expected to load %d bytes but actually loaded %d bytes", expectedLength, actualLength);

    clear();
    if (data != null) {
      data.release(1);
    }
    data = buffer.slice(0, actualLength);
    data.retain(1);
    data.writerIndex(actualLength);
  }

  @Override
  public TransferPair getTransferPair(BufferAllocator allocator) {
    return new TransferImpl(getField(), allocator);
  }

  @Override
  public TransferPair getTransferPair(String ref, BufferAllocator allocator) {
    return new TransferImpl(getField().withPath(ref), allocator);
  }

  @Override
  public TransferPair makeTransferPair(ValueVector to) {
    return new TransferImpl((DateVector) to);
  }

  public void transferTo(DateVector target) {
    target.clear();
    target.data = data.transferOwnership(target.allocator).buffer;
    target.data.writerIndex(data.writerIndex());
    clear();
  }

  public void splitAndTransferTo(int startIndex, int length, DateVector target) {
    int startPoint = startIndex * VALUE_WIDTH;
    int sliceLength = length * VALUE_WIDTH;
    target.clear();
    target.data = data.slice(startPoint, sliceLength).transferOwnership(target.allocator).buffer;
    target.data.writerIndex(sliceLength);
  }

  @Override
  public int getPayloadByteCount(int valueCount) {
    return valueCount * 8;
  }

  @Override
  public int getValueWidth() {
    return 8;
  }

  private class TransferImpl implements TransferPair {
    private DateVector to;

    public TransferImpl(MaterializedField field, BufferAllocator allocator) {
      to = new DateVector(field, allocator);
    }

    public TransferImpl(DateVector to) {
      this.to = to;
    }

    @Override
    public DateVector getTo() {
      return to;
    }

    @Override
    public void transfer() {
      transferTo(to);
    }

    @Override
    public void splitAndTransfer(int startIndex, int length) {
      splitAndTransferTo(startIndex, length, to);
    }

    @Override
    public void copyValueSafe(int fromIndex, int toIndex) {
      to.copyFromSafe(fromIndex, toIndex, DateVector.this);
    }
  }

  public void copyFrom(int fromIndex, int thisIndex, DateVector from) {
 
    data.setLong(thisIndex * VALUE_WIDTH,
        from.data.getLong(fromIndex * VALUE_WIDTH)
    );
     
  }

  public void copyFromSafe(int fromIndex, int thisIndex, DateVector from) {
    while (thisIndex >= getValueCapacity()) {
        reAlloc();
    }
    copyFrom(fromIndex, thisIndex, from);
  }

  @Override
  public void copyEntry(int toIndex, ValueVector from, int fromIndex) {
    copyFromSafe(fromIndex, toIndex, (DateVector) from);
  }

  public void decrementAllocationMonitor() {
    if (allocationMonitor > 0) {
      allocationMonitor = 0;
    }
    --allocationMonitor;
  }

  private void incrementAllocationMonitor() {
    ++allocationMonitor;
  }

  @Override
  public void toNullable(ValueVector nullableVector) {
    NullableDateVector dest = (NullableDateVector) nullableVector;
    dest.getMutator().fromNotNullable(this);
  }

  public final class Accessor extends BaseDataValueVector.BaseAccessor {
    @Override
    public int getValueCount() {
      return data.writerIndex() / VALUE_WIDTH;
    }

    @Override
    public boolean isNull(int index) {
      return false;
    }
 

    public long get(int index) {
      return data.getLong(index * VALUE_WIDTH);
    }

    @Override
    public LocalDate getObject(int index) {
      return LocalDateTime.ofInstant(Instant.ofEpochMilli(get(index)), ZoneOffset.UTC).toLocalDate();
    }

    public void get(int index, DateHolder holder) {

      holder.value = data.getLong(index * VALUE_WIDTH);
    }

    public void get(int index, NullableDateHolder holder) {
      holder.isSet = 1;
      holder.value = data.getLong(index * VALUE_WIDTH);
    }
     
  }

  /**
   * Date.Mutator implements a mutable vector of fixed width values.
   * Elements in the vector are accessed by position from the logical start of
   * the vector. Values should be pushed onto the vector sequentially, but may
   * be randomly accessed.
   * <ul>
   * <li>The width of each element is {@link #VALUE_WIDTH} (= 8)
   * byte(s).</li>
   * <li>The equivalent Java primitive is 'long'</li>
   * </ul>
   *
   * NB: this class is automatically generated from ValueVectorTypes.tdd using
   * FreeMarker.
   */
   public final class Mutator extends BaseDataValueVector.BaseMutator {

    private Mutator() {};

    /**
     * Set the element at the given index to the given value. Note that widths
     * smaller than 32 bits are handled by the DrillBuf interface.
     *
     * @param index
     *          position of the bit to set
     * @param value
     *          value to set
     */
 

    public void set(int index, long value) {
      data.setLong(index * VALUE_WIDTH, value);
    }


    /**
     * Set the value of a required or nullable vector. Grows the vector as needed.
     * Does not enforce size limits; scalar fixed-width types can never overflow
     * a vector.
     * @param index item to write
     */
    public void setSafe(int index, long value) {
      while (index >= getValueCapacity()) {
        reAlloc();
      }
      set(index, value);
    }

    protected void set(int index, DateHolder holder) {
      data.setLong(index * VALUE_WIDTH, holder.value);
    }

    public void setSafe(int index, DateHolder holder) {
      while (index >= getValueCapacity()) {
        reAlloc();
      }
      set(index, holder);
    }

    protected void set(int index, NullableDateHolder holder) {
      data.setLong(index * VALUE_WIDTH, holder.value);
    }

    public void setSafe(int index, NullableDateHolder holder) {
      while (index >= getValueCapacity()) {
        reAlloc();
      }
      set(index, holder);
    }

    @Override
    public void generateTestData(int size) {
      setValueCount(size);
      boolean even = true;
      final int valueCount = getAccessor().getValueCount();
      for(int i = 0; i < valueCount; i++, even = !even) {
        if(even) {
          set(i, Long.MIN_VALUE);
        } else {
          set(i, Long.MAX_VALUE);
        }
      }
    }

    public void generateTestDataAlt(int size) {
      setValueCount(size);
      boolean even = true;
      final int valueCount = getAccessor().getValueCount();
      for(int i = 0; i < valueCount; i++, even = !even) {
        if(even) {
          set(i, (long) 1);
        } else {
          set(i, (long) 0);
        }
      }
    }
   

    @Override
    public void setValueCount(int valueCount) {
      final int currentValueCapacity = getValueCapacity();
      final int idx = VALUE_WIDTH * valueCount;
      while (valueCount > getValueCapacity()) {
        reAlloc();
      }
      if (valueCount > 0 && currentValueCapacity > valueCount * 2) {
        incrementAllocationMonitor();
      } else if (allocationMonitor > 0) {
        allocationMonitor = 0;
      }
      data.writerIndex(idx);
    }
  }
 
}
 

