
/*******************************************************************************

 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 ******************************************************************************/
package org.apache.drill.exec.expr.fn.impl.gcast;


import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkState;

import com.google.common.collect.Lists;
import com.google.common.collect.ObjectArrays;
import com.google.common.base.Charsets;
import com.google.common.collect.ObjectArrays;

import com.google.common.base.Preconditions;
import io.netty.buffer.*;

import org.apache.commons.lang3.ArrayUtils;

import org.apache.drill.common.exceptions.UserException;
import org.apache.drill.exec.expr.fn.impl.StringFunctionUtil;
import org.apache.drill.exec.memory.*;
import org.apache.drill.exec.proto.SchemaDefProtos;
import org.apache.drill.exec.proto.UserBitShared;
import org.apache.drill.exec.proto.UserBitShared.DrillPBError;
import org.apache.drill.exec.proto.UserBitShared.SerializedField;
import org.apache.drill.exec.record.*;
import org.apache.drill.exec.vector.*;
import org.apache.drill.common.exceptions.*;
import org.apache.drill.exec.exception.*;
import org.apache.drill.exec.expr.holders.*;
import org.apache.drill.common.expression.FieldReference;
import org.apache.drill.common.types.TypeProtos.*;
import org.apache.drill.common.types.Types;
import org.apache.drill.common.util.DrillStringUtils;
import org.apache.drill.exec.vector.complex.*;
import org.apache.drill.exec.vector.complex.reader.*;
import org.apache.drill.exec.vector.complex.impl.*;
import org.apache.drill.exec.vector.complex.writer.*;
import org.apache.drill.exec.vector.complex.writer.BaseWriter.MapWriter;
import org.apache.drill.exec.vector.complex.writer.BaseWriter.ListWriter;
import org.apache.drill.exec.util.JsonStringArrayList;

import org.apache.drill.exec.exception.OutOfMemoryException;

import com.sun.codemodel.JType;
import com.sun.codemodel.JCodeModel;

import javax.inject.Inject;

import java.util.Arrays;
import java.util.Random;
import java.util.List;

import java.io.Closeable;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.ByteBuffer;

import java.sql.Date;
import java.sql.Time;
import java.sql.Timestamp;
import java.math.BigDecimal;
import java.math.BigInteger;

import org.joda.time.DateTime;
import org.joda.time.Period;

import org.apache.drill.exec.util.Text;

import org.apache.drill.exec.vector.accessor.sql.TimePrintMillis;
import javax.inject.Inject;






import org.apache.drill.exec.expr.DrillSimpleFunc;
import org.apache.drill.exec.expr.annotations.FunctionTemplate;
import org.apache.drill.exec.expr.annotations.FunctionTemplate.NullHandling;
import org.apache.drill.exec.expr.annotations.Output;
import org.apache.drill.exec.expr.annotations.Param;
import org.apache.drill.exec.expr.holders.*;
import org.apache.drill.exec.record.RecordBatch;
import org.apache.drill.exec.util.DecimalUtility;
import org.apache.drill.exec.expr.annotations.Workspace;

import io.netty.buffer.ByteBuf;

import java.nio.ByteBuffer;

@SuppressWarnings("unused")
@FunctionTemplate(name = "castDECIMAL28SPARSE", scope = FunctionTemplate.FunctionScope.DECIMAL_CAST, nulls=NullHandling.NULL_IF_NULL)
public class CastVarCharDecimal28Sparse implements DrillSimpleFunc {
    @Param VarCharHolder in;
    @Inject DrillBuf buffer;
    @Param BigIntHolder precision;
    @Param BigIntHolder scale;

    @Output Decimal28SparseHolder out;

    public void setup() {
        int size = 5 * (org.apache.drill.exec.util.DecimalUtility.INTEGER_SIZE);
        buffer = buffer.reallocIfNeeded(size);
    }

    public void eval() {

        out.buffer = buffer;
        out.start  = 0;

        out.scale = (int) scale.value;
        out.precision = (int) precision.value;
        boolean sign = false;

        // Initialize the output buffer
        for (int i = 0; i < 5; i++) {
            out.setInteger(i, 0, out.start, out.buffer);
        }

        int startIndex;
        int readIndex = in.start;
        int integerDigits = 0;
        int fractionalDigits = 0;
        int scaleIndex = -1;
        int scaleEndIndex = in.end;

        byte[] buf1 = new byte[in.end - in.start];
        in.buffer.getBytes(in.start, buf1, 0, in.end - in.start);

        Byte next = in.buffer.getByte(readIndex);


        if (next == '-') {
            readIndex++;
            sign = true;
        }

        if (next == '.') {
            readIndex++;
            scaleIndex = readIndex; // Fractional part starts at the first position
        }

        // Check if its an empty string
        if (in.end - readIndex == 0) {
            throw new org.apache.drill.common.exceptions.DrillRuntimeException("Empty String, cannot cast to Decimal");
        }

        // Store start index for the second pass
        startIndex = readIndex;

        int radix = 10;
        boolean leadingDigitFound = false;
        boolean round = false;

        /* This is the first pass, we get the number of integer digits and based on the provided scale
         * we compute which index into the ByteBuf we start storing the integer part of the Decimal
         */
        if (scaleIndex == -1) {

            while (readIndex < in.end) {
                next = in.buffer.getByte(readIndex++);

                if (next == '.') {

                    // We have found the decimal point. we can compute the starting index into the Decimal's bytebuf
                    scaleIndex = readIndex;
                    // We may have to truncate fractional part if > scale
                    if ((in.end - scaleIndex) > out.scale) {
                      scaleEndIndex =  scaleIndex + out.scale;
                      round = true;
                    }
                    break;
                }

                // If its not a '.' we expect only numbers
                next = (byte) Character.digit(next, radix);

                if (next == -1) {
                    // not a valid digit
                    byte[] buf = new byte[in.end - in.start];
                    in.buffer.getBytes(in.start, buf, 0, in.end - in.start);
                    throw new NumberFormatException(new String(buf, com.google.common.base.Charsets.UTF_8));
                }

                if (leadingDigitFound == false && next != 0) {
                    leadingDigitFound = true;
                }

                if (leadingDigitFound == true) {
                    integerDigits++;
                }
            }
        }


        /* Based on the number of integer digits computed and the scale throw an
         * exception if the provided precision is not sufficient to store the value
         */
        if (integerDigits + out.scale > out.precision) {
            byte[] buf = new byte[in.end - in.start];
            in.buffer.getBytes(in.start, buf, 0, in.end - in.start);
            throw new org.apache.drill.common.exceptions.DrillRuntimeException("Precision is insufficient for the provided input: " + new String(buf, com.google.common.base.Charsets.UTF_8) + " Precision: " + out.precision + " Total Digits: " + (out.scale + integerDigits));
        }


        // Compute the number of slots needed in the ByteBuf to store the integer and fractional part
        int scaleRoundedUp   = org.apache.drill.exec.util.DecimalUtility.roundUp(out.scale);
        int integerRoundedUp = org.apache.drill.exec.util.DecimalUtility.roundUp(integerDigits);

        int ndigits = 0;

        int decimalBufferIndex = 5 - scaleRoundedUp - 1;

        /* Compute the end index of the integer part.
         * If we haven't seen a '.' then entire string is integer.
         * If we have seen a '.' it ends before the '.'
         */
        int integerEndIndex = (scaleIndex == -1) ? (in.end - 1) : (scaleIndex - 2);

        // Traverse and extract the integer part
        while (integerEndIndex >= startIndex) {
            next = in.buffer.getByte(integerEndIndex--);

            next = (byte) Character.digit(next, radix);

            int value = (((int) org.apache.drill.exec.util.DecimalUtility.getPowerOfTen(ndigits)) * next) + (out.getInteger(decimalBufferIndex, out.start, out.buffer));
            out.setInteger(decimalBufferIndex, value, out.start, out.buffer);

            ndigits++;

            /* We store the entire decimal as base 1 billion values, which has maximum of 9 digits (MAX_DIGITS)
             * Once we have stored MAX_DIGITS in a given slot move to the next slot.
             */
            if (ndigits >= org.apache.drill.exec.util.DecimalUtility.MAX_DIGITS) {
                ndigits = 0;
                decimalBufferIndex--;
            }
        }

        // Traverse and extract the fractional part
        decimalBufferIndex = (scaleRoundedUp > 0) ? (5 - scaleRoundedUp) : (5 - 1);
        ndigits = 0;

        if (scaleIndex != -1) {
            while (scaleIndex < scaleEndIndex) {

                // check if we have scanned MAX_DIGITS and we need to move to the next index
                if (ndigits >= org.apache.drill.exec.util.DecimalUtility.MAX_DIGITS) {
                    ndigits = 0;
                    decimalBufferIndex++;
                }

                next = in.buffer.getByte(scaleIndex++);

                // We expect only numbers beyond this
                next = (byte) Character.digit(next, radix);

                if (next == -1) {
                    // not a valid digit
                    byte[] buf = new byte[in.end - in.start];
                    in.buffer.getBytes(in.start, buf, 0, in.end - in.start);
                    throw new NumberFormatException(new String(buf, com.google.common.base.Charsets.UTF_8));
                }
                int value = (out.getInteger(decimalBufferIndex, out.start, out.buffer) * radix) + next;
                out.setInteger(decimalBufferIndex, value, out.start, out.buffer);

                // added another digit to the current index
                ndigits++;
            }

            // round up the decimal if we had to chop off a part of it
            if (round == true) {
               next = in.buffer.getByte(scaleEndIndex);

                // We expect only numbers beyond this
                next = (byte) Character.digit(next, radix);

                if (next == -1) {
                    // not a valid digit
                    byte[] buf = new byte[in.end - in.start];
                    in.buffer.getBytes(in.start, buf, 0, in.end - in.start);
                    throw new NumberFormatException(new String(buf, com.google.common.base.Charsets.UTF_8));
                }
                if (next > 4) {
                    // Need to round up
                    out.setInteger(decimalBufferIndex, out.getInteger(decimalBufferIndex, out.start, out.buffer)+1, out.start, out.buffer);
                }
            }
            // Pad zeroes in the fractional part so that number of digits = MAX_DIGITS
            if (out.scale > 0) {
              int padding = (int) org.apache.drill.exec.util.DecimalUtility.getPowerOfTen((int) (org.apache.drill.exec.util.DecimalUtility.MAX_DIGITS - ndigits));
              out.setInteger(decimalBufferIndex, out.getInteger(decimalBufferIndex, out.start, out.buffer) * padding, out.start, out.buffer);
            }

            int carry = 0;
            do {
                // propagate the carry
                int tempValue = out.getInteger(decimalBufferIndex, out.start, out.buffer) + carry;
                if (tempValue >= org.apache.drill.exec.util.DecimalUtility.DIGITS_BASE) {
                    carry = tempValue / org.apache.drill.exec.util.DecimalUtility.DIGITS_BASE;
                    tempValue = (tempValue % org.apache.drill.exec.util.DecimalUtility.DIGITS_BASE);
                } else {
                    carry = 0;
                }
                out.setInteger(decimalBufferIndex--, tempValue, out.start, out.buffer);
            } while (carry > 0 && decimalBufferIndex >= 0);
        }
        out.setSign(sign, out.start, out.buffer);
    }
}
 

  

