/*
 * Decompiled with CFR 0.152.
 */
package org.apache.comet.parquet;

import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.Method;
import java.net.URI;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.zip.CRC32;
import org.apache.comet.parquet.ColumnIndexReader;
import org.apache.comet.parquet.ColumnPageReader;
import org.apache.comet.parquet.CometFileReaderThreadPool;
import org.apache.comet.parquet.CometInputFile;
import org.apache.comet.parquet.IndexFilter;
import org.apache.comet.parquet.ReadOptions;
import org.apache.comet.parquet.RowGroupFilter;
import org.apache.comet.parquet.RowGroupReader;
import org.apache.parquet.ParquetReadOptions;
import org.apache.parquet.Preconditions;
import org.apache.parquet.bytes.ByteBufferInputStream;
import org.apache.parquet.bytes.BytesInput;
import org.apache.parquet.bytes.BytesUtils;
import org.apache.parquet.column.ColumnDescriptor;
import org.apache.parquet.column.page.DataPage;
import org.apache.parquet.column.page.DataPageV1;
import org.apache.parquet.column.page.DataPageV2;
import org.apache.parquet.column.page.DictionaryPage;
import org.apache.parquet.column.page.PageReadStore;
import org.apache.parquet.compression.CompressionCodecFactory;
import org.apache.parquet.crypto.AesCipher;
import org.apache.parquet.crypto.FileDecryptionProperties;
import org.apache.parquet.crypto.InternalColumnDecryptionSetup;
import org.apache.parquet.crypto.InternalFileDecryptor;
import org.apache.parquet.crypto.ModuleCipherFactory;
import org.apache.parquet.crypto.ParquetCryptoRuntimeException;
import org.apache.parquet.filter2.compat.FilterCompat;
import org.apache.parquet.format.BlockCipher;
import org.apache.parquet.format.DataPageHeader;
import org.apache.parquet.format.DataPageHeaderV2;
import org.apache.parquet.format.DictionaryPageHeader;
import org.apache.parquet.format.FileCryptoMetaData;
import org.apache.parquet.format.PageHeader;
import org.apache.parquet.format.Util;
import org.apache.parquet.format.converter.ParquetMetadataConverter;
import org.apache.parquet.hadoop.ParquetFileWriter;
import org.apache.parquet.hadoop.metadata.BlockMetaData;
import org.apache.parquet.hadoop.metadata.ColumnChunkMetaData;
import org.apache.parquet.hadoop.metadata.ColumnPath;
import org.apache.parquet.hadoop.metadata.FileMetaData;
import org.apache.parquet.hadoop.metadata.ParquetMetadata;
import org.apache.parquet.hadoop.util.counters.BenchmarkCounter;
import org.apache.parquet.internal.column.columnindex.OffsetIndex;
import org.apache.parquet.internal.filter2.columnindex.ColumnIndexFilter;
import org.apache.parquet.internal.filter2.columnindex.ColumnIndexStore;
import org.apache.parquet.internal.filter2.columnindex.RowRanges;
import org.apache.parquet.io.InputFile;
import org.apache.parquet.io.ParquetDecodingException;
import org.apache.parquet.io.SeekableInputStream;
import org.apache.parquet.schema.PrimitiveType;
import org.apache.spark.sql.execution.metric.SQLMetric;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class FileReader
implements Closeable {
    private static final Logger LOG = LoggerFactory.getLogger(FileReader.class);
    private final ParquetMetadataConverter converter;
    private final SeekableInputStream f;
    private final InputFile file;
    private final Map<String, SQLMetric> metrics;
    private final Map<ColumnPath, ColumnDescriptor> paths = new HashMap<ColumnPath, ColumnDescriptor>();
    private final FileMetaData fileMetaData;
    private final List<BlockMetaData> blocks;
    private final List<ColumnIndexReader> blockIndexStores;
    private final List<RowRanges> blockRowRanges;
    private final CRC32 crc;
    private final ParquetMetadata footer;
    private final ParquetReadOptions options;
    private final ReadOptions cometOptions;
    private int currentBlock = 0;
    private RowGroupReader currentRowGroup = null;
    private InternalFileDecryptor fileDecryptor;

    public FileReader(InputFile file, ParquetReadOptions options, ReadOptions cometOptions) throws IOException {
        this(file, null, options, cometOptions, null);
    }

    public FileReader(InputFile file, ParquetReadOptions options, ReadOptions cometOptions, Map<String, SQLMetric> metrics) throws IOException {
        this(file, null, options, cometOptions, metrics);
    }

    public FileReader(InputFile file, ParquetMetadata footer, ParquetReadOptions options, ReadOptions cometOptions, Map<String, SQLMetric> metrics) throws IOException {
        this.converter = new ParquetMetadataConverter(options);
        this.file = file;
        this.f = file.newStream();
        this.options = options;
        this.cometOptions = cometOptions;
        this.metrics = metrics;
        if (footer == null) {
            try {
                footer = FileReader.readFooter(file, options, this.f, this.converter);
            }
            catch (Exception e) {
                this.f.close();
                throw e;
            }
        }
        this.footer = footer;
        this.fileMetaData = footer.getFileMetaData();
        this.fileDecryptor = this.fileMetaData.getFileDecryptor();
        if (null != this.fileDecryptor && this.fileDecryptor.plaintextFile()) {
            this.fileDecryptor = null;
        }
        this.blocks = this.filterRowGroups(footer.getBlocks());
        this.blockIndexStores = FileReader.listWithNulls(this.blocks.size());
        this.blockRowRanges = FileReader.listWithNulls(this.blocks.size());
        for (ColumnDescriptor col : footer.getFileMetaData().getSchema().getColumns()) {
            this.paths.put(ColumnPath.get((String[])col.getPath()), col);
        }
        this.crc = options.usePageChecksumVerification() ? new CRC32() : null;
    }

    public ParquetMetadata getFooter() {
        return this.footer;
    }

    public FileMetaData getFileMetaData() {
        return this.fileMetaData;
    }

    public SeekableInputStream getInputStream() {
        return this.f;
    }

    public ParquetReadOptions getOptions() {
        return this.options;
    }

    public List<BlockMetaData> getRowGroups() {
        return this.blocks;
    }

    public void setRequestedSchema(List<ColumnDescriptor> projection) {
        this.paths.clear();
        for (ColumnDescriptor col : projection) {
            this.paths.put(ColumnPath.get((String[])col.getPath()), col);
        }
    }

    public long getRecordCount() {
        long total = 0L;
        for (BlockMetaData block : this.blocks) {
            total += block.getRowCount();
        }
        return total;
    }

    public long getFilteredRecordCount() {
        if (!this.options.useColumnIndexFilter() || !FilterCompat.isFilteringRequired((FilterCompat.Filter)this.options.getRecordFilter())) {
            return this.getRecordCount();
        }
        long total = 0L;
        int n = this.blocks.size();
        for (int i = 0; i < n; ++i) {
            total += this.getRowRanges(i).rowCount();
        }
        return total;
    }

    public boolean skipNextRowGroup() {
        return this.advanceToNextBlock();
    }

    public PageReadStore readNextRowGroup() throws IOException {
        if (this.currentBlock == this.blocks.size()) {
            return null;
        }
        BlockMetaData block = this.blocks.get(this.currentBlock);
        if (block.getRowCount() == 0L) {
            throw new RuntimeException("Illegal row group of 0 rows");
        }
        this.currentRowGroup = new RowGroupReader(block.getRowCount());
        ArrayList<ConsecutivePartList> allParts = new ArrayList<ConsecutivePartList>();
        ConsecutivePartList currentParts = null;
        for (ColumnChunkMetaData mc : block.getColumns()) {
            ColumnPath pathKey = mc.getPath();
            ColumnDescriptor columnDescriptor = this.paths.get(pathKey);
            if (columnDescriptor == null) continue;
            BenchmarkCounter.incrementTotalBytes((long)mc.getTotalSize());
            long startingPos = mc.getStartingPos();
            boolean mergeRanges = this.cometOptions.isIOMergeRangesEnabled();
            int mergeRangeDelta = this.cometOptions.getIOMergeRangesDelta();
            if (currentParts == null || !mergeRanges && currentParts.endPos() != startingPos || mergeRanges && startingPos - currentParts.endPos() > (long)mergeRangeDelta) {
                currentParts = new ConsecutivePartList(startingPos);
                allParts.add(currentParts);
            }
            long delta = startingPos - currentParts.endPos();
            if (mergeRanges && delta > 0L && delta <= (long)mergeRangeDelta) {
                currentParts.addChunk(new ChunkDescriptor(null, null, startingPos, delta));
            }
            currentParts.addChunk(new ChunkDescriptor(columnDescriptor, mc, startingPos, mc.getTotalSize()));
        }
        return this.readChunks(block, allParts, new ChunkListBuilder());
    }

    public PageReadStore readNextFilteredRowGroup() throws IOException {
        if (this.currentBlock == this.blocks.size()) {
            return null;
        }
        if (!this.options.useColumnIndexFilter() || !FilterCompat.isFilteringRequired((FilterCompat.Filter)this.options.getRecordFilter())) {
            return this.readNextRowGroup();
        }
        BlockMetaData block = this.blocks.get(this.currentBlock);
        if (block.getRowCount() == 0L) {
            throw new RuntimeException("Illegal row group of 0 rows");
        }
        ColumnIndexReader ciStore = this.getColumnIndexReader(this.currentBlock);
        RowRanges rowRanges = this.getRowRanges(this.currentBlock);
        long rowCount = rowRanges.rowCount();
        if (rowCount == 0L) {
            this.advanceToNextBlock();
            return this.readNextFilteredRowGroup();
        }
        if (rowCount == block.getRowCount()) {
            return this.readNextRowGroup();
        }
        this.currentRowGroup = new RowGroupReader(rowRanges);
        ChunkListBuilder builder = new ChunkListBuilder();
        ArrayList<ConsecutivePartList> allParts = new ArrayList<ConsecutivePartList>();
        ConsecutivePartList currentParts = null;
        for (ColumnChunkMetaData mc : block.getColumns()) {
            ColumnPath pathKey = mc.getPath();
            ColumnDescriptor columnDescriptor = this.paths.get(pathKey);
            if (columnDescriptor == null) continue;
            OffsetIndex offsetIndex = ciStore.getOffsetIndex(mc.getPath());
            IndexFilter indexFilter = new IndexFilter(rowRanges, offsetIndex, block.getRowCount());
            OffsetIndex filteredOffsetIndex = indexFilter.filterOffsetIndex();
            for (IndexFilter.OffsetRange range : indexFilter.calculateOffsetRanges(filteredOffsetIndex, mc)) {
                BenchmarkCounter.incrementTotalBytes((long)range.length);
                long startingPos = range.offset;
                if (currentParts == null || currentParts.endPos() != startingPos) {
                    currentParts = new ConsecutivePartList(startingPos);
                    allParts.add(currentParts);
                }
                ChunkDescriptor chunkDescriptor = new ChunkDescriptor(columnDescriptor, mc, startingPos, range.length);
                currentParts.addChunk(chunkDescriptor);
                builder.setOffsetIndex(chunkDescriptor, filteredOffsetIndex);
            }
        }
        return this.readChunks(block, allParts, builder);
    }

    ColumnIndexReader getColumnIndexReader(int blockIndex) {
        ColumnIndexReader ciStore = this.blockIndexStores.get(blockIndex);
        if (ciStore == null) {
            ciStore = ColumnIndexReader.create(this.blocks.get(blockIndex), this.paths.keySet(), this.fileDecryptor, this.f);
            this.blockIndexStores.set(blockIndex, ciStore);
        }
        return ciStore;
    }

    private PageReadStore readChunks(BlockMetaData block, List<ConsecutivePartList> allParts, ChunkListBuilder builder) throws IOException {
        if (this.shouldReadParallel()) {
            this.readAllPartsParallel(allParts, builder);
        } else {
            for (ConsecutivePartList consecutiveChunks : allParts) {
                consecutiveChunks.readAll(this.f, builder);
            }
        }
        for (Chunk chunk : builder.build()) {
            this.readChunkPages(chunk, block);
        }
        this.advanceToNextBlock();
        return this.currentRowGroup;
    }

    private boolean shouldReadParallel() {
        if (this.file instanceof CometInputFile) {
            URI uri = ((CometInputFile)this.file).getPath().toUri();
            return FileReader.shouldReadParallel(this.cometOptions, uri.getScheme());
        }
        return false;
    }

    static boolean shouldReadParallel(ReadOptions options, String scheme) {
        return options.isParallelIOEnabled() && FileReader.shouldReadParallelForScheme(scheme);
    }

    private static boolean shouldReadParallelForScheme(String scheme) {
        if (scheme == null) {
            return false;
        }
        switch (scheme) {
            case "s3a": {
                return true;
            }
        }
        return false;
    }

    List<ReadRange> getReadRanges(List<ConsecutivePartList> allParts, int nBuffers) {
        int nThreads = this.cometOptions.parallelIOThreadPoolSize();
        long buffersPerThread = nBuffers / nThreads + 1;
        boolean adjustSkew = this.cometOptions.adjustReadRangesSkew();
        ArrayList<ReadRange> allRanges = new ArrayList<ReadRange>();
        for (ConsecutivePartList consecutiveChunk : allParts) {
            ReadRange readRange = null;
            long offset = consecutiveChunk.offset;
            for (int i = 0; i < consecutiveChunk.buffers.size(); ++i) {
                if (adjustSkew && (long)i % buffersPerThread == 0L || i == 0) {
                    readRange = new ReadRange();
                    allRanges.add(readRange);
                    readRange.offset = offset;
                }
                ByteBuffer b = consecutiveChunk.buffers.get(i);
                readRange.length += (long)b.capacity();
                readRange.buffers.add(b);
                offset += (long)b.capacity();
            }
        }
        if (LOG.isDebugEnabled()) {
            StringBuilder sb = new StringBuilder();
            for (int i = 0; i < allRanges.size(); ++i) {
                sb.append(((ReadRange)allRanges.get(i)).toString());
                if (i >= allRanges.size() - 1) continue;
                sb.append(",");
            }
            LOG.debug("Read Ranges: {}", (Object)sb);
        }
        return allRanges;
    }

    private void readAllRangesParallel(List<ReadRange> allRanges) {
        int nThreads = this.cometOptions.parallelIOThreadPoolSize();
        ExecutorService threadPool = CometFileReaderThreadPool.getOrCreateThreadPool(nThreads);
        ArrayList<Future<Void>> futures = new ArrayList<Future<Void>>();
        for (ReadRange readRange : allRanges) {
            futures.add(threadPool.submit(() -> {
                try (SeekableInputStream inputStream = null;){
                    if (this.file instanceof CometInputFile) {
                        inputStream = ((CometInputFile)this.file).newStream(readRange.offset, readRange.length);
                        LOG.debug("Opened new input file: {}, at offset: {}", (Object)((CometInputFile)this.file).getPath().getName(), (Object)readRange.offset);
                    } else {
                        inputStream = this.file.newStream();
                    }
                    long curPos = readRange.offset;
                    for (ByteBuffer buffer : readRange.buffers) {
                        inputStream.seek(curPos);
                        LOG.debug("Thread: {} Offset: {} Size: {}", new Object[]{Thread.currentThread().getId(), curPos, buffer.capacity()});
                        inputStream.readFully(buffer);
                        buffer.flip();
                        curPos += (long)buffer.capacity();
                    }
                }
                return null;
            }));
        }
        for (Future future : futures) {
            try {
                future.get();
            }
            catch (InterruptedException | ExecutionException e) {
                throw new RuntimeException(e);
            }
        }
    }

    public void readAllPartsParallel(List<ConsecutivePartList> allParts, ChunkListBuilder builder) throws IOException {
        int nBuffers = 0;
        for (ConsecutivePartList consecutiveChunks : allParts) {
            consecutiveChunks.allocateReadBuffers();
            nBuffers += consecutiveChunks.buffers.size();
        }
        List<ReadRange> allRanges = this.getReadRanges(allParts, nBuffers);
        long startNs = System.nanoTime();
        this.readAllRangesParallel(allRanges);
        for (ConsecutivePartList consecutiveChunks : allParts) {
            consecutiveChunks.setReadMetrics(startNs);
            ByteBufferInputStream stream = ByteBufferInputStream.wrap(consecutiveChunks.buffers);
            BenchmarkCounter.incrementBytesRead((long)consecutiveChunks.length);
            for (int i = 0; i < consecutiveChunks.chunks.size(); ++i) {
                ChunkDescriptor descriptor = consecutiveChunks.chunks.get(i);
                if (descriptor.col != null) {
                    builder.add(descriptor, stream.sliceBuffers(descriptor.size));
                    continue;
                }
                stream.skipFully(descriptor.size);
            }
        }
    }

    private void readChunkPages(Chunk chunk, BlockMetaData block) throws IOException {
        if (this.fileDecryptor == null || this.fileDecryptor.plaintextFile()) {
            this.currentRowGroup.addColumn(chunk.descriptor.col, chunk.readAllPages());
            return;
        }
        ColumnPath columnPath = ColumnPath.get((String[])chunk.descriptor.col.getPath());
        InternalColumnDecryptionSetup columnDecryptionSetup = this.fileDecryptor.getColumnSetup(columnPath);
        if (!columnDecryptionSetup.isEncrypted()) {
            this.currentRowGroup.addColumn(chunk.descriptor.col, chunk.readAllPages());
        } else {
            this.currentRowGroup.addColumn(chunk.descriptor.col, chunk.readAllPages(columnDecryptionSetup.getMetaDataDecryptor(), columnDecryptionSetup.getDataDecryptor(), this.fileDecryptor.getFileAAD(), block.getOrdinal(), columnDecryptionSetup.getOrdinal()));
        }
    }

    private boolean advanceToNextBlock() {
        if (this.currentBlock == this.blocks.size()) {
            return false;
        }
        ++this.currentBlock;
        return true;
    }

    public long[] getRowIndices() {
        long[] rowIndices = new long[this.blocks.size() * 2];
        int n = this.blocks.size();
        for (int i = 0; i < n; ++i) {
            BlockMetaData block = this.blocks.get(i);
            rowIndices[i * 2] = this.getRowIndexOffset(block);
            rowIndices[i * 2 + 1] = block.getRowCount();
        }
        return rowIndices;
    }

    private long getRowIndexOffset(BlockMetaData metaData) {
        try {
            Method method = BlockMetaData.class.getMethod("getRowIndexOffset", new Class[0]);
            method.setAccessible(true);
            return (Long)method.invoke((Object)metaData, new Object[0]);
        }
        catch (Exception e) {
            throw new RuntimeException("Error when calling getRowIndexOffset", e);
        }
    }

    private RowRanges getRowRanges(int blockIndex) {
        Preconditions.checkState((boolean)FilterCompat.isFilteringRequired((FilterCompat.Filter)this.options.getRecordFilter()), (String)"Should not be invoked if filter is null or NOOP");
        RowRanges rowRanges = this.blockRowRanges.get(blockIndex);
        if (rowRanges == null) {
            rowRanges = ColumnIndexFilter.calculateRowRanges((FilterCompat.Filter)this.options.getRecordFilter(), (ColumnIndexStore)this.getColumnIndexReader(blockIndex), this.paths.keySet(), (long)this.blocks.get(blockIndex).getRowCount());
            this.blockRowRanges.set(blockIndex, rowRanges);
        }
        return rowRanges;
    }

    private static ParquetMetadata readFooter(InputFile file, ParquetReadOptions options, SeekableInputStream f, ParquetMetadataConverter converter) throws IOException {
        boolean encryptedFooterMode;
        long fileLen = file.getLength();
        String filePath = file.toString();
        LOG.debug("File length {}", (Object)fileLen);
        int FOOTER_LENGTH_SIZE = 4;
        if (fileLen < (long)(ParquetFileWriter.MAGIC.length + FOOTER_LENGTH_SIZE + ParquetFileWriter.MAGIC.length)) {
            throw new RuntimeException(filePath + " is not a Parquet file (length is too low: " + fileLen + ")");
        }
        byte[] magic = new byte[ParquetFileWriter.MAGIC.length];
        long fileMetadataLengthIndex = fileLen - (long)magic.length - (long)FOOTER_LENGTH_SIZE;
        LOG.debug("reading footer index at {}", (Object)fileMetadataLengthIndex);
        f.seek(fileMetadataLengthIndex);
        int fileMetadataLength = BytesUtils.readIntLittleEndian((InputStream)f);
        f.readFully(magic);
        if (Arrays.equals(ParquetFileWriter.MAGIC, magic)) {
            encryptedFooterMode = false;
        } else if (Arrays.equals(ParquetFileWriter.EFMAGIC, magic)) {
            encryptedFooterMode = true;
        } else {
            throw new RuntimeException(filePath + " is not a Parquet file. Expected magic number at tail, but found " + Arrays.toString(magic));
        }
        long fileMetadataIndex = fileMetadataLengthIndex - (long)fileMetadataLength;
        LOG.debug("read footer length: {}, footer index: {}", (Object)fileMetadataLength, (Object)fileMetadataIndex);
        if (fileMetadataIndex < (long)magic.length || fileMetadataIndex >= fileMetadataLengthIndex) {
            throw new RuntimeException("corrupted file: the footer index is not within the file: " + fileMetadataIndex);
        }
        f.seek(fileMetadataIndex);
        FileDecryptionProperties fileDecryptionProperties = options.getDecryptionProperties();
        InternalFileDecryptor fileDecryptor = null;
        if (null != fileDecryptionProperties) {
            fileDecryptor = new InternalFileDecryptor(fileDecryptionProperties);
        }
        byte[] footerBytes = new byte[fileMetadataLength];
        f.readFully(footerBytes);
        ByteBuffer footerBytesBuffer = ByteBuffer.wrap(footerBytes);
        LOG.debug("Finished to read all footer bytes.");
        ByteBufferInputStream footerBytesStream = ByteBufferInputStream.wrap((ByteBuffer[])new ByteBuffer[]{footerBytesBuffer});
        if (!encryptedFooterMode) {
            return converter.readParquetMetadata((InputStream)footerBytesStream, options.getMetadataFilter(), fileDecryptor, false, fileMetadataLength);
        }
        if (fileDecryptor == null) {
            throw new ParquetCryptoRuntimeException("Trying to read file with encrypted footer. No keys available");
        }
        FileCryptoMetaData fileCryptoMetaData = Util.readFileCryptoMetaData((InputStream)footerBytesStream);
        fileDecryptor.setFileCryptoMetaData(fileCryptoMetaData.getEncryption_algorithm(), true, fileCryptoMetaData.getKey_metadata());
        return converter.readParquetMetadata((InputStream)footerBytesStream, options.getMetadataFilter(), fileDecryptor, true, 0);
    }

    private List<BlockMetaData> filterRowGroups(List<BlockMetaData> blocks) {
        FilterCompat.Filter recordFilter = this.options.getRecordFilter();
        if (FilterCompat.isFilteringRequired((FilterCompat.Filter)recordFilter)) {
            ArrayList<RowGroupFilter.FilterLevel> levels = new ArrayList<RowGroupFilter.FilterLevel>();
            if (this.options.useStatsFilter()) {
                levels.add(RowGroupFilter.FilterLevel.STATISTICS);
            }
            if (this.options.useDictionaryFilter()) {
                levels.add(RowGroupFilter.FilterLevel.DICTIONARY);
            }
            if (this.options.useBloomFilter()) {
                levels.add(RowGroupFilter.FilterLevel.BLOOMFILTER);
            }
            return RowGroupFilter.filterRowGroups(levels, recordFilter, blocks, this);
        }
        return blocks;
    }

    private static <T> List<T> listWithNulls(int size) {
        return Stream.generate(() -> null).limit(size).collect(Collectors.toList());
    }

    public void closeStream() throws IOException {
        if (this.f != null) {
            this.f.close();
        }
    }

    @Override
    public void close() throws IOException {
        try {
            if (this.f != null) {
                this.f.close();
            }
        }
        finally {
            this.options.getCodecFactory().release();
        }
    }

    private static class ChunkDescriptor {
        private final ColumnDescriptor col;
        private final ColumnChunkMetaData metadata;
        private final long fileOffset;
        private final long size;

        ChunkDescriptor(ColumnDescriptor col, ColumnChunkMetaData metadata, long fileOffset, long size) {
            this.col = col;
            this.metadata = metadata;
            this.fileOffset = fileOffset;
            this.size = size;
        }

        public int hashCode() {
            return this.col.hashCode();
        }

        public boolean equals(Object obj) {
            if (this == obj) {
                return true;
            }
            if (obj instanceof ChunkDescriptor) {
                return this.col.equals((Object)((ChunkDescriptor)obj).col);
            }
            return false;
        }
    }

    private class ConsecutivePartList {
        private final long offset;
        private final List<ChunkDescriptor> chunks = new ArrayList<ChunkDescriptor>();
        private long length;
        private final SQLMetric fileReadTimeMetric;
        private final SQLMetric fileReadSizeMetric;
        private final SQLMetric readThroughput;
        List<ByteBuffer> buffers;

        ConsecutivePartList(long offset) {
            if (FileReader.this.metrics != null) {
                this.fileReadTimeMetric = FileReader.this.metrics.get("ParquetInputFileReadTime");
                this.fileReadSizeMetric = FileReader.this.metrics.get("ParquetInputFileReadSize");
                this.readThroughput = FileReader.this.metrics.get("ParquetInputFileReadThroughput");
            } else {
                this.fileReadTimeMetric = null;
                this.fileReadSizeMetric = null;
                this.readThroughput = null;
            }
            this.offset = offset;
        }

        public void addChunk(ChunkDescriptor descriptor) {
            this.chunks.add(descriptor);
            this.length += descriptor.size;
        }

        private void allocateReadBuffers() {
            int fullAllocations = Math.toIntExact(this.length / (long)FileReader.this.options.getMaxAllocationSize());
            int lastAllocationSize = Math.toIntExact(this.length % (long)FileReader.this.options.getMaxAllocationSize());
            int numAllocations = fullAllocations + (lastAllocationSize > 0 ? 1 : 0);
            this.buffers = new ArrayList<ByteBuffer>(numAllocations);
            for (int i = 0; i < fullAllocations; ++i) {
                this.buffers.add(FileReader.this.options.getAllocator().allocate(FileReader.this.options.getMaxAllocationSize()));
            }
            if (lastAllocationSize > 0) {
                this.buffers.add(FileReader.this.options.getAllocator().allocate(lastAllocationSize));
            }
        }

        public void readAll(SeekableInputStream f, ChunkListBuilder builder) throws IOException {
            f.seek(this.offset);
            this.allocateReadBuffers();
            long startNs = System.nanoTime();
            for (ByteBuffer buffer : this.buffers) {
                f.readFully(buffer);
                buffer.flip();
            }
            this.setReadMetrics(startNs);
            BenchmarkCounter.incrementBytesRead((long)this.length);
            ByteBufferInputStream stream = ByteBufferInputStream.wrap(this.buffers);
            for (int i = 0; i < this.chunks.size(); ++i) {
                ChunkDescriptor descriptor = this.chunks.get(i);
                if (descriptor.col != null) {
                    builder.add(descriptor, stream.sliceBuffers(descriptor.size));
                    continue;
                }
                stream.skipFully(descriptor.size);
            }
        }

        private void setReadMetrics(long startNs) {
            long totalFileReadTimeNs = System.nanoTime() - startNs;
            double sizeInMb = (double)this.length / 1048576.0;
            double timeInSec = (double)totalFileReadTimeNs / 1.0E11;
            double throughput = sizeInMb / timeInSec;
            LOG.debug("Comet: File Read stats:  Length: {} MB, Time: {} secs, throughput: {} MB/sec ", new Object[]{sizeInMb, timeInSec, throughput});
            if (this.fileReadTimeMetric != null) {
                this.fileReadTimeMetric.add(totalFileReadTimeNs);
            }
            if (this.fileReadSizeMetric != null) {
                this.fileReadSizeMetric.add(this.length);
            }
            if (this.readThroughput != null) {
                this.readThroughput.set(throughput);
            }
        }

        public long endPos() {
            return this.offset + this.length;
        }
    }

    private class Chunk {
        private final ChunkDescriptor descriptor;
        private final ByteBufferInputStream stream;
        final OffsetIndex offsetIndex;

        Chunk(ChunkDescriptor descriptor, List<ByteBuffer> buffers, OffsetIndex offsetIndex) {
            this.descriptor = descriptor;
            this.stream = ByteBufferInputStream.wrap(buffers);
            this.offsetIndex = offsetIndex;
        }

        protected PageHeader readPageHeader(BlockCipher.Decryptor blockDecryptor, byte[] pageHeaderAAD) throws IOException {
            return Util.readPageHeader((InputStream)this.stream, (BlockCipher.Decryptor)blockDecryptor, (byte[])pageHeaderAAD);
        }

        private void verifyCrc(int referenceCrc, byte[] bytes, String exceptionMsg) {
            FileReader.this.crc.reset();
            FileReader.this.crc.update(bytes);
            if (FileReader.this.crc.getValue() != ((long)referenceCrc & 0xFFFFFFFFL)) {
                throw new ParquetDecodingException(exceptionMsg);
            }
        }

        private ColumnPageReader readAllPages() throws IOException {
            return this.readAllPages(null, null, null, -1, -1);
        }

        private ColumnPageReader readAllPages(BlockCipher.Decryptor headerBlockDecryptor, BlockCipher.Decryptor pageBlockDecryptor, byte[] aadPrefix, int rowGroupOrdinal, int columnOrdinal) throws IOException {
            ArrayList<DataPage> pagesInChunk = new ArrayList<DataPage>();
            DictionaryPage dictionaryPage = null;
            PrimitiveType type = FileReader.this.fileMetaData.getSchema().getType(this.descriptor.col.getPath()).asPrimitiveType();
            long valuesCountReadSoFar = 0L;
            int dataPageCountReadSoFar = 0;
            byte[] dataPageHeaderAAD = null;
            if (null != headerBlockDecryptor) {
                dataPageHeaderAAD = AesCipher.createModuleAAD((byte[])aadPrefix, (ModuleCipherFactory.ModuleType)ModuleCipherFactory.ModuleType.DataPageHeader, (int)rowGroupOrdinal, (int)columnOrdinal, (int)this.getPageOrdinal(dataPageCountReadSoFar));
            }
            block5: while (this.hasMorePages(valuesCountReadSoFar, dataPageCountReadSoFar)) {
                byte[] pageHeaderAAD = dataPageHeaderAAD;
                if (null != headerBlockDecryptor) {
                    if (null == dictionaryPage && this.descriptor.metadata.hasDictionaryPage()) {
                        pageHeaderAAD = AesCipher.createModuleAAD((byte[])aadPrefix, (ModuleCipherFactory.ModuleType)ModuleCipherFactory.ModuleType.DictionaryPageHeader, (int)rowGroupOrdinal, (int)columnOrdinal, (int)-1);
                    } else {
                        int pageOrdinal = this.getPageOrdinal(dataPageCountReadSoFar);
                        AesCipher.quickUpdatePageAAD((byte[])dataPageHeaderAAD, (int)pageOrdinal);
                    }
                }
                PageHeader pageHeader = this.readPageHeader(headerBlockDecryptor, pageHeaderAAD);
                int uncompressedPageSize = pageHeader.getUncompressed_page_size();
                int compressedPageSize = pageHeader.getCompressed_page_size();
                switch (pageHeader.type) {
                    case DICTIONARY_PAGE: {
                        if (dictionaryPage != null) {
                            throw new ParquetDecodingException("more than one dictionary page in column " + this.descriptor.col);
                        }
                        BytesInput pageBytes = this.readAsBytesInput(compressedPageSize);
                        if (FileReader.this.options.usePageChecksumVerification() && pageHeader.isSetCrc()) {
                            this.verifyCrc(pageHeader.getCrc(), pageBytes.toByteArray(), "could not verify dictionary page integrity, CRC checksum verification failed");
                        }
                        DictionaryPageHeader dicHeader = pageHeader.getDictionary_page_header();
                        dictionaryPage = new DictionaryPage(pageBytes, uncompressedPageSize, dicHeader.getNum_values(), FileReader.this.converter.getEncoding(dicHeader.getEncoding()));
                        if (!pageHeader.isSetCrc()) continue block5;
                        dictionaryPage.setCrc(pageHeader.getCrc());
                        continue block5;
                    }
                    case DATA_PAGE: {
                        DataPageHeader dataHeaderV1 = pageHeader.getData_page_header();
                        BytesInput pageBytes = this.readAsBytesInput(compressedPageSize);
                        if (FileReader.this.options.usePageChecksumVerification() && pageHeader.isSetCrc()) {
                            this.verifyCrc(pageHeader.getCrc(), pageBytes.toByteArray(), "could not verify page integrity, CRC checksum verification failed");
                        }
                        DataPageV1 dataPageV1 = new DataPageV1(pageBytes, dataHeaderV1.getNum_values(), uncompressedPageSize, FileReader.this.converter.fromParquetStatistics(FileReader.this.getFileMetaData().getCreatedBy(), dataHeaderV1.getStatistics(), type), FileReader.this.converter.getEncoding(dataHeaderV1.getRepetition_level_encoding()), FileReader.this.converter.getEncoding(dataHeaderV1.getDefinition_level_encoding()), FileReader.this.converter.getEncoding(dataHeaderV1.getEncoding()));
                        if (pageHeader.isSetCrc()) {
                            dataPageV1.setCrc(pageHeader.getCrc());
                        }
                        pagesInChunk.add((DataPage)dataPageV1);
                        valuesCountReadSoFar += (long)dataHeaderV1.getNum_values();
                        ++dataPageCountReadSoFar;
                        continue block5;
                    }
                    case DATA_PAGE_V2: {
                        DataPageHeaderV2 dataHeaderV2 = pageHeader.getData_page_header_v2();
                        int dataSize = compressedPageSize - dataHeaderV2.getRepetition_levels_byte_length() - dataHeaderV2.getDefinition_levels_byte_length();
                        pagesInChunk.add((DataPage)new DataPageV2(dataHeaderV2.getNum_rows(), dataHeaderV2.getNum_nulls(), dataHeaderV2.getNum_values(), this.readAsBytesInput(dataHeaderV2.getRepetition_levels_byte_length()), this.readAsBytesInput(dataHeaderV2.getDefinition_levels_byte_length()), FileReader.this.converter.getEncoding(dataHeaderV2.getEncoding()), this.readAsBytesInput(dataSize), uncompressedPageSize, FileReader.this.converter.fromParquetStatistics(FileReader.this.getFileMetaData().getCreatedBy(), dataHeaderV2.getStatistics(), type), dataHeaderV2.isIs_compressed()));
                        valuesCountReadSoFar += (long)dataHeaderV2.getNum_values();
                        ++dataPageCountReadSoFar;
                        continue block5;
                    }
                }
                LOG.debug("skipping page of type {} of size {}", (Object)pageHeader.getType(), (Object)compressedPageSize);
                this.stream.skipFully((long)compressedPageSize);
            }
            if (this.offsetIndex == null && valuesCountReadSoFar != this.descriptor.metadata.getValueCount()) {
                throw new IOException("Expected " + this.descriptor.metadata.getValueCount() + " values in column chunk at " + FileReader.this.file + " offset " + this.descriptor.metadata.getFirstDataPageOffset() + " but got " + valuesCountReadSoFar + " values instead over " + pagesInChunk.size() + " pages ending at file offset " + (this.descriptor.fileOffset + this.stream.position()));
            }
            CompressionCodecFactory.BytesInputDecompressor decompressor = FileReader.this.options.getCodecFactory().getDecompressor(this.descriptor.metadata.getCodec());
            return new ColumnPageReader(decompressor, pagesInChunk, dictionaryPage, this.offsetIndex, FileReader.this.blocks.get(FileReader.this.currentBlock).getRowCount(), pageBlockDecryptor, aadPrefix, rowGroupOrdinal, columnOrdinal);
        }

        private boolean hasMorePages(long valuesCountReadSoFar, int dataPageCountReadSoFar) {
            return this.offsetIndex == null ? valuesCountReadSoFar < this.descriptor.metadata.getValueCount() : dataPageCountReadSoFar < this.offsetIndex.getPageCount();
        }

        private int getPageOrdinal(int dataPageCountReadSoFar) {
            if (null == this.offsetIndex) {
                return dataPageCountReadSoFar;
            }
            return this.offsetIndex.getPageOrdinal(dataPageCountReadSoFar);
        }

        public BytesInput readAsBytesInput(int size) throws IOException {
            return BytesInput.from((List)this.stream.sliceBuffers((long)size));
        }
    }

    private class ChunkListBuilder {
        private final Map<ChunkDescriptor, ChunkData> map = new HashMap<ChunkDescriptor, ChunkData>();

        private ChunkListBuilder() {
        }

        void add(ChunkDescriptor descriptor, List<ByteBuffer> buffers) {
            ChunkData data = this.map.get(descriptor);
            if (data == null) {
                data = new ChunkData();
                this.map.put(descriptor, data);
            }
            data.buffers.addAll(buffers);
        }

        void setOffsetIndex(ChunkDescriptor descriptor, OffsetIndex offsetIndex) {
            ChunkData data = this.map.get(descriptor);
            if (data == null) {
                data = new ChunkData();
                this.map.put(descriptor, data);
            }
            data.offsetIndex = offsetIndex;
        }

        List<Chunk> build() {
            ArrayList<Chunk> chunks = new ArrayList<Chunk>();
            for (Map.Entry<ChunkDescriptor, ChunkData> entry : this.map.entrySet()) {
                ChunkDescriptor descriptor = entry.getKey();
                ChunkData data = entry.getValue();
                chunks.add(new Chunk(descriptor, data.buffers, data.offsetIndex));
            }
            return chunks;
        }

        private class ChunkData {
            final List<ByteBuffer> buffers = new ArrayList<ByteBuffer>();
            OffsetIndex offsetIndex;

            private ChunkData() {
            }
        }
    }

    static class ReadRange {
        long offset = 0L;
        long length = 0L;
        List<ByteBuffer> buffers = new ArrayList<ByteBuffer>();

        ReadRange() {
        }

        public String toString() {
            return "ReadRange{offset=" + this.offset + ", length=" + this.length + ", numBuffers=" + this.buffers.size() + "}";
        }
    }
}

