/*
 * Decompiled with CFR 0.152.
 */
package com.ethlo.time.internal;

import com.ethlo.time.DateTime;
import com.ethlo.time.Field;
import com.ethlo.time.LeapSecondException;
import com.ethlo.time.TimezoneOffset;
import com.ethlo.time.internal.AbstractRfc3339;
import com.ethlo.time.internal.DefaultLeapSecondHandler;
import com.ethlo.time.internal.LeapSecondHandler;
import com.ethlo.time.internal.LimitedCharArrayIntegerUtil;
import com.ethlo.time.internal.W3cDateTimeUtil;
import java.time.DateTimeException;
import java.time.Month;
import java.time.OffsetDateTime;
import java.time.YearMonth;
import java.time.ZoneOffset;
import java.time.format.DateTimeParseException;
import java.util.Arrays;

public class EthloITU
extends AbstractRfc3339
implements W3cDateTimeUtil {
    public static final char DATE_SEPARATOR = '-';
    public static final char TIME_SEPARATOR = ':';
    public static final char SEPARATOR_UPPER = 'T';
    private static final EthloITU instance = new EthloITU();
    private static final char PLUS = '+';
    private static final char MINUS = '-';
    private static final char SEPARATOR_LOWER = 't';
    private static final char SEPARATOR_SPACE = ' ';
    private static final char FRACTION_SEPARATOR = '.';
    private static final char ZULU_UPPER = 'Z';
    private static final char ZULU_LOWER = 'z';
    private static final int[] widths = new int[]{100000000, 10000000, 1000000, 100000, 10000, 1000, 100, 10, 1};
    private static final LeapSecondHandler leapSecondHandler = new DefaultLeapSecondHandler();

    private EthloITU() {
    }

    public static EthloITU getInstance() {
        return instance;
    }

    public static String finish(char[] buf, int length, TimezoneOffset tz) {
        int tzLen = 0;
        if (tz != null) {
            tzLen = EthloITU.writeTz(buf, length, tz);
        }
        return new String(buf, 0, length + tzLen);
    }

    private static int writeTz(char[] buf, int start, TimezoneOffset tz) {
        if (tz.equals(TimezoneOffset.UTC)) {
            buf[start] = 90;
            return 1;
        }
        buf[start] = tz.getTotalSeconds() < 0 ? 45 : 43;
        LimitedCharArrayIntegerUtil.toString(Math.abs(tz.getHours()), buf, start + 1, 2);
        buf[start + 3] = 58;
        LimitedCharArrayIntegerUtil.toString(Math.abs(tz.getMinutes()), buf, start + 4, 2);
        return 6;
    }

    private static int scale(int fractions, int len, String parsedData, int index) {
        switch (len) {
            case 0: {
                throw new DateTimeParseException("Must have at least 1 fraction digit", parsedData, index);
            }
            case 1: {
                return fractions * 100000000;
            }
            case 2: {
                return fractions * 10000000;
            }
            case 3: {
                return fractions * 1000000;
            }
            case 4: {
                return fractions * 100000;
            }
            case 5: {
                return fractions * 10000;
            }
            case 6: {
                return fractions * 1000;
            }
            case 7: {
                return fractions * 100;
            }
            case 8: {
                return fractions * 10;
            }
        }
        return fractions;
    }

    private static Object handleTime(String chars, int year, int month, int day, int hour, int minute, boolean raw) {
        switch (chars.charAt(16)) {
            case ':': {
                return EthloITU.handleTime(year, month, day, hour, minute, chars, raw);
            }
            case '+': 
            case '-': 
            case 'Z': 
            case 'z': {
                TimezoneOffset zoneOffset = EthloITU.parseTimezone(chars, 16);
                if (!raw) {
                    throw EthloITU.raiseMissingField(Field.SECOND, chars, 16);
                }
                return DateTime.of(year, month, day, hour, minute, zoneOffset);
            }
        }
        throw new DateTimeParseException("Unexpected character at position 16: " + chars.charAt(16), chars, 16);
    }

    private static void assertPositionContains(String chars, int offset, char expected) {
        if (offset >= chars.length()) {
            EthloITU.raiseDateTimeException(chars, "Unexpected end of input", offset);
        }
        if (chars.charAt(offset) != expected) {
            throw new DateTimeParseException("Expected character " + expected + " at position " + (offset + 1) + " '" + chars + "'", chars, offset);
        }
    }

    private static void assertPositionContains(String chars, char ... expected) {
        if (10 >= chars.length()) {
            EthloITU.raiseDateTimeException(chars, "Unexpected end of input", 10);
        }
        boolean found = false;
        char needle = chars.charAt(10);
        for (char e : expected) {
            if (needle != e) continue;
            found = true;
            break;
        }
        if (!found) {
            throw new DateTimeParseException("Expected character " + Arrays.toString(expected) + " at position " + 11 + " '" + chars + "'", chars, 10);
        }
    }

    private static TimezoneOffset parseTimezone(String chars, int offset) {
        if (offset >= chars.length()) {
            throw new DateTimeParseException("No timezone information: " + chars, chars, offset);
        }
        int len = chars.length();
        int left = len - offset;
        char c = chars.charAt(offset);
        if (c == 'Z' || c == 'z') {
            EthloITU.assertNoMoreChars(chars, offset);
            return TimezoneOffset.UTC;
        }
        char sign = chars.charAt(offset);
        if (sign != '+' && sign != '-') {
            throw new DateTimeParseException("Invalid character starting at position " + offset + ": " + chars, chars, offset);
        }
        if (left != 6) {
            throw new DateTimeParseException("Invalid timezone offset: " + chars, chars, offset);
        }
        int hours = LimitedCharArrayIntegerUtil.parsePositiveInt(chars, offset + 1, offset + 3);
        int minutes = LimitedCharArrayIntegerUtil.parsePositiveInt(chars, offset + 4, offset + 4 + 2);
        if (sign == '-') {
            hours = -hours;
            minutes = -minutes;
        }
        if (sign == '-' && hours == 0 && minutes == 0) {
            throw new DateTimeParseException("Unknown 'Local Offset Convention' date-time not allowed", chars, offset);
        }
        return TimezoneOffset.ofHoursMinutes(hours, minutes);
    }

    private static void assertNoMoreChars(String chars, int lastUsed) {
        if (chars.length() > lastUsed + 1) {
            throw new DateTimeParseException("Trailing junk data after position " + (lastUsed + 1) + ": " + chars, chars, lastUsed + 1);
        }
    }

    private static Object parse(String chars, boolean raw) {
        if (chars == null) {
            throw new NullPointerException("text cannot be null");
        }
        int len = chars.length();
        int years = LimitedCharArrayIntegerUtil.parsePositiveInt(chars, 0, 4);
        if (4 == len) {
            if (!raw) {
                throw EthloITU.raiseMissingField(Field.YEAR, chars, 2);
            }
            return DateTime.ofYear(years);
        }
        EthloITU.assertPositionContains(chars, 4, '-');
        int months = LimitedCharArrayIntegerUtil.parsePositiveInt(chars, 5, 7);
        if (7 == len) {
            if (!raw) {
                throw EthloITU.raiseMissingField(Field.MONTH, chars, 5);
            }
            return DateTime.ofYearMonth(years, months);
        }
        EthloITU.assertPositionContains(chars, 7, '-');
        int days = LimitedCharArrayIntegerUtil.parsePositiveInt(chars, 8, 10);
        if (10 == len) {
            if (!raw) {
                throw EthloITU.raiseMissingField(Field.DAY, chars, 9);
            }
            return DateTime.ofDate(years, months, days);
        }
        EthloITU.assertPositionContains(chars, 'T', 't', ' ');
        int hours = LimitedCharArrayIntegerUtil.parsePositiveInt(chars, 11, 13);
        EthloITU.assertPositionContains(chars, 13, ':');
        int minutes = LimitedCharArrayIntegerUtil.parsePositiveInt(chars, 14, 16);
        if (len > 16) {
            return EthloITU.handleTime(chars, years, months, days, hours, minutes, raw);
        }
        if (raw) {
            return DateTime.of(years, months, days, hours, minutes, null);
        }
        throw EthloITU.raiseMissingField(Field.SECOND, chars, 16);
    }

    private static DateTimeException raiseMissingField(Field field, String chars, int offset) {
        return new DateTimeParseException("No " + field.name() + " field found", chars, offset);
    }

    private static Object handleTime(int year, int month, int day, int hour, int minute, String chars, boolean raw) {
        int len = chars.length();
        int remaining = len - 17;
        if (remaining == 2) {
            int seconds = LimitedCharArrayIntegerUtil.parsePositiveInt(chars, 17, 19);
            if (raw) {
                return new DateTime(Field.SECOND, year, month, day, hour, minute, seconds, 0, null, 0);
            }
            throw new DateTimeParseException("No timezone information: " + chars, chars, 19);
        }
        if (remaining == 0) {
            if (raw) {
                return new DateTime(Field.SECOND, year, month, day, hour, minute, 0, 0, null, 0);
            }
            throw new DateTimeParseException("No timezone information: " + chars, chars, 16);
        }
        TimezoneOffset offset = null;
        int fractions = 0;
        int fractionDigits = 0;
        if (chars.length() < 20) {
            throw new DateTimeParseException("Unexpected end of input: " + chars, chars, 16);
        }
        char c = chars.charAt(19);
        if (c == '.') {
            if (chars.length() < 21) {
                throw new DateTimeParseException("Unexpected end of input: " + chars, chars, 20);
            }
            int result = 0;
            int idx = 20;
            boolean nonDigitFound = false;
            do {
                if ((c = chars.charAt(idx)) < '0' || c > '9') {
                    nonDigitFound = true;
                    fractionDigits = idx - 20;
                    EthloITU.assertFractionDigits(chars, fractionDigits, idx);
                    fractions = EthloITU.scale(-result, fractionDigits, chars, idx);
                    offset = EthloITU.parseTimezone(chars, idx);
                    continue;
                }
                fractionDigits = idx - 19;
                EthloITU.assertFractionDigits(chars, fractionDigits, idx);
                result = (result << 1) + (result << 3);
                result -= c - 48;
            } while (++idx < len && !nonDigitFound);
            if (!nonDigitFound) {
                fractionDigits = idx - 20;
                fractions = EthloITU.scale(-result, fractionDigits, chars, idx);
                if (!raw) {
                    offset = EthloITU.parseTimezone(chars, idx);
                }
            }
        } else if (c == 'Z' || c == 'z') {
            offset = TimezoneOffset.UTC;
        } else if (c == '+' || c == '-') {
            offset = EthloITU.parseTimezone(chars, 19);
        } else {
            EthloITU.raiseDateTimeException(chars, "Unexpected character at position 19", 19);
        }
        int second = LimitedCharArrayIntegerUtil.parsePositiveInt(chars, 17, 19);
        if (!raw) {
            EthloITU.leapSecondCheck(year, month, day, hour, minute, second, fractions, offset);
            return OffsetDateTime.of(year, month, day, hour, minute, second, fractions, offset.toZoneOffset());
        }
        return fractionDigits > 0 ? DateTime.of(year, month, day, hour, minute, second, fractions, offset, fractionDigits) : DateTime.of(year, month, day, hour, minute, second, offset);
    }

    private static void assertFractionDigits(String chars, int fractionDigits, int idx) {
        if (fractionDigits > 9) {
            throw new DateTimeParseException("Too many fraction digits: " + chars, chars, idx);
        }
    }

    private static void leapSecondCheck(int year, int month, int day, int hour, int minute, int second, int nanos, TimezoneOffset offset) {
        YearMonth needle;
        boolean isValidLeapYearMonth;
        if (second == 60 && ((isValidLeapYearMonth = leapSecondHandler.isValidLeapSecondDate(needle = YearMonth.of(year, month))) || needle.isAfter(leapSecondHandler.getLastKnownLeapSecond()))) {
            int utcHour = hour - offset.getTotalSeconds() / 3600;
            int utcMinute = minute - offset.getTotalSeconds() % 3600 / 60;
            if ((month == Month.DECEMBER.getValue() && day == 31 || month == Month.JUNE.getValue() && day == 30) && utcHour == 23 && utcMinute == 59) {
                OffsetDateTime nearest = OffsetDateTime.of(year, month, day, hour, minute, 59, nanos, offset.toZoneOffset()).plusSeconds(1L);
                throw new LeapSecondException(nearest, second, isValidLeapYearMonth);
            }
        }
    }

    private static void raiseDateTimeException(String chars, String message, int index) {
        throw new DateTimeParseException(message + ": " + chars, chars, index);
    }

    @Override
    public String formatUtc(OffsetDateTime date, int fractionDigits) {
        return this.doFormat(date, ZoneOffset.UTC, Field.SECOND, fractionDigits);
    }

    @Override
    public String formatUtc(OffsetDateTime date, Field lastIncluded) {
        return this.doFormat(date, ZoneOffset.UTC, lastIncluded, 0);
    }

    @Override
    public String format(OffsetDateTime date, ZoneOffset adjustTo, int fractionDigits) {
        return this.doFormat(date, adjustTo, Field.NANO, fractionDigits);
    }

    private String doFormat(OffsetDateTime date, ZoneOffset adjustTo, Field lastIncluded, int fractionDigits) {
        boolean hasFractionDigits;
        this.assertMaxFractionDigits(fractionDigits);
        OffsetDateTime adjusted = date;
        if (!date.getOffset().equals(adjustTo)) {
            adjusted = date.atZoneSameInstant(adjustTo).toOffsetDateTime();
        }
        TimezoneOffset tz = TimezoneOffset.of(adjustTo);
        char[] buffer = new char[31];
        if (this.handleDatePart(lastIncluded, buffer, adjusted.getYear(), 0, 4, Field.YEAR)) {
            return EthloITU.finish(buffer, Field.YEAR.getRequiredLength(), null);
        }
        buffer[4] = 45;
        if (this.handleDatePart(lastIncluded, buffer, adjusted.getMonthValue(), 5, 2, Field.MONTH)) {
            return EthloITU.finish(buffer, Field.MONTH.getRequiredLength(), null);
        }
        buffer[7] = 45;
        if (this.handleDatePart(lastIncluded, buffer, adjusted.getDayOfMonth(), 8, 2, Field.DAY)) {
            return EthloITU.finish(buffer, Field.DAY.getRequiredLength(), null);
        }
        buffer[10] = 84;
        LimitedCharArrayIntegerUtil.toString(adjusted.getHour(), buffer, 11, 2);
        buffer[13] = 58;
        if (this.handleDatePart(lastIncluded, buffer, adjusted.getMinute(), 14, 2, Field.MINUTE)) {
            return EthloITU.finish(buffer, Field.MINUTE.getRequiredLength(), tz);
        }
        buffer[16] = 58;
        LimitedCharArrayIntegerUtil.toString(adjusted.getSecond(), buffer, 17, 2);
        boolean bl = hasFractionDigits = fractionDigits > 0;
        if (hasFractionDigits) {
            buffer[19] = 46;
            this.addFractions(buffer, fractionDigits, adjusted.getNano());
            return EthloITU.finish(buffer, 20 + fractionDigits, tz);
        }
        return EthloITU.finish(buffer, 19, tz);
    }

    private boolean handleDatePart(Field lastIncluded, char[] buffer, int value, int offset, int length, Field field) {
        LimitedCharArrayIntegerUtil.toString(value, buffer, offset, length);
        return lastIncluded == field;
    }

    private void addFractions(char[] buf, int fractionDigits, int nano) {
        double d = widths[fractionDigits - 1];
        LimitedCharArrayIntegerUtil.toString((int)((double)nano / d), buf, 20, fractionDigits);
    }

    @Override
    public OffsetDateTime parseDateTime(String dateTime) {
        return (OffsetDateTime)EthloITU.parse(dateTime, false);
    }

    @Override
    public String formatUtcMilli(OffsetDateTime date) {
        return this.formatUtc(date, 3);
    }

    @Override
    public String formatUtcMicro(OffsetDateTime date) {
        return this.formatUtc(date, 6);
    }

    @Override
    public String formatUtcNano(OffsetDateTime date) {
        return this.formatUtc(date, 9);
    }

    @Override
    public String formatUtc(OffsetDateTime date) {
        return this.formatUtc(date, 0);
    }

    @Override
    public DateTime parse(String chars) {
        return (DateTime)EthloITU.parse(chars, true);
    }
}

