/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.milo.opcua.stack.core.security;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import io.netty.buffer.ByteBufUtil;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.net.URLEncoder;
import java.nio.file.ClosedWatchServiceException;
import java.nio.file.FileSystems;
import java.nio.file.StandardWatchEventKinds;
import java.nio.file.WatchKey;
import java.nio.file.WatchService;
import java.security.cert.CRL;
import java.security.cert.CRLException;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509CRL;
import java.security.cert.X509Certificate;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentMap;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.eclipse.milo.opcua.stack.core.UaException;
import org.eclipse.milo.opcua.stack.core.security.TrustListManager;
import org.eclipse.milo.opcua.stack.core.types.builtin.ByteString;
import org.eclipse.milo.opcua.stack.core.util.CertificateUtil;
import org.eclipse.milo.opcua.stack.core.util.DigestUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DefaultTrustListManager
implements TrustListManager,
AutoCloseable {
    static final int MAX_REJECTED_CERTIFICATES = 128;
    private static final Logger LOGGER = LoggerFactory.getLogger(DefaultTrustListManager.class);
    private final Set<X509Certificate> issuerCertificates = Sets.newConcurrentHashSet();
    private final Set<X509CRL> issuerCrls = Sets.newConcurrentHashSet();
    private final Set<X509Certificate> trustedCertificates = Sets.newConcurrentHashSet();
    private final Set<X509CRL> trustedCrls = Sets.newConcurrentHashSet();
    private final WatchService watchService;
    private final Thread watchThread;
    private final File baseDir;
    private final File issuerDir;
    private final File issuerCertsDir;
    private final File issuerCrlsDir;
    private final File trustedDir;
    private final File trustedCertsDir;
    private final File trustedCrlsDir;
    private final File rejectedDir;

    public DefaultTrustListManager(File baseDir) throws IOException {
        this.baseDir = baseDir;
        DefaultTrustListManager.ensureDirectoryExists(baseDir);
        this.issuerDir = baseDir.toPath().resolve("issuers").toFile();
        DefaultTrustListManager.ensureDirectoryExists(this.issuerDir);
        this.issuerCertsDir = this.issuerDir.toPath().resolve("certs").toFile();
        DefaultTrustListManager.ensureDirectoryExists(this.issuerCertsDir);
        this.issuerCrlsDir = this.issuerDir.toPath().resolve("crls").toFile();
        DefaultTrustListManager.ensureDirectoryExists(this.issuerCrlsDir);
        this.trustedDir = baseDir.toPath().resolve("trusted").toFile();
        DefaultTrustListManager.ensureDirectoryExists(this.trustedDir);
        this.trustedCertsDir = this.trustedDir.toPath().resolve("certs").toFile();
        DefaultTrustListManager.ensureDirectoryExists(this.trustedCertsDir);
        this.trustedCrlsDir = this.trustedDir.toPath().resolve("crls").toFile();
        DefaultTrustListManager.ensureDirectoryExists(this.trustedCrlsDir);
        this.rejectedDir = baseDir.toPath().resolve("rejected").toFile();
        DefaultTrustListManager.ensureDirectoryExists(this.rejectedDir);
        this.watchService = FileSystems.getDefault().newWatchService();
        ConcurrentMap watchKeys = Maps.newConcurrentMap();
        watchKeys.put(this.issuerCertsDir.toPath().register(this.watchService, StandardWatchEventKinds.ENTRY_CREATE, StandardWatchEventKinds.ENTRY_DELETE, StandardWatchEventKinds.ENTRY_MODIFY), this::synchronizeIssuerCerts);
        watchKeys.put(this.issuerCrlsDir.toPath().register(this.watchService, StandardWatchEventKinds.ENTRY_CREATE, StandardWatchEventKinds.ENTRY_DELETE, StandardWatchEventKinds.ENTRY_MODIFY), this::synchronizeIssuerCrls);
        watchKeys.put(this.trustedCertsDir.toPath().register(this.watchService, StandardWatchEventKinds.ENTRY_CREATE, StandardWatchEventKinds.ENTRY_DELETE, StandardWatchEventKinds.ENTRY_MODIFY), this::synchronizeTrustedCerts);
        watchKeys.put(this.trustedCrlsDir.toPath().register(this.watchService, StandardWatchEventKinds.ENTRY_CREATE, StandardWatchEventKinds.ENTRY_DELETE, StandardWatchEventKinds.ENTRY_MODIFY), this::synchronizeTrustedCrls);
        this.watchThread = new Thread(new Watcher(this.watchService, watchKeys));
        this.watchThread.setName("certificate-store-watcher");
        this.watchThread.setDaemon(true);
        this.watchThread.start();
        this.synchronizeIssuerCerts();
        this.synchronizeIssuerCrls();
        this.synchronizeTrustedCerts();
    }

    @Override
    public synchronized void close() throws IOException {
        LOGGER.debug("Closing DefaultCertificateStore at {}", (Object)this.baseDir.getAbsolutePath());
        this.watchService.close();
        try {
            this.watchThread.join(5000L);
        }
        catch (InterruptedException e) {
            throw new IOException(e);
        }
        this.issuerCertificates.clear();
        this.issuerCrls.clear();
        this.trustedCertificates.clear();
    }

    @Override
    public synchronized ImmutableList<X509CRL> getIssuerCrls() {
        return ImmutableList.copyOf(this.issuerCrls);
    }

    @Override
    public synchronized ImmutableList<X509CRL> getTrustedCrls() {
        return ImmutableList.copyOf(this.trustedCrls);
    }

    @Override
    public synchronized ImmutableList<X509Certificate> getIssuerCertificates() {
        return ImmutableList.copyOf(this.issuerCertificates);
    }

    @Override
    public synchronized ImmutableList<X509Certificate> getTrustedCertificates() {
        return ImmutableList.copyOf(this.trustedCertificates);
    }

    @Override
    public synchronized ImmutableList<X509Certificate> getRejectedCertificates() {
        File[] files = this.rejectedDir.listFiles();
        if (files == null) {
            files = new File[]{};
        }
        return ImmutableList.copyOf((Collection)Arrays.stream(files).flatMap(cert -> DefaultTrustListManager.decodeCertificateFile(cert).map(Stream::of).orElse(Stream.empty())).collect(Collectors.toList()));
    }

    @Override
    public synchronized void setIssuerCrls(List<X509CRL> issuerCrls) {
        this.replaceCrlsInDir(issuerCrls, this.issuerCrlsDir);
        this.synchronizeIssuerCrls();
    }

    @Override
    public synchronized void setTrustedCrls(List<X509CRL> trustedCrls) {
        this.replaceCrlsInDir(trustedCrls, this.trustedCrlsDir);
        this.synchronizeTrustedCrls();
    }

    @Override
    public synchronized void setIssuerCertificates(List<X509Certificate> issuerCertificates) {
        this.replaceCertificatesInDir(issuerCertificates, this.issuerCertsDir);
        this.synchronizeIssuerCerts();
    }

    @Override
    public synchronized void setTrustedCertificates(List<X509Certificate> trustedCertificates) {
        this.replaceCertificatesInDir(trustedCertificates, this.trustedCertsDir);
        this.synchronizeTrustedCerts();
    }

    @Override
    public synchronized void addIssuerCertificate(X509Certificate certificate) {
        this.issuerCertificates.add(certificate);
        DefaultTrustListManager.writeCertificateToDir(certificate, this.issuerCertsDir);
    }

    @Override
    public synchronized void addTrustedCertificate(X509Certificate certificate) {
        this.trustedCertificates.add(certificate);
        DefaultTrustListManager.writeCertificateToDir(certificate, this.trustedCertsDir);
    }

    @Override
    public synchronized void addRejectedCertificate(X509Certificate certificate) {
        this.pruneOldRejectedCertificates();
        DefaultTrustListManager.writeCertificateToDir(certificate, this.rejectedDir);
    }

    @Override
    public synchronized boolean removeIssuerCertificate(ByteString thumbprint) {
        boolean found = this.deleteCertificateFile(this.issuerCertsDir, thumbprint);
        this.synchronizeIssuerCerts();
        return found;
    }

    @Override
    public synchronized boolean removeTrustedCertificate(ByteString thumbprint) {
        boolean found = this.deleteCertificateFile(this.trustedCertsDir, thumbprint);
        this.synchronizeTrustedCerts();
        return found;
    }

    @Override
    public synchronized boolean removeRejectedCertificate(ByteString thumbprint) {
        return this.deleteCertificateFile(this.rejectedDir, thumbprint);
    }

    public File getBaseDir() {
        return this.baseDir;
    }

    public File getIssuerDir() {
        return this.issuerDir;
    }

    public File getIssuerCertsDir() {
        return this.issuerCertsDir;
    }

    public File getIssuerCrlsDir() {
        return this.issuerCrlsDir;
    }

    public File getTrustedDir() {
        return this.trustedDir;
    }

    public File getTrustedCertsDir() {
        return this.trustedCertsDir;
    }

    public File getTrustedCrlsDir() {
        return this.trustedCrlsDir;
    }

    public File getRejectedDir() {
        return this.rejectedDir;
    }

    private synchronized boolean deleteCertificateFile(File certificateDir, ByteString thumbprint) {
        File[] files = certificateDir.listFiles();
        if (files == null) {
            files = new File[]{};
        }
        for (File file : files) {
            boolean matchesThumbprint = DefaultTrustListManager.decodeCertificateFile(file).map(c -> {
                try {
                    ByteString bs = CertificateUtil.thumbprint(c);
                    return bs.equals(thumbprint);
                }
                catch (UaException e) {
                    return false;
                }
            }).orElse(false);
            if (!matchesThumbprint) continue;
            if (!file.delete()) {
                LOGGER.warn("Failed to delete issuer certificate: {}", (Object)file);
            }
            return true;
        }
        return false;
    }

    private synchronized void replaceCrlsInDir(List<X509CRL> crls, File dir) {
        File[] files = dir.listFiles();
        if (files == null) {
            files = new File[]{};
        }
        Arrays.stream(files).forEach(File::delete);
        crls.forEach(crl -> DefaultTrustListManager.writeCrlToDir(crl, dir));
    }

    private synchronized void replaceCertificatesInDir(List<X509Certificate> certificates, File dir) {
        File[] files = dir.listFiles();
        if (files == null) {
            files = new File[]{};
        }
        Arrays.stream(files).forEach(File::delete);
        certificates.forEach(certificate -> DefaultTrustListManager.writeCertificateToDir(certificate, dir));
    }

    synchronized void pruneOldRejectedCertificates() {
        File[] files = this.rejectedDir.listFiles();
        if (files != null && files.length >= 128) {
            int excessCount = files.length - 128;
            Arrays.stream(files).sorted((o1, o2) -> (int)(o1.lastModified() - o2.lastModified())).limit(excessCount + 1).forEach(file -> {
                if (!file.delete()) {
                    LOGGER.warn("Unable to delete rejected certificate: {}", file);
                }
            });
        }
    }

    private synchronized void synchronizeIssuerCerts() {
        LOGGER.debug("Synchronizing issuer certs...");
        File[] files = this.issuerCertsDir.listFiles();
        if (files == null) {
            files = new File[]{};
        }
        this.issuerCertificates.clear();
        Arrays.stream(files).flatMap(cert -> DefaultTrustListManager.decodeCertificateFile(cert).map(Stream::of).orElse(Stream.empty())).forEach(this.issuerCertificates::add);
    }

    private synchronized void synchronizeIssuerCrls() {
        LOGGER.debug("Synchronizing issuer CRLs...");
        File[] files = this.issuerCrlsDir.listFiles();
        if (files == null) {
            files = new File[]{};
        }
        this.issuerCrls.clear();
        Arrays.stream(files).flatMap(crl -> DefaultTrustListManager.decodeCrlFile(crl).map(Stream::of).orElse(Stream.empty())).flatMap(Collection::stream).forEach(this.issuerCrls::add);
    }

    private synchronized void synchronizeTrustedCerts() {
        LOGGER.debug("Synchronizing trusted certs...");
        File[] files = this.trustedCertsDir.listFiles();
        if (files == null) {
            files = new File[]{};
        }
        this.trustedCertificates.clear();
        Arrays.stream(files).flatMap(cert -> DefaultTrustListManager.decodeCertificateFile(cert).map(Stream::of).orElse(Stream.empty())).forEach(this.trustedCertificates::add);
    }

    private synchronized void synchronizeTrustedCrls() {
        LOGGER.debug("Synchronizing trusted CRLs...");
        File[] files = this.trustedCrlsDir.listFiles();
        if (files == null) {
            files = new File[]{};
        }
        this.trustedCrls.clear();
        Arrays.stream(files).flatMap(crl -> DefaultTrustListManager.decodeCrlFile(crl).map(Stream::of).orElse(Stream.empty())).flatMap(Collection::stream).forEach(this.trustedCrls::add);
    }

    private static void writeCrlToDir(X509CRL crl, File dir) {
        try {
            String filename = DefaultTrustListManager.getFilename(crl);
            File f = dir.toPath().resolve(filename).toFile();
            try (FileOutputStream fos = new FileOutputStream(f);){
                fos.write(crl.getEncoded());
                fos.flush();
            }
            LOGGER.debug("Wrote CRL entry: {}", (Object)f.getAbsolutePath());
        }
        catch (Exception e) {
            LOGGER.error("Error writing CRL", (Throwable)e);
        }
    }

    private static void writeCertificateToDir(X509Certificate certificate, File dir) {
        try {
            String filename = DefaultTrustListManager.getFilename(certificate);
            File f = dir.toPath().resolve(filename).toFile();
            try (FileOutputStream fos = new FileOutputStream(f);){
                fos.write(certificate.getEncoded());
                fos.flush();
            }
            LOGGER.debug("Wrote certificate entry: {}", (Object)f.getAbsolutePath());
        }
        catch (Exception e) {
            LOGGER.error("Error writing certificate", (Throwable)e);
        }
    }

    private static Optional<List<X509CRL>> decodeCrlFile(File f) {
        try {
            CertificateFactory factory = CertificateFactory.getInstance("X.509");
            Collection<? extends CRL> crls = factory.generateCRLs(new FileInputStream(f));
            return Optional.of(crls.stream().filter(crl -> crl instanceof X509CRL).map(X509CRL.class::cast).collect(Collectors.toList()));
        }
        catch (FileNotFoundException | CRLException | CertificateException e) {
            LOGGER.debug("Error decoding CRL file: {}", (Object)f.toString(), (Object)e);
            return Optional.empty();
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private static Optional<X509Certificate> decodeCertificateFile(File f) {
        try (FileInputStream inputStream = new FileInputStream(f);){
            Optional<X509Certificate> optional = Optional.of(CertificateUtil.decodeCertificate(inputStream));
            return optional;
        }
        catch (Throwable t) {
            LOGGER.debug("Error decoding certificate file: {}", (Object)f.toString(), (Object)t);
            return Optional.empty();
        }
    }

    private static String getFilename(X509CRL crl) throws Exception {
        String thumbprint = ByteBufUtil.hexDump((byte[])DigestUtil.sha1(crl.getEncoded()));
        return String.format("%s.crl", thumbprint);
    }

    private static String getFilename(X509Certificate certificate) throws Exception {
        String[] ss = certificate.getSubjectX500Principal().getName().split(",");
        String name = ss.length > 0 ? ss[0] : certificate.getSubjectX500Principal().getName();
        String thumbprint = ByteBufUtil.hexDump((byte[])DigestUtil.sha1(certificate.getSignature()));
        return String.format("%s [%s].der", thumbprint, URLEncoder.encode(name, "UTF-8"));
    }

    private static void ensureDirectoryExists(File dir) throws IOException {
        if (!dir.exists() && !dir.mkdirs()) {
            throw new IOException("unable to create directory at " + dir.getAbsolutePath());
        }
    }

    private static class Watcher
    implements Runnable {
        private final WatchService watchService;
        private final Map<WatchKey, Runnable> watchKeys;

        Watcher(WatchService watchService, Map<WatchKey, Runnable> watchKeys) {
            this.watchService = watchService;
            this.watchKeys = watchKeys;
        }

        @Override
        public void run() {
            while (true) {
                try {
                    WatchKey key;
                    do {
                        boolean synchronize;
                        if (!this.watchKeys.containsKey(key = this.watchService.take()) || !(synchronize = key.pollEvents().stream().anyMatch(e -> e.kind() != StandardWatchEventKinds.OVERFLOW))) continue;
                        this.watchKeys.get(key).run();
                    } while (key.reset());
                    LOGGER.warn("Failed to reset watch key");
                }
                catch (ClosedWatchServiceException e2) {
                    LOGGER.info("Watcher got closed");
                    return;
                }
                catch (InterruptedException e3) {
                    LOGGER.error("Watcher interrupted.", (Throwable)e3);
                    continue;
                }
                break;
            }
        }
    }
}

