/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.milo.opcua.stack.core.channel;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.CompositeByteBuf;
import io.netty.util.ReferenceCountUtil;
import java.nio.Buffer;
import java.nio.ByteBuffer;
import java.security.GeneralSecurityException;
import java.security.InvalidKeyException;
import java.security.Key;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.Signature;
import java.security.SignatureException;
import java.util.List;
import javax.crypto.Cipher;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.SecretKeySpec;
import org.eclipse.milo.opcua.stack.core.UaException;
import org.eclipse.milo.opcua.stack.core.channel.ChannelParameters;
import org.eclipse.milo.opcua.stack.core.channel.ChannelSecurity;
import org.eclipse.milo.opcua.stack.core.channel.MessageAbortedException;
import org.eclipse.milo.opcua.stack.core.channel.SecureChannel;
import org.eclipse.milo.opcua.stack.core.channel.headers.AsymmetricSecurityHeader;
import org.eclipse.milo.opcua.stack.core.channel.headers.SequenceHeader;
import org.eclipse.milo.opcua.stack.core.channel.headers.SymmetricSecurityHeader;
import org.eclipse.milo.opcua.stack.core.channel.messages.ErrorMessage;
import org.eclipse.milo.opcua.stack.core.security.SecurityAlgorithm;
import org.eclipse.milo.opcua.stack.core.util.BufferUtil;
import org.eclipse.milo.opcua.stack.core.util.SignatureUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class ChunkDecoder {
    private final AsymmetricDecoder asymmetricDecoder = new AsymmetricDecoder();
    private final SymmetricDecoder symmetricDecoder = new SymmetricDecoder();
    private volatile long lastSequenceNumber = -1L;
    private final ChannelParameters parameters;
    private final int maxArrayLength;
    private final int maxStringLength;

    public ChunkDecoder(ChannelParameters parameters, int maxArrayLength, int maxStringLength) {
        this.parameters = parameters;
        this.maxArrayLength = maxArrayLength;
        this.maxStringLength = maxStringLength;
    }

    public void decodeAsymmetric(SecureChannel channel, List<ByteBuf> chunkBuffers, Callback callback) {
        ChunkDecoder.decode(this.asymmetricDecoder, channel, chunkBuffers, callback);
    }

    public void decodeSymmetric(SecureChannel channel, List<ByteBuf> chunkBuffers, Callback callback) {
        ChunkDecoder.decode(this.symmetricDecoder, channel, chunkBuffers, callback);
    }

    private static void decode(AbstractDecoder decoder, SecureChannel channel, List<ByteBuf> chunkBuffers, Callback callback) {
        CompositeByteBuf composite = BufferUtil.compositeBuffer();
        try {
            decoder.decode(channel, composite, chunkBuffers, callback);
        }
        catch (MessageAbortedException e) {
            callback.onMessageAborted(e);
            ChunkDecoder.safeReleaseBuffers(composite, chunkBuffers);
        }
        catch (UaException e) {
            callback.onDecodingError(e);
            ChunkDecoder.safeReleaseBuffers(composite, chunkBuffers);
        }
    }

    private static void safeReleaseBuffers(CompositeByteBuf composite, List<ByteBuf> chunkBuffers) {
        if (composite.refCnt() > 0) {
            ReferenceCountUtil.safeRelease((Object)composite);
        }
        chunkBuffers.forEach(b -> {
            if (b.refCnt() > 0) {
                ReferenceCountUtil.safeRelease((Object)b);
            }
        });
    }

    private final class SymmetricDecoder
    extends AbstractDecoder {
        private volatile ChannelSecurity.SecurityKeys securityKeys;
        private volatile Cipher cipher;
        private volatile long cipherId;

        private SymmetricDecoder() {
            this.cipher = null;
            this.cipherId = -1L;
        }

        @Override
        public void readSecurityHeader(SecureChannel channel, ByteBuf chunkBuffer) throws UaException {
            long receivedTokenId = SymmetricSecurityHeader.decode(chunkBuffer).getTokenId();
            ChannelSecurity channelSecurity = channel.getChannelSecurity();
            if (channelSecurity == null) {
                if (receivedTokenId != 0L) {
                    throw new UaException(0x80870000L, "unknown secure channel token: " + receivedTokenId);
                }
            } else {
                long currentTokenId = channelSecurity.getCurrentToken().getTokenId().longValue();
                if (receivedTokenId == currentTokenId) {
                    this.securityKeys = channelSecurity.getCurrentKeys();
                } else {
                    long previousTokenId = channelSecurity.getPreviousToken().map(t -> t.getTokenId().longValue()).orElse(-1L);
                    this.logger.debug("Attempting to use SecurityKeys from previousTokenId={}", (Object)previousTokenId);
                    if (receivedTokenId != previousTokenId) {
                        this.logger.warn("receivedTokenId={} did not match previousTokenId={}", (Object)receivedTokenId, (Object)previousTokenId);
                        throw new UaException(0x80870000L, "unknown secure channel token: " + receivedTokenId);
                    }
                    if (channel.isSymmetricEncryptionEnabled() && channelSecurity.getPreviousKeys().isPresent()) {
                        this.securityKeys = channelSecurity.getPreviousKeys().get();
                    }
                }
                if (this.cipherId != receivedTokenId && channel.isSymmetricEncryptionEnabled()) {
                    this.cipher = this.initCipher(channel);
                    this.cipherId = receivedTokenId;
                }
            }
        }

        @Override
        public Cipher getCipher(SecureChannel channel) {
            assert (this.cipher != null);
            return this.cipher;
        }

        @Override
        public int getCipherTextBlockSize(SecureChannel channel) {
            return channel.getSymmetricBlockSize();
        }

        @Override
        public int getSignatureSize(SecureChannel channel) {
            return channel.getSymmetricSignatureSize();
        }

        @Override
        public void verifyChunk(SecureChannel channel, ByteBuf chunkBuffer) throws UaException {
            SecurityAlgorithm securityAlgorithm = channel.getSecurityPolicy().getSymmetricSignatureAlgorithm();
            byte[] secretKey = channel.getDecryptionKeys(this.securityKeys).getSignatureKey();
            int signatureSize = channel.getSymmetricSignatureSize();
            ByteBuffer chunkNioBuffer = chunkBuffer.nioBuffer(0, chunkBuffer.writerIndex());
            ((Buffer)chunkNioBuffer).position(0);
            ((Buffer)chunkNioBuffer).limit(chunkBuffer.writerIndex() - signatureSize);
            byte[] signature = SignatureUtil.hmac(securityAlgorithm, secretKey, chunkNioBuffer);
            byte[] signatureBytes = new byte[signatureSize];
            ((Buffer)chunkNioBuffer).limit(chunkNioBuffer.position() + signatureSize);
            chunkNioBuffer.get(signatureBytes);
            if (!MessageDigest.isEqual(signature, signatureBytes)) {
                throw new UaException(2148728832L, "could not verify signature");
            }
        }

        @Override
        protected boolean isAsymmetric() {
            return false;
        }

        @Override
        public boolean isEncryptionEnabled(SecureChannel channel) {
            return channel.isSymmetricEncryptionEnabled();
        }

        @Override
        public boolean isSigningEnabled(SecureChannel channel) {
            return channel.isSymmetricSigningEnabled();
        }

        private Cipher initCipher(SecureChannel channel) throws UaException {
            try {
                String transformation = channel.getSecurityPolicy().getSymmetricEncryptionAlgorithm().getTransformation();
                ChannelSecurity.SecretKeys decryptionKeys = channel.getDecryptionKeys(this.securityKeys);
                SecretKeySpec keySpec = new SecretKeySpec(decryptionKeys.getEncryptionKey(), "AES");
                IvParameterSpec ivSpec = new IvParameterSpec(decryptionKeys.getInitializationVector());
                Cipher cipher = Cipher.getInstance(transformation);
                cipher.init(2, (Key)keySpec, ivSpec);
                return cipher;
            }
            catch (GeneralSecurityException e) {
                throw new UaException(0x80020000L, (Throwable)e);
            }
        }
    }

    private final class AsymmetricDecoder
    extends AbstractDecoder {
        private AsymmetricDecoder() {
        }

        @Override
        public void readSecurityHeader(SecureChannel channel, ByteBuf chunkBuffer) {
            AsymmetricSecurityHeader.decode(chunkBuffer, ChunkDecoder.this.maxArrayLength, ChunkDecoder.this.maxStringLength);
        }

        @Override
        public Cipher getCipher(SecureChannel channel) throws UaException {
            try {
                String transformation = channel.getSecurityPolicy().getAsymmetricEncryptionAlgorithm().getTransformation();
                Cipher cipher = Cipher.getInstance(transformation);
                cipher.init(2, channel.getKeyPair().getPrivate());
                return cipher;
            }
            catch (GeneralSecurityException e) {
                throw new UaException(0x80020000L, (Throwable)e);
            }
        }

        @Override
        public int getCipherTextBlockSize(SecureChannel channel) {
            return channel.getLocalAsymmetricCipherTextBlockSize();
        }

        @Override
        public int getSignatureSize(SecureChannel channel) {
            return channel.getRemoteAsymmetricSignatureSize();
        }

        @Override
        public void verifyChunk(SecureChannel channel, ByteBuf chunkBuffer) throws UaException {
            String transformation = channel.getSecurityPolicy().getAsymmetricSignatureAlgorithm().getTransformation();
            int signatureSize = channel.getRemoteAsymmetricSignatureSize();
            ByteBuffer chunkNioBuffer = chunkBuffer.nioBuffer(0, chunkBuffer.writerIndex());
            ((Buffer)chunkNioBuffer).position(0);
            ((Buffer)chunkNioBuffer).limit(chunkBuffer.writerIndex() - signatureSize);
            try {
                Signature signature = Signature.getInstance(transformation);
                signature.initVerify(channel.getRemoteCertificate().getPublicKey());
                signature.update(chunkNioBuffer);
                byte[] signatureBytes = new byte[signatureSize];
                ((Buffer)chunkNioBuffer).limit(chunkNioBuffer.position() + signatureSize);
                chunkNioBuffer.get(signatureBytes);
                if (!signature.verify(signatureBytes)) {
                    throw new UaException(2148728832L, "could not verify signature");
                }
            }
            catch (NoSuchAlgorithmException e) {
                throw new UaException(0x80020000L, (Throwable)e);
            }
            catch (SignatureException e) {
                throw new UaException(0x80580000L, (Throwable)e);
            }
            catch (InvalidKeyException e) {
                throw new UaException(2148663296L, (Throwable)e);
            }
        }

        @Override
        protected boolean isAsymmetric() {
            return true;
        }

        @Override
        public boolean isEncryptionEnabled(SecureChannel channel) {
            return channel.isAsymmetricEncryptionEnabled();
        }

        @Override
        public boolean isSigningEnabled(SecureChannel channel) {
            return channel.isAsymmetricEncryptionEnabled();
        }
    }

    private abstract class AbstractDecoder {
        protected final Logger logger = LoggerFactory.getLogger(this.getClass());

        private AbstractDecoder() {
        }

        void decode(SecureChannel channel, CompositeByteBuf composite, List<ByteBuf> chunkBuffers, Callback callback) throws UaException {
            int signatureSize = this.getSignatureSize(channel);
            int cipherTextBlockSize = this.getCipherTextBlockSize(channel);
            boolean encrypted = this.isEncryptionEnabled(channel);
            boolean signed = this.isSigningEnabled(channel);
            long requestId = -1L;
            for (ByteBuf chunkBuffer : chunkBuffers) {
                char chunkType = (char)chunkBuffer.getByte(3);
                chunkBuffer.skipBytes(12);
                this.readSecurityHeader(channel, chunkBuffer);
                if (encrypted) {
                    this.decryptChunk(channel, chunkBuffer);
                }
                int encryptedStart = chunkBuffer.readerIndex();
                chunkBuffer.readerIndex(0);
                if (signed) {
                    this.verifyChunk(channel, chunkBuffer);
                }
                int paddingSize = encrypted ? this.getPaddingSize(cipherTextBlockSize, signatureSize, chunkBuffer) : 0;
                int bodyEnd = chunkBuffer.readableBytes() - signatureSize - paddingSize;
                chunkBuffer.readerIndex(encryptedStart);
                SequenceHeader sequenceHeader = SequenceHeader.decode(chunkBuffer);
                long sequenceNumber = sequenceHeader.getSequenceNumber();
                requestId = sequenceHeader.getRequestId();
                if (ChunkDecoder.this.lastSequenceNumber == -1L) {
                    ChunkDecoder.this.lastSequenceNumber = sequenceNumber;
                } else {
                    if (ChunkDecoder.this.lastSequenceNumber + 1L != sequenceNumber) {
                        String message = String.format("expected sequence number %s but received %s", ChunkDecoder.this.lastSequenceNumber + 1L, sequenceNumber);
                        throw new UaException(0x80880000L, message);
                    }
                    ChunkDecoder.this.lastSequenceNumber = sequenceNumber;
                }
                ByteBuf bodyBuffer = chunkBuffer.readSlice(bodyEnd - chunkBuffer.readerIndex());
                if (chunkType == 'A') {
                    ErrorMessage errorMessage = ErrorMessage.decode(bodyBuffer);
                    throw new MessageAbortedException(errorMessage.getError(), errorMessage.getReason(), requestId);
                }
                composite.addComponent(bodyBuffer);
                composite.writerIndex(composite.writerIndex() + bodyBuffer.readableBytes());
            }
            if (ChunkDecoder.this.parameters.getLocalMaxMessageSize() > 0 && composite.readableBytes() > ChunkDecoder.this.parameters.getLocalMaxMessageSize()) {
                String errorMessage = String.format("message size exceeds configured limit: %s > %s", composite.readableBytes(), ChunkDecoder.this.parameters.getLocalMaxMessageSize());
                throw new UaException(0x80800000L, errorMessage);
            }
            callback.onMessageDecoded((ByteBuf)composite, requestId);
        }

        private void decryptChunk(SecureChannel channel, ByteBuf chunkBuffer) throws UaException {
            int cipherTextBlockSize = this.getCipherTextBlockSize(channel);
            int blockCount = chunkBuffer.readableBytes() / cipherTextBlockSize;
            int plainTextBufferSize = cipherTextBlockSize * blockCount;
            ByteBuf plainTextBuffer = BufferUtil.pooledBuffer(plainTextBufferSize);
            ByteBuffer plainTextNioBuffer = plainTextBuffer.writerIndex(plainTextBufferSize).nioBuffer();
            ByteBuffer chunkNioBuffer = chunkBuffer.nioBuffer();
            try {
                Cipher cipher = this.getCipher(channel);
                assert (chunkBuffer.readableBytes() % cipherTextBlockSize == 0);
                if (this.isAsymmetric()) {
                    for (int blockNumber = 0; blockNumber < blockCount; ++blockNumber) {
                        ((Buffer)chunkNioBuffer).limit(chunkNioBuffer.position() + cipherTextBlockSize);
                        cipher.doFinal(chunkNioBuffer, plainTextNioBuffer);
                    }
                } else {
                    cipher.doFinal(chunkNioBuffer, plainTextNioBuffer);
                }
                ((Buffer)plainTextNioBuffer).flip();
                chunkBuffer.writerIndex(chunkBuffer.readerIndex());
                chunkBuffer.writeBytes(plainTextNioBuffer);
            }
            catch (GeneralSecurityException e) {
                throw new UaException(2148728832L, (Throwable)e);
            }
            finally {
                plainTextBuffer.release();
            }
        }

        private int getPaddingSize(int cipherTextBlockSize, int signatureSize, ByteBuf buffer) {
            int lastPaddingByteOffset = buffer.readableBytes() - signatureSize - 1;
            return cipherTextBlockSize <= 256 ? buffer.getUnsignedByte(lastPaddingByteOffset) + 1 : buffer.getUnsignedShortLE(lastPaddingByteOffset - 1) + 2;
        }

        protected abstract void readSecurityHeader(SecureChannel var1, ByteBuf var2) throws UaException;

        protected abstract Cipher getCipher(SecureChannel var1) throws UaException;

        protected abstract int getCipherTextBlockSize(SecureChannel var1);

        protected abstract int getSignatureSize(SecureChannel var1);

        protected abstract void verifyChunk(SecureChannel var1, ByteBuf var2) throws UaException;

        protected abstract boolean isAsymmetric();

        protected abstract boolean isEncryptionEnabled(SecureChannel var1);

        protected abstract boolean isSigningEnabled(SecureChannel var1);
    }

    public static interface Callback {
        public void onDecodingError(UaException var1);

        public void onMessageAborted(MessageAbortedException var1);

        public void onMessageDecoded(ByteBuf var1, long var2);
    }
}

