/*
 * Decompiled with CFR 0.152.
 */
package org.eclipse.milo.opcua.stack.core.util;

import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import java.security.InvalidKeyException;
import java.security.PublicKey;
import java.security.SignatureException;
import java.security.cert.CertPathBuilder;
import java.security.cert.CertPathChecker;
import java.security.cert.CertSelector;
import java.security.cert.CertStore;
import java.security.cert.CertificateExpiredException;
import java.security.cert.CertificateNotYetValidException;
import java.security.cert.CertificateParsingException;
import java.security.cert.CollectionCertStoreParameters;
import java.security.cert.PKIXBuilderParameters;
import java.security.cert.PKIXCertPathBuilderResult;
import java.security.cert.PKIXRevocationChecker;
import java.security.cert.TrustAnchor;
import java.security.cert.X509CRL;
import java.security.cert.X509CertSelector;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.function.Predicate;
import org.eclipse.milo.opcua.stack.core.UaException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CertificateValidationUtil {
    private static final Logger LOGGER = LoggerFactory.getLogger(CertificateValidationUtil.class);
    private static final String KEY_USAGE_OID = "2.5.29.15";
    private static final int SUBJECT_ALT_NAME_URI = 6;
    private static final int SUBJECT_ALT_NAME_DNS_NAME = 2;
    private static final int SUBJECT_ALT_NAME_IP_ADDRESS = 7;

    public static void verifyTrustChain(List<X509Certificate> certificateChain, Set<X509Certificate> trustedCertificates, Set<X509Certificate> issuerCertificates) throws UaException {
        CertificateValidationUtil.verifyTrustChain(certificateChain, trustedCertificates, Collections.emptySet(), issuerCertificates, Collections.emptySet());
    }

    public static void verifyTrustChain(List<X509Certificate> certificateChain, Collection<X509Certificate> trustedCertificates, Collection<X509CRL> trustedCrls, Collection<X509Certificate> issuerCertificates, Collection<X509CRL> issuerCrls) throws UaException {
        Preconditions.checkArgument((!certificateChain.isEmpty() ? 1 : 0) != 0, (Object)"certificateChain must not be empty");
        if (LOGGER.isTraceEnabled()) {
            LOGGER.trace("certificateChain: {}", certificateChain);
            LOGGER.trace("trustedCertificates: {}", trustedCertificates);
            LOGGER.trace("trustedCrls: {}", trustedCrls);
            LOGGER.trace("issuerCertificates: {}", issuerCertificates);
            LOGGER.trace("issuerCrls: {}", issuerCrls);
        }
        X509Certificate certificate = certificateChain.get(0);
        try {
            HashSet<TrustAnchor> trustAnchors = new HashSet<TrustAnchor>();
            for (X509Certificate c : trustedCertificates) {
                if (!CertificateValidationUtil.certificateIsCa(c) || !CertificateValidationUtil.certificateIsSelfSigned(c)) continue;
                trustAnchors.add(new TrustAnchor(c, null));
            }
            for (X509Certificate c : issuerCertificates) {
                if (!CertificateValidationUtil.certificateIsCa(c) || !CertificateValidationUtil.certificateIsSelfSigned(c)) continue;
                trustAnchors.add(new TrustAnchor(c, null));
            }
            X509CertSelector selector = new X509CertSelector();
            selector.setCertificate(certificate);
            PKIXBuilderParameters params = new PKIXBuilderParameters(trustAnchors, (CertSelector)selector);
            ArrayList collection = Lists.newArrayList();
            collection.addAll(certificateChain.subList(1, certificateChain.size()));
            for (X509Certificate c : trustedCertificates) {
                if (!CertificateValidationUtil.certificateIsCa(c) || CertificateValidationUtil.certificateIsSelfSigned(c)) continue;
                collection.add(c);
            }
            for (X509Certificate c : issuerCertificates) {
                if (!CertificateValidationUtil.certificateIsCa(c) || CertificateValidationUtil.certificateIsSelfSigned(c)) continue;
                collection.add(c);
            }
            collection.addAll(trustedCrls);
            collection.addAll(issuerCrls);
            if (collection.size() > 0) {
                CertStore certStore = CertStore.getInstance("Collection", new CollectionCertStoreParameters(collection));
                params.addCertStore(certStore);
            }
            params.setRevocationEnabled(!trustedCrls.isEmpty() || !issuerCrls.isEmpty());
            CertPathBuilder builder = CertPathBuilder.getInstance("PKIX");
            CertPathChecker revocationChecker = builder.getRevocationChecker();
            if (revocationChecker instanceof PKIXRevocationChecker) {
                ((PKIXRevocationChecker)revocationChecker).setOptions(Sets.newHashSet((Object[])new PKIXRevocationChecker.Option[]{PKIXRevocationChecker.Option.NO_FALLBACK, PKIXRevocationChecker.Option.PREFER_CRLS, PKIXRevocationChecker.Option.SOFT_FAIL}));
            }
            PKIXCertPathBuilderResult result = (PKIXCertPathBuilderResult)builder.build(params);
            ArrayList<X509Certificate> certificatePath = new ArrayList<X509Certificate>();
            result.getCertPath().getCertificates().stream().map(X509Certificate.class::cast).forEach(certificatePath::add);
            certificatePath.add(result.getTrustAnchor().getTrustedCert());
            LOGGER.debug("certificate path: {}", certificatePath);
            if (certificatePath.stream().noneMatch(trustedCertificates::contains)) {
                throw new UaException(2148728832L, "certificate path did not contain a trusted certificate");
            }
        }
        catch (UaException e) {
            throw e;
        }
        catch (Throwable t) {
            LOGGER.debug("certificate path validation failed: {}", (Object)t.getMessage());
            throw new UaException(2148728832L, "certificate path validation failed");
        }
    }

    static boolean certificateIsCa(X509Certificate certificate) {
        boolean[] keyUsage = certificate.getKeyUsage();
        int basicConstraints = certificate.getBasicConstraints();
        if (keyUsage == null) {
            return basicConstraints >= 0;
        }
        if (keyUsage[5] && basicConstraints == -1) {
            LOGGER.warn("'{}' violates RFC 5280: KeyUsage keyCertSign bit set without BasicConstraint cA bit set", (Object)certificate.getSubjectX500Principal().getName());
        }
        return keyUsage[5] || basicConstraints >= 0;
    }

    private static boolean certificateIsSelfSigned(X509Certificate cert) throws UaException {
        try {
            PublicKey key = cert.getPublicKey();
            cert.verify(key);
            return Objects.equals(cert.getSubjectX500Principal(), cert.getIssuerX500Principal());
        }
        catch (InvalidKeyException | SignatureException sigEx) {
            return false;
        }
        catch (Exception e) {
            throw new UaException(2148663296L, (Throwable)e);
        }
    }

    public static void validateCertificateValidity(X509Certificate certificate) throws UaException {
        try {
            certificate.checkValidity();
        }
        catch (CertificateExpiredException e) {
            throw new UaException(2148794368L, String.format("certificate is expired: %s - %s", certificate.getNotBefore(), certificate.getNotAfter()));
        }
        catch (CertificateNotYetValidException e) {
            throw new UaException(2148794368L, String.format("certificate not yet valid: %s - %s", certificate.getNotBefore(), certificate.getNotAfter()));
        }
    }

    public static void validateHostnameOrIpAddress(X509Certificate certificate, String ... hostnames) throws UaException {
        boolean dnsNameMatch = Arrays.stream(hostnames).anyMatch(n -> {
            try {
                return CertificateValidationUtil.validateSubjectAltNameField(certificate, 2, n::equals);
            }
            catch (Throwable t) {
                return false;
            }
        });
        boolean ipAddressMatch = Arrays.stream(hostnames).anyMatch(n -> {
            try {
                return CertificateValidationUtil.validateSubjectAltNameField(certificate, 7, n::equals);
            }
            catch (Throwable t) {
                return false;
            }
        });
        if (!dnsNameMatch && !ipAddressMatch) {
            throw new UaException(2148925440L);
        }
    }

    public static void validateApplicationUri(X509Certificate certificate, String applicationUri) throws UaException {
        if (!CertificateValidationUtil.validateSubjectAltNameField(certificate, 6, applicationUri::equals)) {
            throw new UaException(2148990976L);
        }
    }

    public static void validateApplicationCertificateUsage(X509Certificate certificate) throws UaException {
        Set<String> criticalExtensions = certificate.getCriticalExtensionOIDs();
        if (criticalExtensions == null) {
            criticalExtensions = new HashSet<String>();
        }
        if (criticalExtensions.contains(KEY_USAGE_OID)) {
            boolean[] keyUsage = certificate.getKeyUsage();
            boolean digitalSignature = keyUsage[0];
            boolean nonRepudiation = keyUsage[1];
            boolean keyEncipherment = keyUsage[2];
            boolean dataEncipherment = keyUsage[3];
            if (!digitalSignature) {
                throw new UaException(0x80180000L, "required KeyUsage 'digitalSignature' not found");
            }
            if (!nonRepudiation) {
                throw new UaException(0x80180000L, "required KeyUsage 'nonRepudiation' not found");
            }
            if (!keyEncipherment) {
                throw new UaException(0x80180000L, "required KeyUsage 'keyEncipherment' not found");
            }
            if (!dataEncipherment) {
                throw new UaException(0x80180000L, "required KeyUsage 'dataEncipherment' not found");
            }
        }
    }

    public static boolean validateSubjectAltNameField(X509Certificate certificate, int field, Predicate<Object> fieldValidator) throws UaException {
        try {
            Collection<List<?>> subjectAltNames = certificate.getSubjectAlternativeNames();
            if (subjectAltNames == null) {
                subjectAltNames = Collections.emptyList();
            }
            for (List<?> idAndValue : subjectAltNames) {
                if (idAndValue == null || idAndValue.size() != 2 || !idAndValue.get(0).equals(field) || !fieldValidator.test(idAndValue.get(1))) continue;
                return true;
            }
            return false;
        }
        catch (CertificateParsingException e) {
            throw new UaException(2148663296L, (Throwable)e);
        }
    }

    public static String getSubjectAltNameUri(X509Certificate certificate) throws UaException {
        try {
            Collection<List<?>> subjectAltNames = certificate.getSubjectAlternativeNames();
            if (subjectAltNames == null) {
                subjectAltNames = Collections.emptyList();
            }
            for (List<?> idAndValue : subjectAltNames) {
                if (idAndValue == null || idAndValue.size() != 2 || !idAndValue.get(0).equals(6)) continue;
                Object uri = idAndValue.get(1);
                return uri != null ? uri.toString() : null;
            }
            return null;
        }
        catch (CertificateParsingException e) {
            throw new UaException(2148663296L, (Throwable)e);
        }
    }
}

