/*
 * Decompiled with CFR 0.152.
 */
package org.apache.camel.component.kubernetes.cluster.lock;

import io.fabric8.kubernetes.api.model.ConfigMap;
import io.fabric8.kubernetes.api.model.PodList;
import io.fabric8.kubernetes.client.KubernetesClient;
import io.fabric8.kubernetes.client.dsl.FilterWatchListDeletable;
import io.fabric8.kubernetes.client.dsl.NonNamespaceOperation;
import io.fabric8.kubernetes.client.dsl.Replaceable;
import io.fabric8.kubernetes.client.dsl.Resource;
import java.math.BigDecimal;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import org.apache.camel.CamelContext;
import org.apache.camel.Service;
import org.apache.camel.component.kubernetes.cluster.lock.ConfigMapLockUtils;
import org.apache.camel.component.kubernetes.cluster.lock.KubernetesClusterEventHandler;
import org.apache.camel.component.kubernetes.cluster.lock.KubernetesLockConfiguration;
import org.apache.camel.component.kubernetes.cluster.lock.LeaderInfo;
import org.apache.camel.component.kubernetes.cluster.lock.TimedLeaderNotifier;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class KubernetesLeadershipController
implements Service {
    private static final Logger LOG = LoggerFactory.getLogger(KubernetesLeadershipController.class);
    private CamelContext camelContext;
    private KubernetesClient kubernetesClient;
    private KubernetesLockConfiguration lockConfiguration;
    private KubernetesClusterEventHandler eventHandler;
    private State currentState = State.NOT_LEADER;
    private ScheduledExecutorService serializedExecutor;
    private TimedLeaderNotifier leaderNotifier;
    private volatile LeaderInfo latestLeaderInfo;
    private volatile ConfigMap latestConfigMap;
    private volatile Set<String> latestMembers;

    public KubernetesLeadershipController(CamelContext camelContext, KubernetesClient kubernetesClient, KubernetesLockConfiguration lockConfiguration, KubernetesClusterEventHandler eventHandler) {
        this.camelContext = camelContext;
        this.kubernetesClient = kubernetesClient;
        this.lockConfiguration = lockConfiguration;
        this.eventHandler = eventHandler;
    }

    public void start() {
        if (this.serializedExecutor == null) {
            LOG.debug("{} Starting leadership controller...", (Object)this.logPrefix());
            this.serializedExecutor = this.camelContext.getExecutorServiceManager().newSingleThreadScheduledExecutor((Object)this, "CamelKubernetesLeadershipController");
            this.leaderNotifier = new TimedLeaderNotifier(this.camelContext, this.eventHandler);
            this.leaderNotifier.start();
            this.serializedExecutor.execute(this::refreshStatus);
        }
    }

    public void stop() {
        LOG.debug("{} Stopping leadership controller...", (Object)this.logPrefix());
        if (this.serializedExecutor != null) {
            this.serializedExecutor.shutdownNow();
        }
        this.serializedExecutor = null;
        if (this.leaderNotifier != null) {
            this.leaderNotifier.stop();
        }
        this.leaderNotifier = null;
    }

    private void refreshStatus() {
        switch (this.currentState) {
            case NOT_LEADER: {
                this.refreshStatusNotLeader();
                break;
            }
            case BECOMING_LEADER: {
                this.refreshStatusBecomingLeader();
                break;
            }
            case LEADER: {
                this.refreshStatusLeader();
                break;
            }
            default: {
                throw new RuntimeException("Unsupported state " + (Object)((Object)this.currentState));
            }
        }
    }

    private void refreshStatusNotLeader() {
        LOG.debug("{} Pod is not leader, pulling new data from the cluster", (Object)this.logPrefix());
        boolean pulled = this.lookupNewLeaderInfo();
        if (!pulled) {
            this.rescheduleAfterDelay();
            return;
        }
        if (this.latestLeaderInfo.hasEmptyLeader()) {
            LOG.info("{} The cluster has no leaders. Trying to acquire the leadership...", (Object)this.logPrefix());
            boolean acquired = this.tryAcquireLeadership();
            if (acquired) {
                LOG.info("{} Leadership acquired by current pod with immediate effect", (Object)this.logPrefix());
                this.currentState = State.LEADER;
                this.serializedExecutor.execute(this::refreshStatus);
                return;
            }
            LOG.info("{} Unable to acquire the leadership, it may have been acquired by another pod", (Object)this.logPrefix());
        } else if (!this.latestLeaderInfo.hasValidLeader()) {
            LOG.info("{} Leadership has been lost by old owner. Trying to acquire the leadership...", (Object)this.logPrefix());
            boolean acquired = this.tryAcquireLeadership();
            if (acquired) {
                LOG.info("{} Leadership acquired by current pod", (Object)this.logPrefix());
                this.currentState = State.BECOMING_LEADER;
                this.serializedExecutor.execute(this::refreshStatus);
                return;
            }
            LOG.info("{} Unable to acquire the leadership, it may have been acquired by another pod", (Object)this.logPrefix());
        } else if (this.latestLeaderInfo.isValidLeader(this.lockConfiguration.getPodName())) {
            LOG.info("{} Leadership is already owned by current pod", (Object)this.logPrefix());
            this.currentState = State.BECOMING_LEADER;
            this.serializedExecutor.execute(this::refreshStatus);
            return;
        }
        this.leaderNotifier.refreshLeadership(Optional.ofNullable(this.latestLeaderInfo.getLeader()), System.currentTimeMillis(), this.lockConfiguration.getLeaseDurationMillis(), this.latestLeaderInfo.getMembers());
        this.rescheduleAfterDelay();
    }

    private void refreshStatusBecomingLeader() {
        long delay = this.lockConfiguration.getLeaseDurationMillis();
        LOG.info("{} Current pod owns the leadership, but it will be effective in {} seconds...", (Object)this.logPrefix(), (Object)new BigDecimal(delay).divide(BigDecimal.valueOf(1000L), 2, 4));
        try {
            Thread.sleep(delay);
        }
        catch (InterruptedException e) {
            LOG.warn("Thread interrupted", (Throwable)e);
        }
        LOG.info("{} Current pod is becoming the new leader now...", (Object)this.logPrefix());
        this.currentState = State.LEADER;
        this.serializedExecutor.execute(this::refreshStatus);
    }

    private void refreshStatusLeader() {
        LOG.debug("{} Pod should be the leader, pulling new data from the cluster", (Object)this.logPrefix());
        long timeBeforePulling = System.currentTimeMillis();
        boolean pulled = this.lookupNewLeaderInfo();
        if (!pulled) {
            this.rescheduleAfterDelay();
            return;
        }
        if (this.latestLeaderInfo.isValidLeader(this.lockConfiguration.getPodName())) {
            LOG.debug("{} Current Pod is still the leader", (Object)this.logPrefix());
            this.leaderNotifier.refreshLeadership(Optional.of(this.lockConfiguration.getPodName()), timeBeforePulling, this.lockConfiguration.getRenewDeadlineMillis(), this.latestLeaderInfo.getMembers());
            this.rescheduleAfterDelay();
            return;
        }
        LOG.debug("{} Current Pod has lost the leadership", (Object)this.logPrefix());
        this.currentState = State.NOT_LEADER;
        this.leaderNotifier.refreshLeadership(Optional.empty(), System.currentTimeMillis(), this.lockConfiguration.getLeaseDurationMillis(), this.latestLeaderInfo.getMembers());
        this.serializedExecutor.execute(this::refreshStatus);
    }

    private void rescheduleAfterDelay() {
        this.serializedExecutor.schedule(this::refreshStatus, this.jitter(this.lockConfiguration.getRetryPeriodMillis(), this.lockConfiguration.getJitterFactor()), TimeUnit.MILLISECONDS);
    }

    private boolean lookupNewLeaderInfo() {
        Set<String> members;
        ConfigMap configMap;
        LOG.debug("{} Looking up leadership information...", (Object)this.logPrefix());
        try {
            configMap = this.pullConfigMap();
        }
        catch (Throwable e) {
            LOG.warn(this.logPrefix() + " Unable to retrieve the current ConfigMap " + this.lockConfiguration.getConfigMapName() + " from Kubernetes");
            LOG.debug(this.logPrefix() + " Exception thrown during ConfigMap lookup", e);
            return false;
        }
        try {
            members = Objects.requireNonNull(this.pullClusterMembers(), "Retrieved a null set of members");
        }
        catch (Throwable e) {
            LOG.warn(this.logPrefix() + " Unable to retrieve the list of cluster members from Kubernetes");
            LOG.debug(this.logPrefix() + " Exception thrown during Pod list lookup", e);
            return false;
        }
        this.updateLatestLeaderInfo(configMap, members);
        return true;
    }

    private boolean tryAcquireLeadership() {
        boolean canAcquire;
        LOG.debug("{} Trying to acquire the leadership...", (Object)this.logPrefix());
        ConfigMap configMap = this.latestConfigMap;
        Set<String> members = this.latestMembers;
        LeaderInfo latestLeaderInfo = this.latestLeaderInfo;
        if (latestLeaderInfo == null || members == null) {
            LOG.warn(this.logPrefix() + " Unexpected condition. Latest leader info or list of members is empty.");
            return false;
        }
        if (!members.contains(this.lockConfiguration.getPodName())) {
            LOG.warn(this.logPrefix() + " The list of cluster members " + latestLeaderInfo.getMembers() + " does not contain the current Pod. Cannot acquire leadership.");
            return false;
        }
        LeaderInfo newLeaderInfo = new LeaderInfo(this.lockConfiguration.getGroupName(), this.lockConfiguration.getPodName(), new Date(), members);
        if (configMap == null) {
            LOG.debug("{} Lock configmap is not present in the Kubernetes namespace. A new ConfigMap will be created", (Object)this.logPrefix());
            ConfigMap newConfigMap = ConfigMapLockUtils.createNewConfigMap(this.lockConfiguration.getConfigMapName(), newLeaderInfo);
            try {
                ((NonNamespaceOperation)this.kubernetesClient.configMaps().inNamespace(this.lockConfiguration.getKubernetesResourcesNamespaceOrDefault(this.kubernetesClient))).create((Object[])new ConfigMap[]{newConfigMap});
                LOG.debug("{} ConfigMap {} successfully created", (Object)this.logPrefix(), (Object)this.lockConfiguration.getConfigMapName());
                this.updateLatestLeaderInfo(newConfigMap, members);
                return true;
            }
            catch (Exception ex) {
                LOG.warn(this.logPrefix() + " Unable to create the ConfigMap, it may have been created by other cluster members concurrently. If the problem persists, check if the service account has the right permissions to create it");
                LOG.debug(this.logPrefix() + " Exception while trying to create the ConfigMap", (Throwable)ex);
                return false;
            }
        }
        LOG.debug("{} Lock configmap already present in the Kubernetes namespace. Checking...", (Object)this.logPrefix());
        LeaderInfo leaderInfo = ConfigMapLockUtils.getLeaderInfo(configMap, members, this.lockConfiguration.getGroupName());
        boolean bl = canAcquire = !leaderInfo.hasValidLeader();
        if (canAcquire) {
            try {
                ConfigMap updatedConfigMap = ConfigMapLockUtils.getConfigMapWithNewLeader(configMap, newLeaderInfo);
                ((Replaceable)((Resource)((NonNamespaceOperation)this.kubernetesClient.configMaps().inNamespace(this.lockConfiguration.getKubernetesResourcesNamespaceOrDefault(this.kubernetesClient))).withName(this.lockConfiguration.getConfigMapName())).lockResourceVersion(configMap.getMetadata().getResourceVersion())).replace((Object)updatedConfigMap);
                LOG.debug("{} ConfigMap {} successfully updated", (Object)this.logPrefix(), (Object)this.lockConfiguration.getConfigMapName());
                this.updateLatestLeaderInfo(updatedConfigMap, members);
                return true;
            }
            catch (Exception ex) {
                LOG.warn(this.logPrefix() + " Unable to update the lock ConfigMap to set leadership information");
                LOG.debug(this.logPrefix() + " Error received during configmap lock replace", (Throwable)ex);
                return false;
            }
        }
        LOG.debug("{} Another Pod ({}) is the current leader and it is still active", (Object)this.logPrefix(), (Object)this.latestLeaderInfo.getLeader());
        return false;
    }

    private void updateLatestLeaderInfo(ConfigMap configMap, Set<String> members) {
        LOG.debug("{} Updating internal status about the current leader", (Object)this.logPrefix());
        this.latestConfigMap = configMap;
        this.latestMembers = members;
        this.latestLeaderInfo = ConfigMapLockUtils.getLeaderInfo(configMap, members, this.lockConfiguration.getGroupName());
        LOG.debug("{} Current leader info: {}", (Object)this.logPrefix(), (Object)this.latestLeaderInfo);
    }

    private ConfigMap pullConfigMap() {
        return (ConfigMap)((Resource)((NonNamespaceOperation)this.kubernetesClient.configMaps().inNamespace(this.lockConfiguration.getKubernetesResourcesNamespaceOrDefault(this.kubernetesClient))).withName(this.lockConfiguration.getConfigMapName())).get();
    }

    private Set<String> pullClusterMembers() {
        List pods = ((PodList)((FilterWatchListDeletable)((NonNamespaceOperation)this.kubernetesClient.pods().inNamespace(this.lockConfiguration.getKubernetesResourcesNamespaceOrDefault(this.kubernetesClient))).withLabels(this.lockConfiguration.getClusterLabels())).list()).getItems();
        return pods.stream().map(pod -> pod.getMetadata().getName()).collect(Collectors.toSet());
    }

    private long jitter(long num, double factor) {
        return (long)((double)num * (1.0 + Math.random() * (factor - 1.0)));
    }

    private String logPrefix() {
        return "Pod[" + this.lockConfiguration.getPodName() + "]";
    }

    private static enum State {
        NOT_LEADER,
        BECOMING_LEADER,
        LEADER;

    }
}

