/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.namenode;

import com.google.common.base.Preconditions;
import com.google.common.collect.ComparisonChain;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.EnumSet;
import java.util.Iterator;
import java.util.List;
import java.util.TreeSet;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hdfs.server.namenode.EditLogInputStream;
import org.apache.hadoop.hdfs.server.namenode.FSImageStorageInspector;
import org.apache.hadoop.hdfs.server.namenode.FSImageTransactionalStorageInspector;
import org.apache.hadoop.hdfs.server.namenode.FileJournalManager;
import org.apache.hadoop.hdfs.server.namenode.LogsPurgeable;
import org.apache.hadoop.hdfs.server.namenode.NNStorage;
import org.apache.hadoop.hdfs.util.MD5FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class NNStorageRetentionManager {
    private final int numCheckpointsToRetain;
    private final long numExtraEditsToRetain;
    private final int maxExtraEditsSegmentsToRetain;
    private static final Logger LOG = LoggerFactory.getLogger(NNStorageRetentionManager.class);
    private final NNStorage storage;
    private final StoragePurger purger;
    private final LogsPurgeable purgeableLogs;

    public NNStorageRetentionManager(Configuration conf, NNStorage storage, LogsPurgeable purgeableLogs, StoragePurger purger) {
        this.numCheckpointsToRetain = conf.getInt("dfs.namenode.num.checkpoints.retained", 2);
        this.numExtraEditsToRetain = conf.getLong("dfs.namenode.num.extra.edits.retained", 1000000L);
        this.maxExtraEditsSegmentsToRetain = conf.getInt("dfs.namenode.max.extra.edits.segments.retained", 10000);
        Preconditions.checkArgument((this.numCheckpointsToRetain > 0 ? 1 : 0) != 0, (Object)"Must retain at least one checkpoint");
        Preconditions.checkArgument((this.numExtraEditsToRetain >= 0L ? 1 : 0) != 0, (Object)"dfs.namenode.num.extra.edits.retained must not be negative");
        this.storage = storage;
        this.purgeableLogs = purgeableLogs;
        this.purger = purger;
    }

    public NNStorageRetentionManager(Configuration conf, NNStorage storage, LogsPurgeable purgeableLogs) {
        this(conf, storage, purgeableLogs, new DeletionStoragePurger());
    }

    void purgeCheckpoints(NNStorage.NameNodeFile nnf) throws IOException {
        this.purgeCheckpoinsAfter(nnf, -1L);
    }

    void purgeCheckpoinsAfter(NNStorage.NameNodeFile nnf, long fromTxId) throws IOException {
        FSImageTransactionalStorageInspector inspector = new FSImageTransactionalStorageInspector(EnumSet.of(nnf));
        this.storage.inspectStorageDirs(inspector);
        for (FSImageStorageInspector.FSImageFile image : inspector.getFoundImages()) {
            if (image.getCheckpointTxId() <= fromTxId) continue;
            this.purger.purgeImage(image);
        }
    }

    void purgeOldStorage(NNStorage.NameNodeFile nnf) throws IOException {
        FSImageTransactionalStorageInspector inspector = new FSImageTransactionalStorageInspector(EnumSet.of(nnf));
        this.storage.inspectStorageDirs(inspector);
        long minImageTxId = this.getImageTxIdToRetain(inspector);
        this.purgeCheckpointsOlderThan(inspector, minImageTxId);
        if (nnf == NNStorage.NameNodeFile.IMAGE_ROLLBACK) {
            return;
        }
        long minimumRequiredTxId = minImageTxId + 1L;
        long purgeLogsFrom = Math.max(0L, minimumRequiredTxId - this.numExtraEditsToRetain);
        ArrayList<EditLogInputStream> editLogs = new ArrayList<EditLogInputStream>();
        this.purgeableLogs.selectInputStreams(editLogs, purgeLogsFrom, false, false);
        Collections.sort(editLogs, new Comparator<EditLogInputStream>(){

            @Override
            public int compare(EditLogInputStream a, EditLogInputStream b) {
                return ComparisonChain.start().compare(a.getFirstTxId(), b.getFirstTxId()).compare(a.getLastTxId(), b.getLastTxId()).result();
            }
        });
        while (editLogs.size() > 0 && editLogs.get(editLogs.size() - 1).getFirstTxId() >= minimumRequiredTxId) {
            editLogs.remove(editLogs.size() - 1);
        }
        while (editLogs.size() > this.maxExtraEditsSegmentsToRetain) {
            purgeLogsFrom = editLogs.get(0).getLastTxId() + 1L;
            editLogs.remove(0);
        }
        if (purgeLogsFrom > minimumRequiredTxId) {
            throw new AssertionError((Object)("Should not purge more edits than required to restore: " + purgeLogsFrom + " should be <= " + minimumRequiredTxId));
        }
        this.purgeableLogs.purgeLogsOlderThan(purgeLogsFrom);
    }

    private void purgeCheckpointsOlderThan(FSImageTransactionalStorageInspector inspector, long minTxId) {
        for (FSImageStorageInspector.FSImageFile image : inspector.getFoundImages()) {
            if (image.getCheckpointTxId() >= minTxId) continue;
            this.purger.purgeImage(image);
        }
    }

    private long getImageTxIdToRetain(FSImageTransactionalStorageInspector inspector) {
        List<FSImageStorageInspector.FSImageFile> images = inspector.getFoundImages();
        if (images.isEmpty()) {
            return 0L;
        }
        TreeSet imageTxIds = Sets.newTreeSet(Collections.reverseOrder());
        for (FSImageStorageInspector.FSImageFile image : images) {
            imageTxIds.add(image.getCheckpointTxId());
        }
        ArrayList imageTxIdsList = Lists.newArrayList((Iterable)imageTxIds);
        int toRetain = Math.min(this.numCheckpointsToRetain, imageTxIdsList.size());
        long minTxId = (Long)imageTxIdsList.get(toRetain - 1);
        LOG.info("Going to retain {} images with txid >= {}", (Object)toRetain, (Object)minTxId);
        return minTxId;
    }

    void purgeOldLegacyOIVImages(String dir, long txid) {
        File oivImageDir = new File(dir);
        final String oivImagePrefix = NNStorage.NameNodeFile.IMAGE_LEGACY_OIV.getName();
        String[] filesInStorage = oivImageDir.list(new FilenameFilter(){

            @Override
            public boolean accept(File dir, String name) {
                return name.matches(oivImagePrefix + "_(\\d+)");
            }
        });
        if (filesInStorage != null && filesInStorage.length <= this.numCheckpointsToRetain) {
            return;
        }
        TreeSet<Long> sortedTxIds = new TreeSet<Long>();
        if (filesInStorage != null) {
            for (String fName : filesInStorage) {
                long fTxId;
                try {
                    fTxId = Long.parseLong(fName.substring(oivImagePrefix.length() + 1));
                }
                catch (NumberFormatException nfe) {
                    LOG.warn("Invalid file name. Skipping " + fName);
                    continue;
                }
                sortedTxIds.add(fTxId);
            }
        }
        Iterator iter = sortedTxIds.iterator();
        for (int numFilesToDelete = sortedTxIds.size() - this.numCheckpointsToRetain; numFilesToDelete > 0 && iter.hasNext(); --numFilesToDelete) {
            long txIdVal = (Long)iter.next();
            String fileName = NNStorage.getLegacyOIVImageFileName(txIdVal);
            LOG.info("Deleting " + fileName);
            File fileToDelete = new File(oivImageDir, fileName);
            if (fileToDelete.delete()) continue;
            LOG.warn("Failed to delete image file: " + fileToDelete);
        }
    }

    static class DeletionStoragePurger
    implements StoragePurger {
        DeletionStoragePurger() {
        }

        @Override
        public void purgeLog(FileJournalManager.EditLogFile log) {
            LOG.info("Purging old edit log {}", (Object)log);
            DeletionStoragePurger.deleteOrWarn(log.getFile());
        }

        @Override
        public void purgeImage(FSImageStorageInspector.FSImageFile image) {
            LOG.info("Purging old image {}", (Object)image);
            DeletionStoragePurger.deleteOrWarn(image.getFile());
            DeletionStoragePurger.deleteOrWarn(MD5FileUtils.getDigestFileForFile(image.getFile()));
        }

        private static void deleteOrWarn(File file) {
            if (!file.delete()) {
                LOG.warn("Could not delete {}", (Object)file);
            }
        }

        @Override
        public void markStale(FileJournalManager.EditLogFile log) {
            try {
                log.moveAsideStaleInprogressFile();
            }
            catch (IOException e) {
                LOG.warn("Could not mark {} as stale", (Object)log, (Object)e);
            }
        }
    }

    static interface StoragePurger {
        public void purgeLog(FileJournalManager.EditLogFile var1);

        public void purgeImage(FSImageStorageInspector.FSImageFile var1);

        public void markStale(FileJournalManager.EditLogFile var1);
    }
}

