/*
 * Decompiled with CFR 0.152.
 */
package io.atomix.copycat.server.storage;

import io.atomix.catalyst.buffer.Buffer;
import io.atomix.catalyst.buffer.FileBuffer;
import io.atomix.catalyst.buffer.HeapBuffer;
import io.atomix.catalyst.buffer.MappedBuffer;
import io.atomix.catalyst.buffer.SlicedBuffer;
import io.atomix.catalyst.serializer.Serializer;
import io.atomix.catalyst.util.Assert;
import io.atomix.copycat.server.storage.SegmentDescriptor;
import io.atomix.copycat.server.storage.SegmentFile;
import io.atomix.copycat.server.storage.SegmentManager;
import io.atomix.copycat.server.storage.entry.Entry;
import io.atomix.copycat.server.storage.index.OffsetIndex;
import io.atomix.copycat.server.storage.util.OffsetPredicate;
import io.atomix.copycat.server.storage.util.TermIndex;
import java.util.zip.CRC32;

public class Segment
implements AutoCloseable {
    private final SegmentFile file;
    private final SegmentDescriptor descriptor;
    private final Serializer serializer;
    private final Buffer buffer;
    private final HeapBuffer memory = HeapBuffer.allocate();
    private final OffsetIndex offsetIndex;
    private final OffsetPredicate offsetPredicate;
    private final TermIndex termIndex = new TermIndex();
    private final SegmentManager manager;
    private long skip = 0L;
    private boolean open = true;

    Segment(SegmentFile file, Buffer buffer, SegmentDescriptor descriptor, OffsetIndex offsetIndex, OffsetPredicate offsetPredicate, Serializer serializer, SegmentManager manager) {
        this.serializer = (Serializer)Assert.notNull((Object)serializer, (String)"serializer");
        this.file = (SegmentFile)Assert.notNull((Object)file, (String)"file");
        this.buffer = (Buffer)Assert.notNull((Object)buffer, (String)"buffer");
        this.descriptor = (SegmentDescriptor)Assert.notNull((Object)descriptor, (String)"descriptor");
        this.offsetIndex = (OffsetIndex)Assert.notNull((Object)offsetIndex, (String)"offsetIndex");
        this.offsetPredicate = (OffsetPredicate)Assert.notNull((Object)offsetPredicate, (String)"offsetPredicate");
        this.manager = (SegmentManager)Assert.notNull((Object)manager, (String)"manager");
        this.buildIndex();
    }

    private void buildIndex() {
        long position = this.buffer.mark().position();
        int length = this.buffer.readInt();
        while (length != 0) {
            this.buffer.read(this.memory.clear().limit((long)length));
            this.memory.flip();
            long checksum = this.memory.readUnsignedInt();
            long offset = this.memory.readLong();
            Long term = this.memory.readBoolean() ? Long.valueOf(this.memory.readLong()) : null;
            int entryPosition = (int)this.memory.position();
            int entryLength = length - entryPosition;
            CRC32 crc32 = new CRC32();
            crc32.update(this.memory.array(), entryPosition, entryLength);
            if (checksum != crc32.getValue()) break;
            if (term != null) {
                this.termIndex.index(offset, term);
            }
            this.offsetIndex.index(offset, position);
            position = this.buffer.position();
            length = this.buffer.mark().readInt();
        }
        this.buffer.reset();
    }

    public SegmentFile file() {
        return this.file;
    }

    public SegmentDescriptor descriptor() {
        return this.descriptor;
    }

    public boolean isOpen() {
        return this.open;
    }

    public boolean isEmpty() {
        return this.offsetIndex.size() > 0 ? this.offsetIndex.lastOffset() + 1L + this.skip == 0L : this.skip == 0L;
    }

    public boolean isCompacted() {
        return this.descriptor.version() > 1L;
    }

    public boolean isFull() {
        return this.size() >= this.descriptor.maxSegmentSize() || this.offsetIndex.size() >= this.descriptor.maxEntries();
    }

    public long size() {
        return this.buffer.offset() + this.buffer.position();
    }

    public long length() {
        return !this.isEmpty() ? this.offsetIndex.lastOffset() + 1L + this.skip : 0L;
    }

    public int count() {
        return this.offsetIndex.size();
    }

    long index() {
        return this.descriptor.index();
    }

    public long firstIndex() {
        this.assertSegmentOpen();
        return !this.isEmpty() ? this.descriptor.index() : 0L;
    }

    public long lastIndex() {
        this.assertSegmentOpen();
        return !this.isEmpty() ? this.offsetIndex.lastOffset() + this.descriptor.index() + this.skip : this.descriptor.index() - 1L;
    }

    public long nextIndex() {
        return !this.isEmpty() ? this.lastIndex() + 1L : this.descriptor.index() + this.skip;
    }

    public long offset(long index) {
        return this.offsetIndex.find(this.relativeOffset(index));
    }

    private long relativeOffset(long index) {
        return index - this.descriptor.index();
    }

    private void checkRange(long index) {
        Assert.indexNot((boolean)this.isEmpty(), (String)"segment is empty", (Object[])new Object[0]);
        Assert.indexNot((index < this.firstIndex() ? 1 : 0) != 0, (String)(index + " is less than the first index in the segment"), (Object[])new Object[0]);
        Assert.indexNot((index > this.lastIndex() ? 1 : 0) != 0, (String)(index + " is greater than the last index in the segment"), (Object[])new Object[0]);
    }

    public long append(Entry entry) {
        Assert.notNull((Object)entry, (String)"entry");
        Assert.stateNot((boolean)this.isFull(), (String)"segment is full", (Object[])new Object[0]);
        long index = this.nextIndex();
        Assert.index((index == entry.getIndex() ? 1 : 0) != 0, (String)"inconsistent index: %s", (Object[])new Object[]{entry.getIndex()});
        long offset = this.relativeOffset(index);
        long term = entry.getTerm();
        long lastTerm = this.termIndex.term();
        Assert.arg((term > 0L && term >= lastTerm ? 1 : 0) != 0, (String)"term must be monotonically increasing", (Object[])new Object[0]);
        long position = this.buffer.position();
        boolean skipTerm = term == lastTerm;
        int headerLength = 13 + (skipTerm ? 0 : 8);
        this.memory.clear().skip((long)headerLength);
        this.serializer.writeObject((Object)entry, (Buffer)this.memory);
        this.memory.flip();
        int totalLength = (int)this.memory.limit();
        int entryLength = totalLength - headerLength;
        entry.setSize(totalLength);
        CRC32 crc32 = new CRC32();
        crc32.update(this.memory.array(), headerLength, entryLength);
        long checksum = crc32.getValue();
        this.memory.rewind().writeUnsignedInt(checksum).writeLong(offset);
        if (skipTerm) {
            this.memory.writeBoolean(false);
        } else {
            this.memory.writeBoolean(true).writeLong(term);
        }
        this.buffer.writeInt(totalLength).write(this.memory.rewind());
        this.offsetIndex.index(offset, position);
        if (term > lastTerm) {
            this.termIndex.index(offset, term);
        }
        this.skip = 0L;
        return index;
    }

    public long term(long index) {
        this.assertSegmentOpen();
        this.checkRange(index);
        long offset = this.relativeOffset(index);
        return this.termIndex.lookup(offset);
    }

    public synchronized <T extends Entry> T get(long index) {
        this.assertSegmentOpen();
        this.checkRange(index);
        long offset = this.relativeOffset(index);
        long position = this.offsetIndex.position(offset);
        if (position != -1L) {
            int length = this.buffer.readInt(position);
            try (Buffer slice = this.buffer.slice(position + 4L, (long)length);){
                slice.read(this.memory.clear().limit((long)length));
                this.memory.flip();
            }
            long checksum = this.memory.readUnsignedInt();
            long entryOffset = this.memory.readLong();
            Assert.state((entryOffset == offset ? 1 : 0) != 0, (String)"inconsistent index: %s", (Object[])new Object[]{index});
            if (this.memory.readBoolean()) {
                this.memory.skip(8L);
            }
            int entryPosition = (int)this.memory.position();
            int entryLength = length - entryPosition;
            CRC32 crc32 = new CRC32();
            crc32.update(this.memory.array(), entryPosition, entryLength);
            if (checksum == crc32.getValue()) {
                Entry entry = (Entry)this.serializer.readObject((Buffer)this.memory);
                ((Entry)((Entry)entry.setIndex(index)).setTerm(this.termIndex.lookup(offset))).setSize(length);
                return (T)entry;
            }
        }
        return null;
    }

    boolean validIndex(long index) {
        this.assertSegmentOpen();
        return !this.isEmpty() && index >= this.firstIndex() && index <= this.lastIndex();
    }

    public boolean contains(long index) {
        this.assertSegmentOpen();
        if (!this.validIndex(index)) {
            return false;
        }
        long offset = this.relativeOffset(index);
        return this.offsetIndex.contains(offset);
    }

    public boolean release(long index) {
        this.assertSegmentOpen();
        long offset = this.offsetIndex.find(this.relativeOffset(index));
        return offset != -1L && this.offsetPredicate.release(offset);
    }

    public boolean isLive(long index) {
        this.assertSegmentOpen();
        return this.offsetPredicate.test(this.offsetIndex.find(this.relativeOffset(index)));
    }

    public long releaseCount() {
        this.assertSegmentOpen();
        return this.offsetPredicate.count();
    }

    public OffsetPredicate offsetPredicate() {
        return this.offsetPredicate;
    }

    public Segment skip(long entries) {
        this.assertSegmentOpen();
        this.skip += entries;
        return this;
    }

    public Segment truncate(long index) {
        this.assertSegmentOpen();
        Assert.index((index >= this.manager.commitIndex() ? 1 : 0) != 0, (String)"cannot truncate committed index", (Object[])new Object[0]);
        long offset = this.relativeOffset(index);
        long lastOffset = this.offsetIndex.lastOffset();
        long diff = Math.abs(lastOffset - offset);
        this.skip = Math.max(this.skip - diff, 0L);
        if (offset < lastOffset) {
            long position = this.offsetIndex.truncate(offset);
            ((Buffer)this.buffer.position(position).zero(position)).flush();
            this.termIndex.truncate(offset);
        }
        return this;
    }

    public Segment flush() {
        this.buffer.flush();
        this.offsetIndex.flush();
        return this;
    }

    @Override
    public void close() {
        this.buffer.close();
        this.offsetIndex.close();
        this.offsetPredicate.close();
        this.descriptor.close();
        this.open = false;
    }

    public void delete() {
        Buffer buffer;
        Buffer buffer2 = buffer = this.buffer instanceof SlicedBuffer ? ((SlicedBuffer)this.buffer).root() : this.buffer;
        if (buffer instanceof FileBuffer) {
            ((FileBuffer)buffer).delete();
        } else if (buffer instanceof MappedBuffer) {
            ((MappedBuffer)buffer).delete();
        }
        this.offsetIndex.delete();
    }

    public String toString() {
        return String.format("Segment[id=%d, version=%d, index=%d, length=%d]", this.descriptor.id(), this.descriptor.version(), this.firstIndex(), this.length());
    }

    private void assertSegmentOpen() {
        Assert.state((boolean)this.isOpen(), (String)"segment not open", (Object[])new Object[0]);
    }
}

