/*
 * Decompiled with CFR 0.152.
 */
package io.atomix.copycat.client.session;

import io.atomix.catalyst.concurrent.Scheduled;
import io.atomix.catalyst.concurrent.ThreadContext;
import io.atomix.catalyst.util.Assert;
import io.atomix.copycat.client.ConnectionStrategy;
import io.atomix.copycat.client.session.ClientSessionState;
import io.atomix.copycat.client.util.ClientConnection;
import io.atomix.copycat.error.CopycatError;
import io.atomix.copycat.protocol.KeepAliveRequest;
import io.atomix.copycat.protocol.RegisterRequest;
import io.atomix.copycat.protocol.Response;
import io.atomix.copycat.protocol.UnregisterRequest;
import io.atomix.copycat.session.ClosedSessionException;
import io.atomix.copycat.session.Session;
import java.net.ConnectException;
import java.time.Duration;
import java.util.concurrent.CompletableFuture;

final class ClientSessionManager {
    private final ClientSessionState state;
    private final ClientConnection connection;
    private final ThreadContext context;
    private final ConnectionStrategy strategy;
    private final Duration sessionTimeout;
    private Duration interval;
    private Scheduled keepAlive;

    ClientSessionManager(ClientConnection connection, ClientSessionState state, ThreadContext context, ConnectionStrategy connectionStrategy, Duration sessionTimeout) {
        this.connection = (ClientConnection)Assert.notNull((Object)connection, (String)"connection");
        this.state = (ClientSessionState)Assert.notNull((Object)state, (String)"state");
        this.context = (ThreadContext)Assert.notNull((Object)context, (String)"context");
        this.strategy = (ConnectionStrategy)Assert.notNull((Object)connectionStrategy, (String)"connectionStrategy");
        this.sessionTimeout = (Duration)Assert.notNull((Object)sessionTimeout, (String)"sessionTimeout");
    }

    public CompletableFuture<Void> open() {
        CompletableFuture<Void> future = new CompletableFuture<Void>();
        this.context.executor().execute(() -> this.register(new RegisterAttempt(1, future)));
        return future;
    }

    public CompletableFuture<Void> expire() {
        CompletableFuture<Void> future = new CompletableFuture<Void>();
        this.context.executor().execute(() -> {
            if (this.keepAlive != null) {
                this.keepAlive.cancel();
            }
            this.state.setState(Session.State.EXPIRED);
            future.complete(null);
        });
        return future;
    }

    private void register(RegisterAttempt attempt) {
        this.state.getLogger().debug("Registering session: attempt {}", (Object)attempt.attempt);
        RegisterRequest request = RegisterRequest.builder().withClient(this.state.getClientId()).withTimeout(this.sessionTimeout.toMillis()).build();
        this.state.getLogger().trace("Sending {}", (Object)request);
        this.connection.reset().sendAndReceive(request).whenComplete((response, error) -> {
            if (error == null) {
                this.state.getLogger().trace("Received {}", response);
                if (response.status() == Response.Status.OK) {
                    this.interval = Duration.ofMillis(response.timeout()).dividedBy(2L);
                    this.connection.reset(response.leader(), response.members());
                    this.state.setSessionId(response.session()).setState(Session.State.OPEN);
                    this.state.getLogger().info("Registered session {}", (Object)response.session());
                    attempt.complete();
                    this.keepAlive();
                } else {
                    this.strategy.attemptFailed(attempt);
                }
            } else {
                this.strategy.attemptFailed(attempt);
            }
        });
    }

    private void keepAlive() {
        this.keepAlive(true);
    }

    private void keepAlive(boolean retryOnFailure) {
        long sessionId = this.state.getSessionId();
        if (this.state.getState() == Session.State.UNSTABLE) {
            this.connection.reset();
        }
        KeepAliveRequest request = ((KeepAliveRequest.Builder)KeepAliveRequest.builder().withSession(sessionId)).withCommandSequence(this.state.getCommandResponse()).withEventIndex(this.state.getEventIndex()).build();
        this.state.getLogger().trace("{} - Sending {}", (Object)sessionId, (Object)request);
        this.connection.sendAndReceive(request).whenComplete((response, error) -> {
            if (this.state.getState() != Session.State.CLOSED) {
                if (error == null) {
                    this.state.getLogger().trace("{} - Received {}", (Object)sessionId, response);
                    if (response.status() == Response.Status.OK) {
                        this.connection.reset(response.leader(), response.members());
                        this.state.setState(Session.State.OPEN);
                        this.scheduleKeepAlive();
                    } else if (response.error() == CopycatError.Type.UNKNOWN_SESSION_ERROR) {
                        this.state.setState(Session.State.EXPIRED);
                    } else if (retryOnFailure && this.connection.leader() != null) {
                        this.connection.reset(null, this.connection.servers());
                        this.keepAlive(false);
                    } else {
                        this.state.setState(Session.State.UNSTABLE);
                        this.scheduleKeepAlive();
                    }
                } else if (retryOnFailure && this.connection.leader() != null) {
                    this.connection.reset(null, this.connection.servers());
                    this.keepAlive(false);
                } else {
                    this.state.setState(Session.State.UNSTABLE);
                    this.scheduleKeepAlive();
                }
            }
        });
    }

    private void scheduleKeepAlive() {
        if (this.keepAlive != null) {
            this.keepAlive.cancel();
        }
        this.keepAlive = this.context.schedule(this.interval, () -> {
            this.keepAlive = null;
            if (this.state.getState().active()) {
                this.keepAlive();
            }
        });
    }

    public CompletableFuture<Void> close() {
        if (this.state.getState() == Session.State.EXPIRED) {
            return CompletableFuture.completedFuture(null);
        }
        CompletableFuture<Void> future = new CompletableFuture<Void>();
        this.context.executor().execute(() -> {
            if (this.keepAlive != null) {
                this.keepAlive.cancel();
                this.keepAlive = null;
            }
            this.unregister(future);
        });
        return future;
    }

    private void unregister(CompletableFuture<Void> future) {
        this.unregister(true, future);
    }

    private void unregister(boolean retryOnFailure, CompletableFuture<Void> future) {
        long sessionId = this.state.getSessionId();
        if (this.state.getState() == Session.State.CLOSED) {
            future.complete(null);
            return;
        }
        this.state.getLogger().debug("Unregistering session: {}", (Object)sessionId);
        if (this.keepAlive != null) {
            this.keepAlive.cancel();
            this.keepAlive = null;
        }
        if (this.state.getState() == Session.State.UNSTABLE) {
            this.connection.reset();
        }
        UnregisterRequest request = (UnregisterRequest)((UnregisterRequest.Builder)UnregisterRequest.builder().withSession(sessionId)).build();
        this.state.getLogger().trace("{} - Sending {}", (Object)sessionId, (Object)request);
        this.connection.sendAndReceive(request).whenComplete((response, error) -> {
            if (this.state.getState() != Session.State.CLOSED) {
                if (error == null) {
                    this.state.getLogger().trace("{} - Received {}", (Object)sessionId, response);
                    if (response.status() == Response.Status.OK) {
                        this.state.setState(Session.State.CLOSED);
                        future.complete(null);
                    } else if (response.error() == CopycatError.Type.UNKNOWN_SESSION_ERROR) {
                        this.state.setState(Session.State.EXPIRED);
                        future.complete(null);
                    } else if (retryOnFailure && this.connection.leader() != null) {
                        this.connection.reset(null, this.connection.servers());
                        this.unregister(false, future);
                    } else {
                        this.state.setState(Session.State.UNSTABLE);
                        future.completeExceptionally((Throwable)new ClosedSessionException("failed to unregister session"));
                    }
                } else if (retryOnFailure && this.connection.leader() != null) {
                    this.connection.reset(null, this.connection.servers());
                    this.unregister(false, future);
                } else {
                    this.state.setState(Session.State.UNSTABLE);
                    future.completeExceptionally((Throwable)new ClosedSessionException("failed to unregister session"));
                }
            }
        });
    }

    public CompletableFuture<Void> kill() {
        return CompletableFuture.runAsync(() -> {
            if (this.keepAlive != null) {
                this.keepAlive.cancel();
            }
            this.state.setState(Session.State.CLOSED);
        }, this.context.executor());
    }

    public String toString() {
        return String.format("%s[session=%d]", this.getClass().getSimpleName(), this.state.getSessionId());
    }

    private final class RegisterAttempt
    implements ConnectionStrategy.Attempt {
        private final int attempt;
        private final CompletableFuture<Void> future;

        private RegisterAttempt(int attempt, CompletableFuture<Void> future) {
            this.attempt = attempt;
            this.future = future;
        }

        @Override
        public int attempt() {
            return this.attempt;
        }

        public void complete() {
            this.complete(null);
        }

        public void complete(Void result) {
            this.future.complete(result);
        }

        @Override
        public void fail() {
            this.future.completeExceptionally(new ConnectException("failed to register session"));
        }

        @Override
        public void fail(Throwable error) {
            this.future.completeExceptionally(error);
        }

        @Override
        public void retry() {
            ClientSessionManager.this.state.getLogger().debug("Retrying session register attempt");
            ClientSessionManager.this.register(new RegisterAttempt(this.attempt + 1, this.future));
        }

        @Override
        public void retry(Duration after) {
            ClientSessionManager.this.state.getLogger().debug("Retrying session register attempt");
            ClientSessionManager.this.context.schedule(after, () -> ClientSessionManager.this.register(new RegisterAttempt(this.attempt + 1, this.future)));
        }
    }
}

