/*
 * Decompiled with CFR 0.152.
 */
package org.apache.camel.oauth;

import com.google.gson.Gson;
import com.google.gson.JsonElement;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import com.google.gson.JsonPrimitive;
import com.nimbusds.jose.JWSVerifier;
import com.nimbusds.jose.crypto.RSASSAVerifier;
import com.nimbusds.jose.jwk.JWKSet;
import com.nimbusds.jose.jwk.RSAKey;
import com.nimbusds.jwt.SignedJWT;
import java.security.interfaces.RSAPublicKey;
import java.util.ArrayList;
import java.util.Map;
import java.util.Optional;
import org.apache.camel.oauth.JWTOptions;
import org.apache.camel.oauth.OAuthConfig;
import org.apache.camel.oauth.OAuthException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class UserProfile {
    static final Logger log = LoggerFactory.getLogger(UserProfile.class);
    private final JsonObject principal;
    private final JsonObject attributes;

    public UserProfile(JsonObject principal, JsonObject attributes) {
        this.principal = principal;
        this.attributes = attributes;
    }

    public static UserProfile fromJson(OAuthConfig config, JsonObject json) {
        JsonObject tokenJwt;
        JsonObject principal = json.deepCopy();
        JsonObject attributes = new JsonObject();
        long now = System.currentTimeMillis() / 1000L;
        if (principal.has("expires_in")) {
            long expiresIn = json.get("expires_in").getAsLong();
            attributes.addProperty("exp", (Number)(now + expiresIn));
            attributes.addProperty("iat", (Number)now);
        }
        if (principal.has("access_token")) {
            String accessToken = json.get("access_token").getAsString();
            tokenJwt = UserProfile.verifyToken(config, accessToken, false);
            attributes.add("accessToken", (JsonElement)tokenJwt);
            UserProfile.copyProperties(tokenJwt, attributes, true, "exp", "iat", "nbf", "sub");
            attributes.add("rootClaim", (JsonElement)new JsonPrimitive("accessToken"));
        }
        if (principal.has("id_token")) {
            String idToken = json.get("id_token").getAsString();
            tokenJwt = UserProfile.verifyToken(config, idToken, true);
            attributes.add("idToken", (JsonElement)tokenJwt);
            UserProfile.copyProperties(tokenJwt, attributes, false, "sub", "name", "email", "picture");
            UserProfile.copyProperties(tokenJwt, principal, true, "amr");
        }
        return new UserProfile(principal, attributes);
    }

    public Map<String, Object> attributes() {
        return (Map)new Gson().fromJson((JsonElement)this.attributes, Map.class);
    }

    public Map<String, Object> principal() {
        return (Map)new Gson().fromJson((JsonElement)this.principal, Map.class);
    }

    public boolean expired() {
        return this.ttl() <= 0L;
    }

    public long ttl() {
        long now = System.currentTimeMillis() / 1000L;
        Long ttl = Optional.ofNullable(this.attributes.get("exp")).map(Object::toString).map(s -> Long.parseLong(s) - now).orElse(0L);
        return ttl;
    }

    public String subject() {
        JsonObject idToken;
        if (this.principal.has("username")) {
            return this.principal.get("username").getAsString();
        }
        if (this.principal.has("userHandle")) {
            return this.principal.get("userHandle").getAsString();
        }
        if (this.attributes.has("idToken") && (idToken = this.attributes.get("idToken").getAsJsonObject()).has("sub")) {
            return idToken.get("sub").getAsString();
        }
        Optional<JsonElement> maybeSub = Optional.ofNullable(this.getClaim("sub"));
        return maybeSub.map(JsonElement::getAsString).orElse(null);
    }

    public Optional<String> accessToken() {
        JsonElement accessToken = this.principal.get("access_token");
        return Optional.ofNullable(accessToken).map(JsonElement::getAsString);
    }

    public Optional<String> idToken() {
        JsonElement idToken = this.principal.get("id_token");
        return Optional.ofNullable(idToken).map(JsonElement::getAsString);
    }

    public Optional<String> refreshToken() {
        JsonElement refreshToken = this.principal.get("refresh_token");
        return Optional.ofNullable(refreshToken).map(JsonElement::getAsString);
    }

    public JsonElement getClaim(String key) {
        JsonObject rootClaim;
        Optional<JsonObject> maybeRootClaim = Optional.ofNullable(this.attributes.get("rootClaim")).filter(JsonObject.class::isInstance).map(JsonObject.class::cast);
        if (maybeRootClaim.isPresent() && (rootClaim = maybeRootClaim.get()).has(key)) {
            return rootClaim.get(key);
        }
        if (this.attributes.has(key)) {
            return this.attributes.get(key);
        }
        return this.principal.get(key);
    }

    public void merge(UserProfile other) {
        other.principal.entrySet().forEach(en -> this.principal.add((String)en.getKey(), (JsonElement)en.getValue()));
        other.attributes.entrySet().forEach(en -> this.attributes.add((String)en.getKey(), (JsonElement)en.getValue()));
    }

    public void logDetails() {
        log.debug("User Attributes ...");
        this.attributes().forEach((k, v) -> log.debug("   {}: {}", k, v));
        log.debug("User Principal ...");
        this.principal().forEach((k, v) -> log.debug("   {}: {}", k, v));
    }

    private static JsonObject verifyToken(OAuthConfig config, String token, boolean idToken) throws OAuthException {
        JsonObject tokenJwt;
        try {
            String issuer;
            SignedJWT signedJWT = SignedJWT.parse((String)token);
            String keyID = signedJWT.getHeader().getKeyID();
            JWKSet jwkSet = config.getJWKSet();
            if (!jwkSet.isEmpty()) {
                RSAKey rsaKey = (RSAKey)jwkSet.getKeyByKeyId(keyID);
                if (rsaKey == null) {
                    throw new OAuthException("No matching key found for: " + keyID);
                }
                RSAPublicKey publicKey = rsaKey.toRSAPublicKey();
                if (!signedJWT.verify((JWSVerifier)new RSASSAVerifier(publicKey))) {
                    throw new OAuthException("Invalid token signature");
                }
            }
            String payload = signedJWT.getPayload().toString();
            tokenJwt = JsonParser.parseString((String)payload).getAsJsonObject();
            ArrayList<String> target = new ArrayList<String>();
            JWTOptions jwtOptions = config.getJWTOptions();
            if (tokenJwt.has("aud")) {
                try {
                    JsonElement aud = tokenJwt.get("aud");
                    if (aud.isJsonPrimitive()) {
                        target.add(aud.getAsString());
                    } else {
                        target.addAll(aud.getAsJsonArray().asList().stream().map(JsonElement::getAsString).toList());
                    }
                }
                catch (RuntimeException ex) {
                    throw new OAuthException("User audience isn't a JsonArray or String");
                }
            }
            if (!target.isEmpty()) {
                if (!idToken && jwtOptions.getAudience() != null) {
                    for (String el : jwtOptions.getAudience()) {
                        if (target.contains(el)) continue;
                        throw new OAuthException("Invalid JWT audience. expected: " + el);
                    }
                } else if (!target.contains(config.getClientId())) {
                    throw new OAuthException("Invalid JWT audience. expected: " + config.getClientId());
                }
            }
            if ((issuer = jwtOptions.getIssuer()) != null && !issuer.equals(tokenJwt.get("iss").getAsString())) {
                throw new OAuthException("Invalid JWT issuer");
            }
            if (idToken && tokenJwt.has("azp")) {
                String clientId = config.getClientId();
                if (!clientId.equals(tokenJwt.get("azp").getAsString())) {
                    throw new OAuthException("Invalid authorised party != config.clientID");
                }
                if (!target.isEmpty() && !target.contains(tokenJwt.get("azp").getAsString())) {
                    throw new OAuthException("ID Token with multiple audiences, doesn't contain azp Claim value");
                }
            }
        }
        catch (OAuthException ex) {
            throw ex;
        }
        catch (Exception ex) {
            throw new OAuthException("Invalid token", ex);
        }
        return tokenJwt;
    }

    private static void copyProperties(JsonObject source, JsonObject target, boolean overwrite, String ... keys) {
        if (keys.length > 0) {
            for (String key : keys) {
                if (!source.has(key) || target.has(key) && !overwrite) continue;
                target.add(key, source.get(key));
            }
        } else {
            for (Map.Entry en : source.entrySet()) {
                if (target.has((String)en.getKey()) && !overwrite) continue;
                target.add((String)en.getKey(), (JsonElement)en.getValue());
            }
        }
    }
}

