/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.camel.component.ignite.messaging;

import java.net.URI;
import java.util.Map;

import org.apache.camel.Consumer;
import org.apache.camel.Processor;
import org.apache.camel.Producer;
import org.apache.camel.component.ignite.AbstractIgniteEndpoint;
import org.apache.camel.component.ignite.ClusterGroupExpression;
import org.apache.camel.component.ignite.IgniteComponent;
import org.apache.camel.spi.Metadata;
import org.apache.camel.spi.UriEndpoint;
import org.apache.camel.spi.UriParam;
import org.apache.ignite.Ignite;
import org.apache.ignite.IgniteMessaging;

/**
 * Ignite Messaging endpoint.
 */
@UriEndpoint(scheme = "ignite:messaging", title = "Ignite Messaging", syntax = "ignite:messaging:[topic]", label = "nosql,cache,messaging", 
    consumerClass = IgniteMessagingConsumer.class)
public class IgniteMessagingEndpoint extends AbstractIgniteEndpoint {

    @UriParam
    @Metadata(required = "true")
    private String topic;

    @UriParam
    private ClusterGroupExpression clusterGroupExpression;

    @UriParam
    private IgniteMessagingSendMode sendMode = IgniteMessagingSendMode.UNORDERED;

    @UriParam
    private Long timeout;

    public IgniteMessagingEndpoint(String endpointUri, URI remainingUri, Map<String, Object> parameters, IgniteComponent igniteComponent) {
        super(endpointUri, igniteComponent);
        topic = remainingUri.getHost();
    }

    @Override
    public Producer createProducer() throws Exception {
        // Validate options.
        if (topic == null) {
            throw new IllegalStateException("Cannot initialize an Ignite Messaging Producer with a null topic.");
        }

        if (sendMode == IgniteMessagingSendMode.ORDERED && timeout == null) {
            throw new IllegalStateException("Cannot initialize an Ignite Messaging Producer in ORDERED send mode without a timeout.");
        }

        // Initialize the Producer.
        IgniteMessaging messaging = createIgniteMessaging();
        return new IgniteMessagingProducer(this, igniteComponent().getIgnite(), messaging);
    }

    @Override
    public Consumer createConsumer(Processor processor) throws Exception {
        // Validate options.
        if (topic == null) {
            throw new IllegalStateException("Cannot initialize an Ignite Messaging Consumer with a null topic.");
        }

        // Initialize the Consumer.
        IgniteMessaging messaging = createIgniteMessaging();
        IgniteMessagingConsumer consumer = new IgniteMessagingConsumer(this, processor, messaging);
        configureConsumer(consumer);
        return consumer;
    }

    private IgniteMessaging createIgniteMessaging() {
        Ignite ignite = ignite();
        IgniteMessaging messaging = clusterGroupExpression == null ? ignite.message() : ignite.message(clusterGroupExpression.getClusterGroup(ignite));
        return messaging;
    }

    public String getTopic() {
        return topic;
    }

    public void setTopic(String topic) {
        this.topic = topic;
    }

    public ClusterGroupExpression getClusterGroupExpression() {
        return clusterGroupExpression;
    }

    public void setClusterGroupExpression(ClusterGroupExpression clusterGroupExpression) {
        this.clusterGroupExpression = clusterGroupExpression;
    }

    public Long getTimeout() {
        return timeout;
    }

    public void setTimeout(Long timeout) {
        this.timeout = timeout;
    }

    public IgniteMessagingSendMode getSendMode() {
        return sendMode;
    }

    public void setSendMode(IgniteMessagingSendMode sendMode) {
        this.sendMode = sendMode;
    }

}
