/*
 * Decompiled with CFR 0.152.
 */
package org.apache.bookkeeper.metadata.etcd;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.util.concurrent.UncheckedExecutionException;
import io.etcd.jetcd.ByteSequence;
import io.etcd.jetcd.Client;
import io.etcd.jetcd.KV;
import io.etcd.jetcd.KeyValue;
import io.etcd.jetcd.Txn;
import io.etcd.jetcd.Watch;
import io.etcd.jetcd.kv.DeleteResponse;
import io.etcd.jetcd.kv.GetResponse;
import io.etcd.jetcd.kv.TxnResponse;
import io.etcd.jetcd.op.Cmp;
import io.etcd.jetcd.op.CmpTarget;
import io.etcd.jetcd.op.Op;
import io.etcd.jetcd.options.DeleteOption;
import io.etcd.jetcd.options.GetOption;
import io.etcd.jetcd.options.PutOption;
import io.etcd.jetcd.options.WatchOption;
import io.etcd.jetcd.watch.WatchEvent;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;
import org.apache.bookkeeper.bookie.BookieException;
import org.apache.bookkeeper.discover.BookieServiceInfo;
import org.apache.bookkeeper.discover.RegistrationManager;
import org.apache.bookkeeper.meta.LedgerLayout;
import org.apache.bookkeeper.metadata.etcd.EtcdBookieRegister;
import org.apache.bookkeeper.metadata.etcd.EtcdConstants;
import org.apache.bookkeeper.metadata.etcd.EtcdLedgerManagerFactory;
import org.apache.bookkeeper.metadata.etcd.EtcdUtils;
import org.apache.bookkeeper.net.BookieId;
import org.apache.bookkeeper.versioning.LongVersion;
import org.apache.bookkeeper.versioning.Version;
import org.apache.bookkeeper.versioning.Versioned;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class EtcdRegistrationManager
implements RegistrationManager {
    private static final Logger log = LoggerFactory.getLogger(EtcdRegistrationManager.class);
    private final String scope;
    private final Client client;
    private final boolean ownClient;
    private final KV kvClient;
    private final EtcdBookieRegister bkRegister;

    @VisibleForTesting
    EtcdRegistrationManager(Client client, String scope) {
        this(client, scope, 60L);
    }

    @VisibleForTesting
    EtcdRegistrationManager(Client client, String scope, long ttlSeconds) {
        this(client, scope, ttlSeconds, () -> {});
    }

    @VisibleForTesting
    EtcdRegistrationManager(Client client, String scope, long ttlSeconds, RegistrationManager.RegistrationListener listener) {
        this(client, scope, new EtcdBookieRegister(client.getLeaseClient(), ttlSeconds).addRegistrationListener(listener).start(), true);
    }

    EtcdRegistrationManager(Client client, String scope, EtcdBookieRegister bkRegister) {
        this(client, scope, bkRegister, false);
    }

    private EtcdRegistrationManager(Client client, String scope, EtcdBookieRegister bkRegister, boolean ownClient) {
        this.scope = scope;
        this.client = client;
        this.kvClient = client.getKVClient();
        this.bkRegister = bkRegister;
        this.ownClient = ownClient;
    }

    public void close() {
        if (this.ownClient) {
            log.info("Closing registration manager under scope '{}'", (Object)this.scope);
            this.bkRegister.close();
            this.client.close();
            log.info("Successfully closed registration manager under scope '{}'", (Object)this.scope);
        }
    }

    public void registerBookie(BookieId bookieId, boolean readOnly, BookieServiceInfo bookieServiceInfo) throws BookieException {
        if (readOnly) {
            this.doRegisterReadonlyBookie(bookieId, this.bkRegister.get());
        } else {
            this.doRegisterBookie(EtcdUtils.getWritableBookiePath(this.scope, bookieId), this.bkRegister.get());
        }
    }

    private boolean checkRegNodeAndWaitExpired(String regPath, long leaseId) throws BookieException.MetadataStoreException {
        ByteSequence regPathBs = ByteSequence.from((String)regPath, (Charset)StandardCharsets.UTF_8);
        GetResponse getResp = (GetResponse)EtcdUtils.msResult(this.kvClient.get(regPathBs));
        if (getResp.getCount() <= 0L) {
            return false;
        }
        return this.waitUntilRegNodeExpired(regPath, leaseId);
    }

    private boolean waitUntilRegNodeExpired(String regPath, long leaseId) throws BookieException.MetadataStoreException {
        ByteSequence regPathBs = ByteSequence.from((String)regPath, (Charset)StandardCharsets.UTF_8);
        GetResponse getResp = (GetResponse)EtcdUtils.msResult(this.kvClient.get(regPathBs));
        if (getResp.getCount() <= 0L) {
            return false;
        }
        KeyValue kv = (KeyValue)getResp.getKvs().get(0);
        if (kv.getLease() != leaseId) {
            Watch watchClient = this.client.getWatchClient();
            CompletableFuture watchFuture = new CompletableFuture();
            log.info("Previous bookie registration (lease = {}) still exists at {}, so new lease '{}' will be waiting previous lease for {} seconds to be expired", new Object[]{kv.getLease(), regPath, leaseId, this.bkRegister.getTtlSeconds()});
            try (Watch.Watcher watcher = watchClient.watch(regPathBs, WatchOption.newBuilder().withRevision(getResp.getHeader().getRevision() + 1L).build(), response -> {
                for (WatchEvent event : response.getEvents()) {
                    log.info("Received watch event on '{}' : EventType = {}, lease {}", new Object[]{regPath, event.getEventType(), leaseId});
                    if (WatchEvent.EventType.DELETE != event.getEventType()) continue;
                    watchFuture.complete(null);
                    return;
                }
            }, exception -> {
                log.warn("Exception in keepAlive for watch event on {}, lease {}", new Object[]{regPath, leaseId, exception});
                watchFuture.completeExceptionally(new UncheckedExecutionException("Interrupted at waiting previous registration under " + regPath + " (lease = " + kv.getLease() + ") to be expired", exception));
            });){
                EtcdUtils.msResult(watchFuture, 2L * this.bkRegister.getTtlSeconds(), TimeUnit.SECONDS);
                boolean bl = false;
                return bl;
            }
        }
        return true;
    }

    private void doRegisterBookie(String regPath, long leaseId) throws BookieException.MetadataStoreException {
        if (this.checkRegNodeAndWaitExpired(regPath, leaseId)) {
            return;
        }
        ByteSequence regPathBs = ByteSequence.from((String)regPath, (Charset)StandardCharsets.UTF_8);
        Txn txn = this.kvClient.txn().If(new Cmp[]{new Cmp(regPathBs, Cmp.Op.GREATER, (CmpTarget)CmpTarget.createRevision((long)0L))}).Then(new Op[]{Op.get((ByteSequence)regPathBs, (GetOption)GetOption.DEFAULT)}).Else(new Op[]{Op.put((ByteSequence)regPathBs, (ByteSequence)ByteSequence.from((byte[])new byte[0]), (PutOption)PutOption.newBuilder().withLeaseId(this.bkRegister.get().longValue()).build())});
        TxnResponse txnResp = (TxnResponse)EtcdUtils.msResult(txn.commit());
        if (txnResp.isSucceeded()) {
            GetResponse getResp = (GetResponse)txnResp.getGetResponses().get(0);
            if (getResp.getCount() <= 0L) {
                throw new BookieException.MetadataStoreException("Failed to register bookie under '" + regPath + "', but no bookie is registered there.");
            }
            KeyValue kv = (KeyValue)getResp.getKvs().get(0);
            throw new BookieException.MetadataStoreException("Another bookie already registered under '" + regPath + "': lease = " + kv.getLease());
        }
        log.info("Successfully registered bookie at {}", (Object)regPath);
    }

    private void doRegisterReadonlyBookie(BookieId bookieId, long leaseId) throws BookieException.MetadataStoreException {
        String readonlyRegPath = EtcdUtils.getReadonlyBookiePath(this.scope, bookieId);
        this.doRegisterBookie(readonlyRegPath, leaseId);
        String writableRegPath = EtcdUtils.getWritableBookiePath(this.scope, bookieId);
        EtcdUtils.msResult(this.kvClient.delete(ByteSequence.from((String)writableRegPath, (Charset)StandardCharsets.UTF_8)));
    }

    public void unregisterBookie(BookieId bookieId, boolean readOnly) throws BookieException {
        String regPath = readOnly ? EtcdUtils.getReadonlyBookiePath(this.scope, bookieId) : EtcdUtils.getWritableBookiePath(this.scope, bookieId);
        DeleteResponse delResp = (DeleteResponse)EtcdUtils.msResult(this.kvClient.delete(ByteSequence.from((String)regPath, (Charset)StandardCharsets.UTF_8)));
        if (delResp.getDeleted() > 0L) {
            log.info("Successfully unregistered bookie {} from {}", (Object)bookieId, (Object)regPath);
        } else {
            log.info("Bookie disappeared from {} before unregistering", (Object)regPath);
        }
    }

    public boolean isBookieRegistered(BookieId bookieId) throws BookieException {
        CompletableFuture getWritableFuture = this.kvClient.get(ByteSequence.from((String)EtcdUtils.getWritableBookiePath(this.scope, bookieId), (Charset)StandardCharsets.UTF_8), GetOption.newBuilder().withCountOnly(true).build());
        CompletableFuture getReadonlyFuture = this.kvClient.get(ByteSequence.from((String)EtcdUtils.getReadonlyBookiePath(this.scope, bookieId), (Charset)StandardCharsets.UTF_8), GetOption.newBuilder().withCountOnly(true).build());
        return ((GetResponse)EtcdUtils.msResult(getWritableFuture)).getCount() > 0L || ((GetResponse)EtcdUtils.msResult(getReadonlyFuture)).getCount() > 0L;
    }

    public void writeCookie(BookieId bookieId, Versioned<byte[]> cookieData) throws BookieException {
        ByteSequence cookiePath = ByteSequence.from((String)EtcdUtils.getCookiePath(this.scope, bookieId), (Charset)StandardCharsets.UTF_8);
        Txn txn = this.kvClient.txn();
        if (Version.NEW == cookieData.getVersion()) {
            txn.If(new Cmp[]{new Cmp(cookiePath, Cmp.Op.GREATER, (CmpTarget)CmpTarget.createRevision((long)0L))}).Else(new Op[]{Op.put((ByteSequence)cookiePath, (ByteSequence)ByteSequence.from((byte[])((byte[])cookieData.getValue())), (PutOption)PutOption.DEFAULT)});
        } else {
            if (!(cookieData.getVersion() instanceof LongVersion)) {
                throw new BookieException.BookieIllegalOpException("Invalid version type, expected it to be LongVersion");
            }
            txn.If(new Cmp[]{new Cmp(cookiePath, Cmp.Op.EQUAL, (CmpTarget)CmpTarget.modRevision((long)((LongVersion)cookieData.getVersion()).getLongVersion()))}).Then(new Op[]{Op.put((ByteSequence)cookiePath, (ByteSequence)ByteSequence.from((byte[])((byte[])cookieData.getValue())), (PutOption)PutOption.DEFAULT)});
        }
        TxnResponse response = (TxnResponse)EtcdUtils.msResult(txn.commit());
        if (response.isSucceeded() != (Version.NEW != cookieData.getVersion())) {
            throw new BookieException.MetadataStoreException("Conflict on writing cookie for bookie " + bookieId);
        }
    }

    public Versioned<byte[]> readCookie(BookieId bookieId) throws BookieException {
        ByteSequence cookiePath = ByteSequence.from((String)EtcdUtils.getCookiePath(this.scope, bookieId), (Charset)StandardCharsets.UTF_8);
        GetResponse resp = (GetResponse)EtcdUtils.msResult(this.kvClient.get(cookiePath));
        if (resp.getCount() <= 0L) {
            throw new BookieException.CookieNotFoundException(bookieId.toString());
        }
        KeyValue kv = (KeyValue)resp.getKvs().get(0);
        return new Versioned((Object)kv.getValue().getBytes(), (Version)new LongVersion(kv.getModRevision()));
    }

    public void removeCookie(BookieId bookieId, Version version) throws BookieException {
        ByteSequence cookiePath = ByteSequence.from((String)EtcdUtils.getCookiePath(this.scope, bookieId), (Charset)StandardCharsets.UTF_8);
        Txn delTxn = this.kvClient.txn().If(new Cmp[]{new Cmp(cookiePath, Cmp.Op.EQUAL, (CmpTarget)CmpTarget.modRevision((long)((LongVersion)version).getLongVersion()))}).Then(new Op[]{Op.delete((ByteSequence)cookiePath, (DeleteOption)DeleteOption.DEFAULT)}).Else(new Op[]{Op.get((ByteSequence)cookiePath, (GetOption)GetOption.newBuilder().withCountOnly(true).build())});
        TxnResponse txnResp = (TxnResponse)EtcdUtils.msResult(delTxn.commit());
        if (!txnResp.isSucceeded()) {
            GetResponse getResp = (GetResponse)txnResp.getGetResponses().get(0);
            if (getResp.getCount() > 0L) {
                throw new BookieException.MetadataStoreException("Failed to remove cookie from " + cookiePath.toString(StandardCharsets.UTF_8) + " for bookie " + bookieId + " : bad version '" + version + "'");
            }
            throw new BookieException.CookieNotFoundException(bookieId.toString());
        }
        log.info("Removed cookie from {} for bookie {}", (Object)cookiePath.toString(StandardCharsets.UTF_8), (Object)bookieId);
    }

    public String getClusterInstanceId() throws BookieException {
        GetResponse response = (GetResponse)EtcdUtils.msResult(this.kvClient.get(ByteSequence.from((String)EtcdUtils.getClusterInstanceIdPath(this.scope), (Charset)StandardCharsets.UTF_8)));
        if (response.getCount() <= 0L) {
            log.error("BookKeeper metadata doesn't exist in Etcd. Has the cluster been initialized? Try running bin/bookkeeper shell initNewCluster");
            throw new BookieException.MetadataStoreException("BookKeeper is not initialized under '" + this.scope + "' yet");
        }
        KeyValue kv = (KeyValue)response.getKvs().get(0);
        return new String(kv.getValue().getBytes(), StandardCharsets.UTF_8);
    }

    public boolean prepareFormat() throws Exception {
        ByteSequence rootScopeKey = ByteSequence.from((String)this.scope, (Charset)StandardCharsets.UTF_8);
        GetResponse resp = (GetResponse)EtcdUtils.msResult(this.kvClient.get(rootScopeKey));
        return resp.getCount() > 0L;
    }

    public boolean initNewCluster() throws Exception {
        return EtcdRegistrationManager.initNewCluster(this.kvClient, this.scope);
    }

    static boolean initNewCluster(KV kvClient, String scope) throws Exception {
        ByteSequence rootScopeKey = ByteSequence.from((String)scope, (Charset)StandardCharsets.UTF_8);
        String instanceId = UUID.randomUUID().toString();
        LedgerLayout layout = new LedgerLayout(EtcdLedgerManagerFactory.class.getName(), 0);
        Txn initTxn = kvClient.txn().If(new Cmp[]{new Cmp(rootScopeKey, Cmp.Op.GREATER, (CmpTarget)CmpTarget.createRevision((long)0L))}).Else(new Op[]{Op.put((ByteSequence)rootScopeKey, (ByteSequence)EtcdConstants.EMPTY_BS, (PutOption)PutOption.DEFAULT), Op.put((ByteSequence)ByteSequence.from((String)EtcdUtils.getLayoutKey(scope), (Charset)StandardCharsets.UTF_8), (ByteSequence)ByteSequence.from((byte[])layout.serialize()), (PutOption)PutOption.DEFAULT), Op.put((ByteSequence)ByteSequence.from((String)EtcdUtils.getClusterInstanceIdPath(scope), (Charset)StandardCharsets.UTF_8), (ByteSequence)ByteSequence.from((String)instanceId, (Charset)StandardCharsets.UTF_8), (PutOption)PutOption.DEFAULT), Op.put((ByteSequence)ByteSequence.from((String)EtcdUtils.getCookiesPath(scope), (Charset)StandardCharsets.UTF_8), (ByteSequence)EtcdConstants.EMPTY_BS, (PutOption)PutOption.DEFAULT), Op.put((ByteSequence)ByteSequence.from((String)EtcdUtils.getBookiesPath(scope), (Charset)StandardCharsets.UTF_8), (ByteSequence)EtcdConstants.EMPTY_BS, (PutOption)PutOption.DEFAULT), Op.put((ByteSequence)ByteSequence.from((String)EtcdUtils.getWritableBookiesPath(scope), (Charset)StandardCharsets.UTF_8), (ByteSequence)EtcdConstants.EMPTY_BS, (PutOption)PutOption.DEFAULT), Op.put((ByteSequence)ByteSequence.from((String)EtcdUtils.getReadonlyBookiesPath(scope), (Charset)StandardCharsets.UTF_8), (ByteSequence)EtcdConstants.EMPTY_BS, (PutOption)PutOption.DEFAULT), Op.put((ByteSequence)ByteSequence.from((String)EtcdUtils.getLedgersPath(scope), (Charset)StandardCharsets.UTF_8), (ByteSequence)EtcdConstants.EMPTY_BS, (PutOption)PutOption.DEFAULT), Op.put((ByteSequence)ByteSequence.from((String)EtcdUtils.getBucketsPath(scope), (Charset)StandardCharsets.UTF_8), (ByteSequence)EtcdConstants.EMPTY_BS, (PutOption)PutOption.DEFAULT), Op.put((ByteSequence)ByteSequence.from((String)EtcdUtils.getUnderreplicationPath(scope), (Charset)StandardCharsets.UTF_8), (ByteSequence)EtcdConstants.EMPTY_BS, (PutOption)PutOption.DEFAULT)});
        return !((TxnResponse)EtcdUtils.msResult(initTxn.commit())).isSucceeded();
    }

    public boolean format() throws Exception {
        return EtcdRegistrationManager.format(this.kvClient, this.scope);
    }

    static boolean format(KV kvClient, String scope) throws Exception {
        ByteSequence rootScopeKey = ByteSequence.from((String)scope, (Charset)StandardCharsets.UTF_8);
        GetResponse resp = (GetResponse)EtcdUtils.msResult(kvClient.get(rootScopeKey));
        if (resp.getCount() <= 0L) {
            return EtcdRegistrationManager.initNewCluster(kvClient, scope);
        }
        if (EtcdRegistrationManager.nukeExistingCluster(kvClient, scope)) {
            return EtcdRegistrationManager.initNewCluster(kvClient, scope);
        }
        return false;
    }

    public boolean nukeExistingCluster() throws Exception {
        return EtcdRegistrationManager.nukeExistingCluster(this.kvClient, this.scope);
    }

    public void addRegistrationListener(RegistrationManager.RegistrationListener listener) {
        this.bkRegister.addRegistrationListener(listener);
    }

    static boolean nukeExistingCluster(KV kvClient, String scope) throws Exception {
        ByteSequence rootScopeKey = ByteSequence.from((String)scope, (Charset)StandardCharsets.UTF_8);
        GetResponse resp = (GetResponse)EtcdUtils.msResult(kvClient.get(rootScopeKey));
        if (resp.getCount() <= 0L) {
            log.info("There is no existing cluster with under scope '{}' in Etcd, so exiting nuke operation", (Object)scope);
            return true;
        }
        String bookiesPath = EtcdUtils.getBookiesPath(scope);
        String bookiesEndPath = EtcdUtils.getBookiesEndPath(scope);
        resp = (GetResponse)EtcdUtils.msResult(kvClient.get(ByteSequence.from((String)bookiesPath, (Charset)StandardCharsets.UTF_8), GetOption.newBuilder().withRange(ByteSequence.from((String)bookiesEndPath, (Charset)StandardCharsets.UTF_8)).withKeysOnly(true).build()));
        String writableBookiesPath = EtcdUtils.getWritableBookiesPath(scope);
        String readonlyBookiesPath = EtcdUtils.getReadonlyBookiesPath(scope);
        boolean hasBookiesAlive = false;
        for (KeyValue kv : resp.getKvs()) {
            String keyStr = new String(kv.getKey().getBytes(), StandardCharsets.UTF_8);
            if (keyStr.equals(bookiesPath) || keyStr.equals(writableBookiesPath) || keyStr.equals(readonlyBookiesPath)) continue;
            hasBookiesAlive = true;
            break;
        }
        if (hasBookiesAlive) {
            log.error("Bookies are still up and connected to this cluster, stop all bookies before nuking the cluster");
            return false;
        }
        DeleteResponse delResp = (DeleteResponse)EtcdUtils.msResult(kvClient.delete(rootScopeKey, DeleteOption.newBuilder().withRange(ByteSequence.from((String)EtcdUtils.getScopeEndKey(scope), (Charset)StandardCharsets.UTF_8)).build()));
        log.info("Successfully nuked cluster under scope '{}' : {} kv pairs deleted", (Object)scope, (Object)delResp.getDeleted());
        return true;
    }

    Client getClient() {
        return this.client;
    }

    EtcdBookieRegister getBkRegister() {
        return this.bkRegister;
    }
}

