/*
 * Decompiled with CFR 0.152.
 */
package org.apache.bk_v4_2_0.bookkeeper.bookie;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileFilter;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.RandomAccessFile;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.bk_v4_2_0.bookkeeper.bookie.Bookie;
import org.apache.bk_v4_2_0.bookkeeper.bookie.BufferedChannel;
import org.apache.bk_v4_2_0.bookkeeper.bookie.LedgerDirsManager;
import org.apache.bk_v4_2_0.bookkeeper.conf.ServerConfiguration;
import org.apache.bk_v4_2_0.bookkeeper.util.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class EntryLogger {
    private static final Logger LOG = LoggerFactory.getLogger(EntryLogger.class);
    volatile File currentDir;
    private LedgerDirsManager ledgerDirsManager;
    private AtomicBoolean shouldCreateNewEntryLog = new AtomicBoolean(false);
    private long logId;
    final long logSizeLimit;
    private volatile BufferedChannel logChannel;
    static final int LOGFILE_HEADER_SIZE = 1024;
    final ByteBuffer LOGFILE_HEADER = ByteBuffer.allocate(1024);
    static final int MIN_SANE_ENTRY_SIZE = 16;
    static final long MB = 0x100000L;
    private ConcurrentHashMap<Long, BufferedChannel> channels = new ConcurrentHashMap();

    public EntryLogger(ServerConfiguration conf, LedgerDirsManager ledgerDirsManager) throws IOException {
        this.ledgerDirsManager = ledgerDirsManager;
        this.logSizeLimit = conf.getEntryLogSizeLimit();
        this.LOGFILE_HEADER.put("BKLO".getBytes());
        this.logId = -1L;
        for (File dir : ledgerDirsManager.getAllLedgerDirs()) {
            if (!dir.exists()) {
                throw new FileNotFoundException("Entry log directory does not exist");
            }
            long lastLogId = this.getLastLogId(dir);
            if (lastLogId <= this.logId) continue;
            this.logId = lastLogId;
        }
        this.initialize();
    }

    synchronized long getCurrentLogId() {
        return this.logId;
    }

    protected void initialize() throws IOException {
        this.ledgerDirsManager.addLedgerDirsListener(this.getLedgerDirsListener());
        this.createNewLog();
    }

    private LedgerDirsManager.LedgerDirsListener getLedgerDirsListener() {
        return new LedgerDirsManager.LedgerDirsListener(){

            @Override
            public void diskFull(File disk) {
                if (EntryLogger.this.currentDir != null && EntryLogger.this.currentDir.equals(disk)) {
                    EntryLogger.this.shouldCreateNewEntryLog.set(true);
                }
            }

            @Override
            public void diskFailed(File disk) {
            }

            @Override
            public void allDisksFull() {
            }

            @Override
            public void fatalError() {
            }
        };
    }

    void createNewLog() throws IOException {
        if (this.logChannel != null) {
            this.logChannel.flush(true);
        }
        String logFileName = null;
        block0: do {
            logFileName = Long.toHexString(++this.logId) + ".log";
            for (File dir : this.ledgerDirsManager.getAllLedgerDirs()) {
                File newLogFile = new File(dir, logFileName);
                if (!newLogFile.exists()) continue;
                LOG.warn("Found existed entry log " + newLogFile + " when trying to create it as a new log.");
                logFileName = null;
                continue block0;
            }
        } while (logFileName == null);
        this.currentDir = this.ledgerDirsManager.pickRandomWritableDir();
        this.setLastLogId(this.currentDir, this.logId);
        File newLogFile = new File(this.currentDir, logFileName);
        this.logChannel = new BufferedChannel(new RandomAccessFile(newLogFile, "rw").getChannel(), 65536);
        this.logChannel.write((ByteBuffer)this.LOGFILE_HEADER.clear());
        this.channels.put(this.logId, this.logChannel);
    }

    protected boolean removeEntryLog(long entryLogId) {
        File entryLogFile;
        BufferedChannel bc = this.channels.remove(entryLogId);
        if (null != bc) {
            try {
                bc.getFileChannel().close();
            }
            catch (IOException ie) {
                LOG.warn("Exception while closing garbage collected entryLog file : ", (Throwable)ie);
            }
        }
        try {
            entryLogFile = this.findFile(entryLogId);
        }
        catch (FileNotFoundException e) {
            LOG.error("Trying to delete an entryLog file that could not be found: " + entryLogId + ".log");
            return false;
        }
        if (!entryLogFile.delete()) {
            LOG.warn("Could not delete entry log file {}", (Object)entryLogFile);
        }
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void setLastLogId(File dir, long logId) throws IOException {
        FileOutputStream fos = new FileOutputStream(new File(dir, "lastId"));
        BufferedWriter bw = new BufferedWriter(new OutputStreamWriter(fos));
        try {
            bw.write(Long.toHexString(logId) + "\n");
            bw.flush();
        }
        finally {
            try {
                bw.close();
            }
            catch (IOException e) {}
        }
    }

    private long getLastLogId(File dir) {
        long id = this.readLastLogId(dir);
        if (id > 0L) {
            return id;
        }
        File[] logFiles = dir.listFiles(new FileFilter(){

            @Override
            public boolean accept(File file) {
                return file.getName().endsWith(".log");
            }
        });
        ArrayList<Long> logs = new ArrayList<Long>();
        for (File lf : logFiles) {
            String idString = lf.getName().split("\\.")[0];
            try {
                long lid = Long.parseLong(idString, 16);
                logs.add(lid);
            }
            catch (NumberFormatException nfe) {
                // empty catch block
            }
        }
        if (0 == logs.size()) {
            return -1L;
        }
        Collections.sort(logs);
        return (Long)logs.get(logs.size() - 1);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private long readLastLogId(File f) {
        FileInputStream fis;
        try {
            fis = new FileInputStream(new File(f, "lastId"));
        }
        catch (FileNotFoundException e) {
            return -1L;
        }
        BufferedReader br = new BufferedReader(new InputStreamReader(fis));
        try {
            String lastIdString = br.readLine();
            long l = Long.parseLong(lastIdString, 16);
            return l;
        }
        catch (IOException e) {
            long l = -1L;
            return l;
        }
        catch (NumberFormatException e) {
            long l = -1L;
            return l;
        }
        finally {
            try {
                br.close();
            }
            catch (IOException e) {}
        }
    }

    synchronized void flush() throws IOException {
        if (this.logChannel != null) {
            this.logChannel.flush(true);
        }
    }

    synchronized long addEntry(long ledger, ByteBuffer entry) throws IOException {
        boolean createNewLog = this.shouldCreateNewEntryLog.get();
        if (createNewLog || this.logChannel.position() + (long)entry.remaining() + 4L > this.logSizeLimit) {
            this.createNewLog();
            if (createNewLog) {
                this.shouldCreateNewEntryLog.set(false);
            }
        }
        ByteBuffer buff = ByteBuffer.allocate(4);
        buff.putInt(entry.remaining());
        buff.flip();
        this.logChannel.write(buff);
        long pos = this.logChannel.position();
        this.logChannel.write(entry);
        return this.logId << 32 | pos;
    }

    byte[] readEntry(long ledgerId, long entryId, long location) throws IOException, Bookie.NoEntryException {
        BufferedChannel fc;
        long entryLogId = location >> 32;
        long pos = location & 0xFFFFFFFFL;
        ByteBuffer sizeBuff = ByteBuffer.allocate(4);
        pos -= 4L;
        try {
            fc = this.getChannelForLogId(entryLogId);
        }
        catch (FileNotFoundException e) {
            FileNotFoundException newe = new FileNotFoundException(e.getMessage() + " for " + ledgerId + " with location " + location);
            newe.setStackTrace(e.getStackTrace());
            throw newe;
        }
        if (fc.read(sizeBuff, pos) != sizeBuff.capacity()) {
            throw new Bookie.NoEntryException("Short read from entrylog " + entryLogId, ledgerId, entryId);
        }
        pos += 4L;
        sizeBuff.flip();
        int entrySize = sizeBuff.getInt();
        if ((long)entrySize > 0x100000L) {
            LOG.error("Sanity check failed for entry size of " + entrySize + " at location " + pos + " in " + entryLogId);
        }
        if (entrySize < 16) {
            LOG.error("Read invalid entry length {}", (Object)entrySize);
            throw new IOException("Invalid entry length " + entrySize);
        }
        byte[] data = new byte[entrySize];
        ByteBuffer buff = ByteBuffer.wrap(data);
        int rc = fc.read(buff, pos);
        if (rc != data.length) {
            throw new Bookie.NoEntryException("Short read for " + ledgerId + "@" + entryId + " in " + entryLogId + "@" + pos + "(" + rc + "!=" + data.length + ")", ledgerId, entryId);
        }
        buff.flip();
        long thisLedgerId = buff.getLong();
        if (thisLedgerId != ledgerId) {
            throw new IOException("problem found in " + entryLogId + "@" + entryId + " at position + " + pos + " entry belongs to " + thisLedgerId + " not " + ledgerId);
        }
        long thisEntryId = buff.getLong();
        if (thisEntryId != entryId) {
            throw new IOException("problem found in " + entryLogId + "@" + entryId + " at position + " + pos + " entry is " + thisEntryId + " not " + entryId);
        }
        return data;
    }

    private BufferedChannel getChannelForLogId(long entryLogId) throws IOException {
        BufferedChannel fc = this.channels.get(entryLogId);
        if (fc != null) {
            return fc;
        }
        File file = this.findFile(entryLogId);
        FileChannel newFc = new RandomAccessFile(file, "r").getChannel();
        newFc.position(newFc.size());
        fc = new BufferedChannel(newFc, 8192);
        BufferedChannel oldfc = this.channels.putIfAbsent(entryLogId, fc);
        if (oldfc != null) {
            newFc.close();
            return oldfc;
        }
        return fc;
    }

    boolean logExists(long logId) {
        for (File d : this.ledgerDirsManager.getAllLedgerDirs()) {
            File f = new File(d, Long.toHexString(logId) + ".log");
            if (!f.exists()) continue;
            return true;
        }
        return false;
    }

    private File findFile(long logId) throws FileNotFoundException {
        for (File d : this.ledgerDirsManager.getAllLedgerDirs()) {
            File f = new File(d, Long.toHexString(logId) + ".log");
            if (!f.exists()) continue;
            return f;
        }
        throw new FileNotFoundException("No file for log " + Long.toHexString(logId));
    }

    protected void scanEntryLog(long entryLogId, EntryLogScanner scanner) throws IOException {
        BufferedChannel bc;
        ByteBuffer sizeBuff = ByteBuffer.allocate(4);
        ByteBuffer lidBuff = ByteBuffer.allocate(8);
        try {
            bc = this.getChannelForLogId(entryLogId);
        }
        catch (IOException e) {
            LOG.warn("Failed to get channel to scan entry log: " + entryLogId + ".log");
            throw e;
        }
        long pos = 1024L;
        while (pos < bc.size()) {
            if (bc.read(sizeBuff, pos) != sizeBuff.capacity()) {
                throw new IOException("Short read for entry size from entrylog " + entryLogId);
            }
            long offset = pos;
            pos += 4L;
            sizeBuff.flip();
            int entrySize = sizeBuff.getInt();
            if ((long)entrySize > 0x100000L) {
                LOG.warn("Found large size entry of " + entrySize + " at location " + pos + " in " + entryLogId);
            }
            sizeBuff.clear();
            if (bc.read(lidBuff, pos) != lidBuff.capacity()) {
                throw new IOException("Short read for ledger id from entrylog " + entryLogId);
            }
            lidBuff.flip();
            long lid = lidBuff.getLong();
            lidBuff.clear();
            if (!scanner.accept(lid)) {
                pos += (long)entrySize;
                continue;
            }
            byte[] data = new byte[entrySize];
            ByteBuffer buff = ByteBuffer.wrap(data);
            int rc = bc.read(buff, pos);
            if (rc != data.length) {
                throw new IOException("Short read for ledger entry from entryLog " + entryLogId + "@" + pos + "(" + rc + "!=" + data.length + ")");
            }
            buff.flip();
            scanner.process(lid, offset, buff);
            pos += (long)entrySize;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void shutdown() {
        try {
            this.flush();
            for (Map.Entry<Long, BufferedChannel> channelEntry : this.channels.entrySet()) {
                channelEntry.getValue().getFileChannel().close();
            }
        }
        catch (IOException ie) {
            try {
                LOG.error("Error flush entry log during shutting down, which may cause entry log corrupted.", (Throwable)ie);
            }
            catch (Throwable throwable) {
                for (Map.Entry<Long, BufferedChannel> channelEntry : this.channels.entrySet()) {
                    FileChannel fileChannel = channelEntry.getValue().getFileChannel();
                    if (!fileChannel.isOpen()) continue;
                    IOUtils.close(LOG, fileChannel);
                }
                throw throwable;
            }
            for (Map.Entry<Long, BufferedChannel> channelEntry : this.channels.entrySet()) {
                FileChannel fileChannel = channelEntry.getValue().getFileChannel();
                if (!fileChannel.isOpen()) continue;
                IOUtils.close(LOG, fileChannel);
            }
        }
        for (Map.Entry<Long, BufferedChannel> channelEntry : this.channels.entrySet()) {
            FileChannel fileChannel = channelEntry.getValue().getFileChannel();
            if (!fileChannel.isOpen()) continue;
            IOUtils.close(LOG, fileChannel);
        }
    }

    static interface EntryLogScanner {
        public boolean accept(long var1);

        public void process(long var1, long var3, ByteBuffer var5) throws IOException;
    }
}

