/*
 * Decompiled with CFR 0.152.
 */
package org.apache.bk_v4_2_0.bookkeeper.replication;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.CountDownLatch;
import org.apache.bk_v4_2_0.bookkeeper.KeeperException;
import org.apache.bk_v4_2_0.bookkeeper.ZooKeeper;
import org.apache.bk_v4_2_0.bookkeeper.client.BKException;
import org.apache.bk_v4_2_0.bookkeeper.client.BookKeeper;
import org.apache.bk_v4_2_0.bookkeeper.client.BookKeeperAdmin;
import org.apache.bk_v4_2_0.bookkeeper.client.LedgerChecker;
import org.apache.bk_v4_2_0.bookkeeper.client.LedgerFragment;
import org.apache.bk_v4_2_0.bookkeeper.client.LedgerHandle;
import org.apache.bk_v4_2_0.bookkeeper.conf.ClientConfiguration;
import org.apache.bk_v4_2_0.bookkeeper.conf.ServerConfiguration;
import org.apache.bk_v4_2_0.bookkeeper.meta.LedgerManagerFactory;
import org.apache.bk_v4_2_0.bookkeeper.meta.LedgerUnderreplicationManager;
import org.apache.bk_v4_2_0.bookkeeper.proto.BookkeeperInternalCallbacks;
import org.apache.bk_v4_2_0.bookkeeper.replication.ReplicationException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ReplicationWorker
implements Runnable {
    private static Logger LOG = LoggerFactory.getLogger(ReplicationWorker.class);
    private final LedgerUnderreplicationManager underreplicationManager;
    private ServerConfiguration conf;
    private ZooKeeper zkc;
    private volatile boolean workerRunning = false;
    private final BookKeeperAdmin admin;
    private LedgerChecker ledgerChecker;
    private InetSocketAddress targetBookie;
    private BookKeeper bkc;
    private Thread workerThread;
    private long openLedgerRereplicationGracePeriod;
    private Timer pendingReplicationTimer;

    public ReplicationWorker(ZooKeeper zkc, ServerConfiguration conf, InetSocketAddress targetBKAddr) throws ReplicationException.CompatibilityException, KeeperException, InterruptedException, IOException {
        this.zkc = zkc;
        this.conf = conf;
        this.targetBookie = targetBKAddr;
        LedgerManagerFactory mFactory = LedgerManagerFactory.newLedgerManagerFactory(this.conf, this.zkc);
        this.underreplicationManager = mFactory.newLedgerUnderreplicationManager();
        this.bkc = new BookKeeper(new ClientConfiguration(conf), zkc);
        this.admin = new BookKeeperAdmin(this.bkc);
        this.ledgerChecker = new LedgerChecker(this.bkc);
        this.workerThread = new Thread((Runnable)this, "ReplicationWorker");
        this.openLedgerRereplicationGracePeriod = conf.getOpenLedgerRereplicationGracePeriod();
        this.pendingReplicationTimer = new Timer("PendingReplicationTimer");
    }

    public void start() {
        this.workerThread.start();
    }

    @Override
    public void run() {
        this.workerRunning = true;
        while (this.workerRunning) {
            try {
                this.rereplicate();
            }
            catch (InterruptedException e) {
                this.shutdown();
                Thread.currentThread().interrupt();
                LOG.info("InterruptedException while replicating fragments", (Throwable)e);
                return;
            }
            catch (BKException e) {
                this.shutdown();
                LOG.error("BKException while replicating fragments", (Throwable)e);
                return;
            }
            catch (ReplicationException.UnavailableException e) {
                this.shutdown();
                LOG.error("UnavailableException while replicating fragments", (Throwable)e);
                return;
            }
        }
    }

    private void rereplicate() throws InterruptedException, BKException, ReplicationException.UnavailableException {
        LedgerHandle lh;
        long ledgerIdToReplicate = this.underreplicationManager.getLedgerToRereplicate();
        LOG.info("Going to replicate the fragments of the ledger: " + ledgerIdToReplicate);
        try {
            lh = this.admin.openLedgerNoRecovery(ledgerIdToReplicate);
        }
        catch (BKException.BKNoSuchLedgerExistsException e) {
            LOG.info("BKNoSuchLedgerExistsException while opening ledger for replication. Other clients might have deleted the ledger. So, no harm to continue");
            this.underreplicationManager.markLedgerReplicated(ledgerIdToReplicate);
            return;
        }
        catch (BKException.BKReadException e) {
            LOG.info("BKReadException while opening ledger for replication. Enough Bookies might not have availableSo, no harm to continue");
            this.underreplicationManager.releaseUnderreplicatedLedger(ledgerIdToReplicate);
            return;
        }
        catch (BKException.BKBookieHandleNotAvailableException e) {
            LOG.info("BKBookieHandleNotAvailableException while opening ledger for replication. Enough Bookies might not have availableSo, no harm to continue");
            this.underreplicationManager.releaseUnderreplicatedLedger(ledgerIdToReplicate);
            return;
        }
        Set<LedgerFragment> fragments = this.getUnderreplicatedFragments(lh);
        LOG.info("Founds fragments " + fragments + " for replication from ledger: " + ledgerIdToReplicate);
        boolean foundOpenFragments = false;
        for (LedgerFragment ledgerFragment : fragments) {
            if (!ledgerFragment.isClosed()) {
                foundOpenFragments = true;
                continue;
            }
            if (this.isTargetBookieExistsInFragmentEnsemble(lh, ledgerFragment)) {
                LOG.info("Target Bookie[" + this.targetBookie + "] found in the fragment ensemble:" + ledgerFragment.getEnsemble());
                continue;
            }
            try {
                this.admin.replicateLedgerFragment(lh, ledgerFragment, this.targetBookie);
            }
            catch (BKException.BKBookieHandleNotAvailableException e) {
                LOG.warn("BKBookieHandleNotAvailableException while replicating the fragment", (Throwable)e);
            }
            catch (BKException.BKLedgerRecoveryException e) {
                LOG.warn("BKLedgerRecoveryException while replicating the fragment", (Throwable)e);
            }
        }
        if (foundOpenFragments) {
            this.deferLedgerLockRelease(ledgerIdToReplicate);
            return;
        }
        fragments = this.getUnderreplicatedFragments(lh);
        if (fragments.size() == 0) {
            LOG.info("Ledger replicated successfully. ledger id is: " + ledgerIdToReplicate);
            this.underreplicationManager.markLedgerReplicated(ledgerIdToReplicate);
        } else {
            this.underreplicationManager.releaseUnderreplicatedLedger(ledgerIdToReplicate);
        }
    }

    private Set<LedgerFragment> getUnderreplicatedFragments(LedgerHandle lh) throws InterruptedException {
        CheckerCallback checkerCb = new CheckerCallback();
        this.ledgerChecker.checkLedger(lh, checkerCb);
        Set<LedgerFragment> fragments = checkerCb.waitAndGetResult();
        return fragments;
    }

    private void deferLedgerLockRelease(final long ledgerId) {
        long gracePeriod = this.openLedgerRereplicationGracePeriod;
        TimerTask timerTask = new TimerTask(){

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             * Enabled aggressive block sorting
             * Enabled unnecessary exception pruning
             * Enabled aggressive exception aggregation
             */
            @Override
            public void run() {
                LedgerHandle lh = null;
                try {
                    LedgerFragment fragment;
                    lh = ReplicationWorker.this.admin.openLedgerNoRecovery(ledgerId);
                    Set fragments = ReplicationWorker.this.getUnderreplicatedFragments(lh);
                    Iterator i$ = fragments.iterator();
                    do {
                        if (!i$.hasNext()) return;
                    } while ((fragment = (LedgerFragment)i$.next()).isClosed());
                    lh = ReplicationWorker.this.admin.openLedger(ledgerId);
                    return;
                }
                catch (InterruptedException e) {
                    Thread.currentThread().interrupt();
                    LOG.info("InterruptedException while replicating fragments", (Throwable)e);
                    return;
                }
                catch (BKException e) {
                    LOG.error("BKException while fencing the ledger for rereplication of postponed ledgers", (Throwable)e);
                    return;
                }
                finally {
                    try {
                        if (lh != null) {
                            lh.close();
                        }
                    }
                    catch (InterruptedException e) {
                        Thread.currentThread().interrupt();
                        LOG.info("InterruptedException while closing ledger", (Throwable)e);
                    }
                    catch (BKException e) {
                        LOG.warn("BKException while closing ledger ", (Throwable)e);
                    }
                    finally {
                        try {
                            ReplicationWorker.this.underreplicationManager.releaseUnderreplicatedLedger(ledgerId);
                        }
                        catch (ReplicationException.UnavailableException e) {
                            ReplicationWorker.this.shutdown();
                            LOG.error("UnavailableException while replicating fragments", (Throwable)e);
                        }
                    }
                }
            }
        };
        this.pendingReplicationTimer.schedule(timerTask, gracePeriod);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void shutdown() {
        ReplicationWorker replicationWorker = this;
        synchronized (replicationWorker) {
            if (!this.workerRunning) {
                return;
            }
            this.workerRunning = false;
        }
        this.pendingReplicationTimer.cancel();
        try {
            this.workerThread.interrupt();
            this.workerThread.join();
        }
        catch (InterruptedException e) {
            LOG.error("Interrupted during shutting down replication worker : ", (Throwable)e);
            Thread.currentThread().interrupt();
        }
        try {
            this.bkc.close();
        }
        catch (InterruptedException e) {
            LOG.warn("Interrupted while closing the Bookie client", (Throwable)e);
            Thread.currentThread().interrupt();
        }
        catch (BKException e) {
            LOG.warn("Exception while closing the Bookie client", (Throwable)e);
        }
        try {
            this.underreplicationManager.close();
        }
        catch (ReplicationException.UnavailableException e) {
            LOG.warn("Exception while closing the ZkLedgerUnderrepliationManager", (Throwable)e);
        }
    }

    boolean isRunning() {
        return this.workerRunning;
    }

    private boolean isTargetBookieExistsInFragmentEnsemble(LedgerHandle lh, LedgerFragment ledgerFragment) {
        List<InetSocketAddress> ensemble = ledgerFragment.getEnsemble();
        for (InetSocketAddress bkAddr : ensemble) {
            if (!this.targetBookie.equals(bkAddr)) continue;
            return true;
        }
        return false;
    }

    private static class CheckerCallback
    implements BookkeeperInternalCallbacks.GenericCallback<Set<LedgerFragment>> {
        private Set<LedgerFragment> result = null;
        private CountDownLatch latch = new CountDownLatch(1);

        private CheckerCallback() {
        }

        @Override
        public void operationComplete(int rc, Set<LedgerFragment> result) {
            this.result = result;
            this.latch.countDown();
        }

        Set<LedgerFragment> waitAndGetResult() throws InterruptedException {
            this.latch.await();
            return this.result;
        }
    }
}

