/*
 * Decompiled with CFR 0.152.
 */
package org.apache.beam.sdk.io.hdfs;

import java.io.FileNotFoundException;
import java.io.InputStream;
import java.net.URI;
import java.nio.ByteBuffer;
import java.nio.channels.Channels;
import java.nio.channels.ReadableByteChannel;
import java.nio.channels.WritableByteChannel;
import java.nio.charset.StandardCharsets;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import org.apache.beam.sdk.io.FileSystems;
import org.apache.beam.sdk.io.TextIO;
import org.apache.beam.sdk.io.fs.CreateOptions;
import org.apache.beam.sdk.io.fs.MatchResult;
import org.apache.beam.sdk.io.fs.MoveOptions;
import org.apache.beam.sdk.io.fs.ResourceId;
import org.apache.beam.sdk.io.hdfs.HadoopFileSystem;
import org.apache.beam.sdk.io.hdfs.HadoopFileSystemOptions;
import org.apache.beam.sdk.io.hdfs.HadoopResourceId;
import org.apache.beam.sdk.options.PipelineOptions;
import org.apache.beam.sdk.testing.ExpectedLogs;
import org.apache.beam.sdk.testing.PAssert;
import org.apache.beam.sdk.testing.TestPipeline;
import org.apache.beam.sdk.transforms.PTransform;
import org.apache.beam.sdk.values.PCollection;
import org.apache.beam.vendor.guava.v26_0_jre.com.google.common.collect.ImmutableList;
import org.apache.beam.vendor.guava.v26_0_jre.com.google.common.collect.Iterables;
import org.apache.beam.vendor.guava.v26_0_jre.com.google.common.io.ByteStreams;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hdfs.MiniDFSCluster;
import org.hamcrest.Matcher;
import org.hamcrest.MatcherAssert;
import org.hamcrest.Matchers;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Ignore;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;
import org.junit.rules.TemporaryFolder;
import org.junit.runner.RunWith;
import org.junit.runners.JUnit4;
import org.slf4j.helpers.MessageFormatter;

@RunWith(value=JUnit4.class)
public class HadoopFileSystemTest {
    @Rule
    public TestPipeline p = TestPipeline.create();
    @Rule
    public TemporaryFolder tmpFolder = new TemporaryFolder();
    @Rule
    public ExpectedException thrown = ExpectedException.none();
    @Rule
    public final ExpectedLogs expectedLogs = ExpectedLogs.none(HadoopFileSystem.class);
    private MiniDFSCluster hdfsCluster;
    private URI hdfsClusterBaseUri;
    private HadoopFileSystem fileSystem;

    @Before
    public void setUp() throws Exception {
        Configuration configuration = new Configuration();
        configuration.set("hdfs.minidfs.basedir", this.tmpFolder.getRoot().getAbsolutePath());
        MiniDFSCluster.Builder builder = new MiniDFSCluster.Builder(configuration);
        this.hdfsCluster = builder.build();
        this.hdfsClusterBaseUri = new URI(configuration.get("fs.defaultFS") + "/");
        this.fileSystem = new HadoopFileSystem(Objects.requireNonNull(this.hdfsClusterBaseUri).getScheme(), configuration);
    }

    @After
    public void tearDown() {
        this.hdfsCluster.shutdown();
    }

    @Test
    public void testCreateAndReadFile() throws Exception {
        byte[] bytes = "testData".getBytes(StandardCharsets.UTF_8);
        this.create("testFile", bytes);
        Assert.assertArrayEquals((byte[])bytes, (byte[])this.read("testFile", 0L));
    }

    @Test
    public void testCreateAndReadFileWithShift() throws Exception {
        byte[] bytes = "testData".getBytes(StandardCharsets.UTF_8);
        this.create("testFile", bytes);
        int bytesToSkip = 3;
        byte[] expected = Arrays.copyOfRange(bytes, bytesToSkip, bytes.length);
        byte[] actual = this.read("testFile", bytesToSkip);
        Assert.assertArrayEquals((byte[])expected, (byte[])actual);
    }

    @Test
    public void testCreateAndReadFileWithShiftToEnd() throws Exception {
        byte[] bytes = "testData".getBytes(StandardCharsets.UTF_8);
        this.create("testFile", bytes);
        int bytesToSkip = bytes.length;
        byte[] expected = Arrays.copyOfRange(bytes, bytesToSkip, bytes.length);
        Assert.assertArrayEquals((byte[])expected, (byte[])this.read("testFile", bytesToSkip));
    }

    @Test
    public void testCopy() throws Exception {
        this.create("testFileA", "testDataA".getBytes(StandardCharsets.UTF_8));
        this.create("testFileB", "testDataB".getBytes(StandardCharsets.UTF_8));
        this.fileSystem.copy((List)ImmutableList.of((Object)this.testPath("testFileA"), (Object)this.testPath("testFileB")), (List)ImmutableList.of((Object)this.testPath("copyTestFileA"), (Object)this.testPath("copyTestFileB")));
        Assert.assertArrayEquals((byte[])"testDataA".getBytes(StandardCharsets.UTF_8), (byte[])this.read("testFileA", 0L));
        Assert.assertArrayEquals((byte[])"testDataB".getBytes(StandardCharsets.UTF_8), (byte[])this.read("testFileB", 0L));
        Assert.assertArrayEquals((byte[])"testDataA".getBytes(StandardCharsets.UTF_8), (byte[])this.read("copyTestFileA", 0L));
        Assert.assertArrayEquals((byte[])"testDataB".getBytes(StandardCharsets.UTF_8), (byte[])this.read("copyTestFileB", 0L));
    }

    @Test(expected=FileNotFoundException.class)
    public void testCopySourceMissing() throws Exception {
        this.fileSystem.copy((List)ImmutableList.of((Object)this.testPath("missingFile")), (List)ImmutableList.of((Object)this.testPath("copyTestFile")));
    }

    @Test
    public void testDelete() throws Exception {
        this.create("testFileA", "testDataA".getBytes(StandardCharsets.UTF_8));
        this.create("testFileB", "testDataB".getBytes(StandardCharsets.UTF_8));
        this.create("testFileC", "testDataC".getBytes(StandardCharsets.UTF_8));
        Assert.assertArrayEquals((byte[])"testDataA".getBytes(StandardCharsets.UTF_8), (byte[])this.read("testFileA", 0L));
        Assert.assertArrayEquals((byte[])"testDataB".getBytes(StandardCharsets.UTF_8), (byte[])this.read("testFileB", 0L));
        Assert.assertArrayEquals((byte[])"testDataC".getBytes(StandardCharsets.UTF_8), (byte[])this.read("testFileC", 0L));
        this.fileSystem.delete((Collection)ImmutableList.of((Object)this.testPath("testFileA"), (Object)this.testPath("testFileC")));
        List results = this.fileSystem.match((List)ImmutableList.of((Object)this.testPath("testFile*").toString()));
        MatcherAssert.assertThat((Object)results, (Matcher)Matchers.contains((Object[])new MatchResult[]{MatchResult.create((MatchResult.Status)MatchResult.Status.OK, (List)ImmutableList.of((Object)MatchResult.Metadata.builder().setResourceId((ResourceId)this.testPath("testFileB")).setIsReadSeekEfficient(true).setSizeBytes((long)"testDataB".getBytes(StandardCharsets.UTF_8).length).setLastModifiedMillis(this.lastModified("testFileB")).build()))}));
    }

    @Test
    public void testDeleteNonExisting() throws Exception {
        this.fileSystem.delete((Collection)ImmutableList.of((Object)this.testPath("MissingFile")));
    }

    @Test
    public void testMatch() throws Exception {
        this.create("testFileAA", "testDataAA".getBytes(StandardCharsets.UTF_8));
        this.create("testFileA", "testDataA".getBytes(StandardCharsets.UTF_8));
        this.create("testFileB", "testDataB".getBytes(StandardCharsets.UTF_8));
        Assert.assertArrayEquals((byte[])"testDataAA".getBytes(StandardCharsets.UTF_8), (byte[])this.read("testFileAA", 0L));
        Assert.assertArrayEquals((byte[])"testDataA".getBytes(StandardCharsets.UTF_8), (byte[])this.read("testFileA", 0L));
        Assert.assertArrayEquals((byte[])"testDataB".getBytes(StandardCharsets.UTF_8), (byte[])this.read("testFileB", 0L));
        List results = this.fileSystem.match((List)ImmutableList.of((Object)this.testPath("testFileA*").toString()));
        Assert.assertEquals((Object)MatchResult.Status.OK, (Object)((MatchResult)Iterables.getOnlyElement((Iterable)results)).status());
        MatcherAssert.assertThat((Object)((MatchResult)Iterables.getOnlyElement((Iterable)results)).metadata(), (Matcher)Matchers.containsInAnyOrder((Object[])new MatchResult.Metadata[]{MatchResult.Metadata.builder().setResourceId((ResourceId)this.testPath("testFileAA")).setIsReadSeekEfficient(true).setSizeBytes((long)"testDataAA".getBytes(StandardCharsets.UTF_8).length).setLastModifiedMillis(this.lastModified("testFileAA")).build(), MatchResult.Metadata.builder().setResourceId((ResourceId)this.testPath("testFileA")).setIsReadSeekEfficient(true).setSizeBytes((long)"testDataA".getBytes(StandardCharsets.UTF_8).length).setLastModifiedMillis(this.lastModified("testFileA")).build()}));
    }

    @Test
    public void testMatchDirectory() throws Exception {
        this.create("dir/file", "data".getBytes(StandardCharsets.UTF_8));
        MatchResult matchResult = (MatchResult)Iterables.getOnlyElement((Iterable)this.fileSystem.match(Collections.singletonList(this.testPath("dir").toString())));
        MatcherAssert.assertThat((Object)matchResult, (Matcher)Matchers.equalTo((Object)MatchResult.create((MatchResult.Status)MatchResult.Status.OK, (List)ImmutableList.of((Object)MatchResult.Metadata.builder().setResourceId((ResourceId)this.testPath("dir")).setIsReadSeekEfficient(true).setSizeBytes(0L).setLastModifiedMillis(this.lastModified("dir")).build()))));
    }

    @Test
    public void testMatchForNonExistentFile() throws Exception {
        this.create("testFileAA", "testDataAA".getBytes(StandardCharsets.UTF_8));
        this.create("testFileBB", "testDataBB".getBytes(StandardCharsets.UTF_8));
        Assert.assertArrayEquals((byte[])"testDataAA".getBytes(StandardCharsets.UTF_8), (byte[])this.read("testFileAA", 0L));
        Assert.assertArrayEquals((byte[])"testDataBB".getBytes(StandardCharsets.UTF_8), (byte[])this.read("testFileBB", 0L));
        List matchResults = this.fileSystem.match((List)ImmutableList.of((Object)this.testPath("testFileAA").toString(), (Object)this.testPath("testFileA").toString(), (Object)this.testPath("testFileBB").toString()));
        MatcherAssert.assertThat((Object)matchResults, (Matcher)Matchers.hasSize((int)3));
        ImmutableList expected = ImmutableList.of((Object)MatchResult.create((MatchResult.Status)MatchResult.Status.OK, (List)ImmutableList.of((Object)MatchResult.Metadata.builder().setResourceId((ResourceId)this.testPath("testFileAA")).setIsReadSeekEfficient(true).setSizeBytes((long)"testDataAA".getBytes(StandardCharsets.UTF_8).length).setLastModifiedMillis(this.lastModified("testFileAA")).build())), (Object)MatchResult.create((MatchResult.Status)MatchResult.Status.NOT_FOUND, (List)ImmutableList.of()), (Object)MatchResult.create((MatchResult.Status)MatchResult.Status.OK, (List)ImmutableList.of((Object)MatchResult.Metadata.builder().setResourceId((ResourceId)this.testPath("testFileBB")).setIsReadSeekEfficient(true).setSizeBytes((long)"testDataBB".getBytes(StandardCharsets.UTF_8).length).setLastModifiedMillis(this.lastModified("testFileBB")).build())));
        MatcherAssert.assertThat((Object)matchResults, (Matcher)Matchers.equalTo((Object)expected));
    }

    @Test
    public void testMatchForRecursiveGlob() throws Exception {
        this.create("1/testFile1", "testData1".getBytes(StandardCharsets.UTF_8));
        this.create("1/A/testFile1A", "testData1A".getBytes(StandardCharsets.UTF_8));
        this.create("1/A/A/testFile1AA", "testData1AA".getBytes(StandardCharsets.UTF_8));
        this.create("1/B/testFile1B", "testData1B".getBytes(StandardCharsets.UTF_8));
        Assert.assertArrayEquals((byte[])"testData1".getBytes(StandardCharsets.UTF_8), (byte[])this.read("1/testFile1", 0L));
        Assert.assertArrayEquals((byte[])"testData1A".getBytes(StandardCharsets.UTF_8), (byte[])this.read("1/A/testFile1A", 0L));
        Assert.assertArrayEquals((byte[])"testData1AA".getBytes(StandardCharsets.UTF_8), (byte[])this.read("1/A/A/testFile1AA", 0L));
        Assert.assertArrayEquals((byte[])"testData1B".getBytes(StandardCharsets.UTF_8), (byte[])this.read("1/B/testFile1B", 0L));
        List matchResults = this.fileSystem.match((List)ImmutableList.of((Object)this.testPath("**testFile1*").toString()));
        MatcherAssert.assertThat((Object)matchResults, (Matcher)Matchers.hasSize((int)1));
        MatcherAssert.assertThat((Object)((MatchResult)Iterables.getOnlyElement((Iterable)matchResults)).metadata(), (Matcher)Matchers.containsInAnyOrder((Object[])new MatchResult.Metadata[]{MatchResult.Metadata.builder().setResourceId((ResourceId)this.testPath("1/testFile1")).setIsReadSeekEfficient(true).setSizeBytes((long)"testData1".getBytes(StandardCharsets.UTF_8).length).setLastModifiedMillis(this.lastModified("1/testFile1")).build(), MatchResult.Metadata.builder().setResourceId((ResourceId)this.testPath("1/A/testFile1A")).setIsReadSeekEfficient(true).setSizeBytes((long)"testData1A".getBytes(StandardCharsets.UTF_8).length).setLastModifiedMillis(this.lastModified("1/A/testFile1A")).build(), MatchResult.Metadata.builder().setResourceId((ResourceId)this.testPath("1/A/A/testFile1AA")).setIsReadSeekEfficient(true).setSizeBytes((long)"testData1AA".getBytes(StandardCharsets.UTF_8).length).setLastModifiedMillis(this.lastModified("1/A/A/testFile1AA")).build(), MatchResult.Metadata.builder().setResourceId((ResourceId)this.testPath("1/B/testFile1B")).setIsReadSeekEfficient(true).setSizeBytes((long)"testData1B".getBytes(StandardCharsets.UTF_8).length).setLastModifiedMillis(this.lastModified("1/B/testFile1B")).build()}));
        matchResults = this.fileSystem.match((List)ImmutableList.of((Object)this.testPath("1**File1A").toString(), (Object)this.testPath("1**A**testFile1AA").toString(), (Object)this.testPath("1/B**").toString(), (Object)this.testPath("2**").toString()));
        ImmutableList expected = ImmutableList.of((Object)MatchResult.create((MatchResult.Status)MatchResult.Status.OK, (List)ImmutableList.of((Object)MatchResult.Metadata.builder().setResourceId((ResourceId)this.testPath("1/A/testFile1A")).setIsReadSeekEfficient(true).setSizeBytes((long)"testData1A".getBytes(StandardCharsets.UTF_8).length).setLastModifiedMillis(this.lastModified("1/A/testFile1A")).build())), (Object)MatchResult.create((MatchResult.Status)MatchResult.Status.OK, (List)ImmutableList.of((Object)MatchResult.Metadata.builder().setResourceId((ResourceId)this.testPath("1/A/A/testFile1AA")).setIsReadSeekEfficient(true).setSizeBytes((long)"testData1AA".getBytes(StandardCharsets.UTF_8).length).setLastModifiedMillis(this.lastModified("1/A/A/testFile1AA")).build())), (Object)MatchResult.create((MatchResult.Status)MatchResult.Status.OK, (List)ImmutableList.of((Object)MatchResult.Metadata.builder().setResourceId((ResourceId)this.testPath("1/B/testFile1B")).setIsReadSeekEfficient(true).setSizeBytes((long)"testData1B".getBytes(StandardCharsets.UTF_8).length).setLastModifiedMillis(this.lastModified("1/B/testFile1B")).build())), (Object)MatchResult.create((MatchResult.Status)MatchResult.Status.NOT_FOUND, (List)ImmutableList.of()));
        MatcherAssert.assertThat((Object)matchResults, (Matcher)Matchers.hasSize((int)4));
        MatcherAssert.assertThat((Object)matchResults, (Matcher)Matchers.equalTo((Object)expected));
    }

    @Test
    public void testRename() throws Exception {
        this.create("testFileA", "testDataA".getBytes(StandardCharsets.UTF_8));
        this.create("testFileB", "testDataB".getBytes(StandardCharsets.UTF_8));
        Assert.assertArrayEquals((byte[])"testDataA".getBytes(StandardCharsets.UTF_8), (byte[])this.read("testFileA", 0L));
        Assert.assertArrayEquals((byte[])"testDataB".getBytes(StandardCharsets.UTF_8), (byte[])this.read("testFileB", 0L));
        this.fileSystem.rename((List)ImmutableList.of((Object)this.testPath("testFileA"), (Object)this.testPath("testFileB")), (List)ImmutableList.of((Object)this.testPath("renameFileA"), (Object)this.testPath("renameFileB")), new MoveOptions[0]);
        List results = this.fileSystem.match((List)ImmutableList.of((Object)this.testPath("*").toString()));
        Assert.assertEquals((Object)MatchResult.Status.OK, (Object)((MatchResult)Iterables.getOnlyElement((Iterable)results)).status());
        MatcherAssert.assertThat((Object)((MatchResult)Iterables.getOnlyElement((Iterable)results)).metadata(), (Matcher)Matchers.containsInAnyOrder((Object[])new MatchResult.Metadata[]{MatchResult.Metadata.builder().setResourceId((ResourceId)this.testPath("renameFileA")).setIsReadSeekEfficient(true).setSizeBytes((long)"testDataA".getBytes(StandardCharsets.UTF_8).length).setLastModifiedMillis(this.lastModified("renameFileA")).build(), MatchResult.Metadata.builder().setResourceId((ResourceId)this.testPath("renameFileB")).setIsReadSeekEfficient(true).setSizeBytes((long)"testDataB".getBytes(StandardCharsets.UTF_8).length).setLastModifiedMillis(this.lastModified("renameFileB")).build()}));
        Assert.assertArrayEquals((byte[])"testDataA".getBytes(StandardCharsets.UTF_8), (byte[])this.read("renameFileA", 0L));
        Assert.assertArrayEquals((byte[])"testDataB".getBytes(StandardCharsets.UTF_8), (byte[])this.read("renameFileB", 0L));
    }

    @Test
    public void testRenameMissingTargetDir() throws Exception {
        this.create("pathA/testFileA", "testDataA".getBytes(StandardCharsets.UTF_8));
        this.create("pathA/testFileB", "testDataB".getBytes(StandardCharsets.UTF_8));
        Assert.assertArrayEquals((byte[])"testDataA".getBytes(StandardCharsets.UTF_8), (byte[])this.read("pathA/testFileA", 0L));
        Assert.assertArrayEquals((byte[])"testDataB".getBytes(StandardCharsets.UTF_8), (byte[])this.read("pathA/testFileB", 0L));
        this.fileSystem.rename((List)ImmutableList.of((Object)this.testPath("pathA/testFileA"), (Object)this.testPath("pathA/testFileB")), (List)ImmutableList.of((Object)this.testPath("pathB/testFileA"), (Object)this.testPath("pathB/pathC/pathD/testFileB")), new MoveOptions[0]);
        this.expectedLogs.verifyDebug(MessageFormatter.format((String)"Creating directory {}", (Object)"/pathB").getMessage());
        this.expectedLogs.verifyDebug(MessageFormatter.format((String)"Creating directory {}", (Object)"/pathB/pathC/pathD").getMessage());
        Assert.assertArrayEquals((byte[])"testDataA".getBytes(StandardCharsets.UTF_8), (byte[])this.read("pathB/testFileA", 0L));
        Assert.assertArrayEquals((byte[])"testDataB".getBytes(StandardCharsets.UTF_8), (byte[])this.read("pathB/pathC/pathD/testFileB", 0L));
    }

    @Test(expected=FileNotFoundException.class)
    public void testRenameMissingSource() throws Exception {
        this.fileSystem.rename((List)ImmutableList.of((Object)this.testPath("missingFile")), (List)ImmutableList.of((Object)this.testPath("testFileA")), new MoveOptions[0]);
    }

    @Test
    public void testRenameExistingDestination() throws Exception {
        this.create("testFileA", "testDataA".getBytes(StandardCharsets.UTF_8));
        this.create("testFileB", "testDataB".getBytes(StandardCharsets.UTF_8));
        Assert.assertArrayEquals((byte[])"testDataA".getBytes(StandardCharsets.UTF_8), (byte[])this.read("testFileA", 0L));
        Assert.assertArrayEquals((byte[])"testDataB".getBytes(StandardCharsets.UTF_8), (byte[])this.read("testFileB", 0L));
        this.fileSystem.rename((List)ImmutableList.of((Object)this.testPath("testFileA")), (List)ImmutableList.of((Object)this.testPath("testFileB")), new MoveOptions[0]);
        this.expectedLogs.verifyDebug(MessageFormatter.format((String)"Deleting existing file {}", (Object)"/testFileB").getMessage());
        Assert.assertArrayEquals((byte[])"testDataA".getBytes(StandardCharsets.UTF_8), (byte[])this.read("testFileB", 0L));
    }

    @Test(expected=FileNotFoundException.class)
    public void testRenameRetryScenario() throws Exception {
        this.testRename();
        this.fileSystem.rename((List)ImmutableList.of((Object)this.testPath("testFileA"), (Object)this.testPath("testFileB")), (List)ImmutableList.of((Object)this.testPath("renameFileA"), (Object)this.testPath("renameFileB")), new MoveOptions[0]);
    }

    @Test
    public void testMatchNewResource() {
        Assert.assertEquals((Object)this.testPath("file"), (Object)this.fileSystem.matchNewResource(this.testPath("file").toString(), false));
        Assert.assertEquals((Object)this.testPath("dir/"), (Object)this.fileSystem.matchNewResource(this.testPath("dir").toString(), true));
        Assert.assertEquals((Object)this.testPath("dir/"), (Object)this.fileSystem.matchNewResource(this.testPath("dir/").toString(), true));
        this.thrown.expect(IllegalArgumentException.class);
        this.thrown.expectMessage("Expected file path but received directory path");
        this.fileSystem.matchNewResource(this.testPath("dir/").toString(), false);
    }

    @Test
    @Ignore(value="TestPipeline needs a way to take in HadoopFileSystemOptions")
    public void testReadPipeline() throws Exception {
        this.create("testFileA", "testDataA".getBytes(StandardCharsets.UTF_8));
        this.create("testFileB", "testDataB".getBytes(StandardCharsets.UTF_8));
        this.create("testFileC", "testDataC".getBytes(StandardCharsets.UTF_8));
        HadoopFileSystemOptions options = (HadoopFileSystemOptions)TestPipeline.testingPipelineOptions().as(HadoopFileSystemOptions.class);
        options.setHdfsConfiguration((List)ImmutableList.of((Object)this.fileSystem.configuration));
        FileSystems.setDefaultPipelineOptions((PipelineOptions)options);
        PCollection pc = (PCollection)this.p.apply((PTransform)TextIO.read().from(this.testPath("testFile*").toString()));
        PAssert.that((PCollection)pc).containsInAnyOrder((Object[])new String[]{"testDataA", "testDataB", "testDataC"});
        this.p.run();
    }

    private void create(String relativePath, byte[] contents) throws Exception {
        try (WritableByteChannel channel = this.fileSystem.create(this.testPath(relativePath), (CreateOptions)((CreateOptions.StandardCreateOptions.Builder)CreateOptions.StandardCreateOptions.builder().setMimeType("application/octet-stream")).build());){
            channel.write(ByteBuffer.wrap(contents));
        }
    }

    private byte[] read(String relativePath, long bytesToSkip) throws Exception {
        try (ReadableByteChannel channel = this.fileSystem.open(this.testPath(relativePath));){
            InputStream inputStream = Channels.newInputStream(channel);
            if (bytesToSkip > 0L) {
                ByteStreams.skipFully((InputStream)inputStream, (long)bytesToSkip);
            }
            byte[] byArray = ByteStreams.toByteArray((InputStream)inputStream);
            return byArray;
        }
    }

    private long lastModified(String relativePath) throws Exception {
        Path testPath = this.testPath(relativePath).toPath();
        return testPath.getFileSystem(this.fileSystem.configuration).getFileStatus(this.testPath(relativePath).toPath()).getModificationTime();
    }

    private HadoopResourceId testPath(String relativePath) {
        return new HadoopResourceId(this.hdfsClusterBaseUri.resolve(relativePath));
    }
}

