/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.beam.sdk.extensions.sql.impl.interpreter.operator.math;

import java.util.List;
import java.util.Random;
import org.apache.beam.sdk.extensions.sql.impl.interpreter.operator.BeamSqlExpression;
import org.apache.beam.sdk.extensions.sql.impl.interpreter.operator.BeamSqlPrimitive;
import org.apache.beam.sdk.transforms.windowing.BoundedWindow;
import org.apache.beam.sdk.values.BeamRecord;
import org.apache.calcite.sql.type.SqlTypeName;

/**
 * {@code BeamSqlMathUnaryExpression} for 'RAND_INTEGER([seed, ] numeric)'
 * function.
 */
public class BeamSqlRandIntegerExpression extends BeamSqlExpression {
  private Random rand = new Random();
  private Integer seed = null;

  public BeamSqlRandIntegerExpression(List<BeamSqlExpression> subExps) {
    super(subExps, SqlTypeName.INTEGER);
  }

  @Override
  public boolean accept() {
    return true;
  }

  @Override
  public BeamSqlPrimitive evaluate(BeamRecord inputRecord, BoundedWindow window) {
    int numericIdx = 0;
    if (operands.size() == 2) {
      int rowSeed = opValueEvaluated(0, inputRecord, window);
      if (seed == null || seed != rowSeed) {
        rand.setSeed(rowSeed);
      }
      numericIdx = 1;
    }
    return BeamSqlPrimitive.of(SqlTypeName.INTEGER,
        rand.nextInt((int) opValueEvaluated(numericIdx, inputRecord, window)));
  }
}
