/*
 * Decompiled with CFR 0.152.
 */
package org.apache.beam.sdk.util;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Random;
import org.apache.beam.repackaged.beam_sdks_java_core.com.google.common.base.Charsets;
import org.apache.beam.repackaged.beam_sdks_java_core.com.google.common.collect.ImmutableList;
import org.apache.beam.sdk.coders.ByteArrayCoder;
import org.apache.beam.sdk.util.BufferedElementCountingOutputStream;
import org.apache.beam.sdk.util.VarInt;
import org.hamcrest.Matcher;
import org.hamcrest.collection.IsIterableContainingInOrder;
import org.junit.Assert;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;
import org.junit.runner.RunWith;
import org.junit.runners.JUnit4;

@RunWith(value=JUnit4.class)
public class BufferedElementCountingOutputStreamTest {
    @Rule
    public final ExpectedException expectedException = ExpectedException.none();
    private static final int BUFFER_SIZE = 8;

    @Test
    public void testEmptyValues() throws Exception {
        this.testValues(Collections.emptyList());
    }

    @Test
    public void testSingleValue() throws Exception {
        this.testValues(this.toBytes("abc"));
    }

    @Test
    public void testSingleValueGreaterThanBuffer() throws Exception {
        this.testValues(this.toBytes("abcdefghijklmnopqrstuvwxyz"));
    }

    @Test
    public void testMultipleValuesLessThanBuffer() throws Exception {
        this.testValues(this.toBytes("a", "b", "c"));
    }

    @Test
    public void testMultipleValuesThatBecomeGreaterThanBuffer() throws Exception {
        this.testValues(this.toBytes("a", "b", "c", "d", "e", "f", "g", "h", "i", "j", "k", "l", "m", "n", "o", "p", "q", "r", "s", "t", "u", "v", "w", "x", "y", "z"));
    }

    @Test
    public void testMultipleRandomSizedValues() throws Exception {
        Random r = new Random(234589023580234890L);
        byte[] randomData = new byte[r.nextInt(18)];
        for (int i = 0; i < 1000; ++i) {
            ArrayList<byte[]> bytes = new ArrayList<byte[]>();
            for (int j = 0; j < 100; ++j) {
                r.nextBytes(randomData);
                bytes.add(randomData);
            }
            this.testValues(bytes);
        }
    }

    @Test
    public void testFlushInMiddleOfElement() throws Exception {
        ByteArrayOutputStream bos = new ByteArrayOutputStream();
        BufferedElementCountingOutputStream os = new BufferedElementCountingOutputStream((OutputStream)bos);
        os.markElementStart();
        os.write(1);
        os.flush();
        os.write(2);
        os.close();
        Assert.assertArrayEquals((byte[])new byte[]{1, 1, 2, 0}, (byte[])bos.toByteArray());
    }

    @Test
    public void testFlushInMiddleOfElementUsingByteArrays() throws Exception {
        ByteArrayOutputStream bos = new ByteArrayOutputStream();
        BufferedElementCountingOutputStream os = new BufferedElementCountingOutputStream((OutputStream)bos);
        os.markElementStart();
        os.write(new byte[]{1});
        os.flush();
        os.write(new byte[]{2});
        os.close();
        Assert.assertArrayEquals((byte[])new byte[]{1, 1, 2, 0}, (byte[])bos.toByteArray());
    }

    @Test
    public void testFlushingWhenFinishedIsNoOp() throws Exception {
        BufferedElementCountingOutputStream os = this.testValues(this.toBytes("a"));
        os.flush();
        os.flush();
        os.flush();
    }

    @Test
    public void testFinishingWhenFinishedIsNoOp() throws Exception {
        BufferedElementCountingOutputStream os = this.testValues(this.toBytes("a"));
        os.finish();
        os.finish();
        os.finish();
    }

    @Test
    public void testClosingFinishesTheStream() throws Exception {
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        BufferedElementCountingOutputStream os = this.createAndWriteValues(this.toBytes("abcdefghij"), baos);
        os.close();
        this.verifyValues(this.toBytes("abcdefghij"), new ByteArrayInputStream(baos.toByteArray()));
    }

    @Test
    public void testAddingElementWhenFinishedThrows() throws Exception {
        this.expectedException.expect(IOException.class);
        this.expectedException.expectMessage("Stream has been finished.");
        this.testValues(this.toBytes("a")).markElementStart();
    }

    @Test
    public void testWritingByteWhenFinishedThrows() throws Exception {
        this.expectedException.expect(IOException.class);
        this.expectedException.expectMessage("Stream has been finished.");
        this.testValues(this.toBytes("a")).write(1);
    }

    @Test
    public void testWritingBytesWhenFinishedThrows() throws Exception {
        this.expectedException.expect(IOException.class);
        this.expectedException.expectMessage("Stream has been finished.");
        this.testValues(this.toBytes("a")).write("b".getBytes(Charsets.UTF_8));
    }

    @Test
    public void testBuffersAreTakenAndReturned() throws Exception {
        BufferedElementCountingOutputStream.BUFFER_POOL.clear();
        BufferedElementCountingOutputStream.BUFFER_POOL.offer(ByteBuffer.allocate(256));
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        BufferedElementCountingOutputStream os = this.createAndWriteValues(this.toBytes("abcdefghij"), baos);
        Assert.assertEquals((long)0L, (long)BufferedElementCountingOutputStream.BUFFER_POOL.size());
        os.finish();
        Assert.assertEquals((long)1L, (long)BufferedElementCountingOutputStream.BUFFER_POOL.size());
    }

    @Test
    public void testBehaviorWhenBufferPoolFull() throws Exception {
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        while (BufferedElementCountingOutputStream.BUFFER_POOL.remainingCapacity() > 0) {
            BufferedElementCountingOutputStream.BUFFER_POOL.offer(ByteBuffer.allocate(256));
        }
        BufferedElementCountingOutputStream os = this.createAndWriteValues(this.toBytes("abcdefghij"), baos);
        os.finish();
        Assert.assertEquals((long)0L, (long)BufferedElementCountingOutputStream.BUFFER_POOL.remainingCapacity());
    }

    @Test
    public void testBehaviorWhenBufferPoolEmpty() throws Exception {
        BufferedElementCountingOutputStream.BUFFER_POOL.clear();
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        BufferedElementCountingOutputStream os = this.createAndWriteValues(this.toBytes("abcdefghij"), baos);
        Assert.assertEquals((long)0L, (long)BufferedElementCountingOutputStream.BUFFER_POOL.size());
        os.finish();
        Assert.assertEquals((long)1L, (long)BufferedElementCountingOutputStream.BUFFER_POOL.size());
    }

    private List<byte[]> toBytes(String ... values) {
        ImmutableList.Builder builder = ImmutableList.builder();
        for (String value : values) {
            builder.add((Object)value.getBytes(Charsets.UTF_8));
        }
        return builder.build();
    }

    private BufferedElementCountingOutputStream testValues(List<byte[]> expectedValues) throws Exception {
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        BufferedElementCountingOutputStream os = this.createAndWriteValues(expectedValues, baos);
        os.finish();
        this.verifyValues(expectedValues, new ByteArrayInputStream(baos.toByteArray()));
        return os;
    }

    private void verifyValues(List<byte[]> expectedValues, InputStream is) throws Exception {
        long count;
        ArrayList<byte[]> values = new ArrayList<byte[]>();
        do {
            count = VarInt.decodeLong((InputStream)is);
            int i = 0;
            while ((long)i < count) {
                values.add(ByteArrayCoder.of().decode(is));
                ++i;
            }
        } while (count > 0L);
        if (expectedValues.isEmpty()) {
            Assert.assertTrue((boolean)values.isEmpty());
        } else {
            Assert.assertThat(values, (Matcher)IsIterableContainingInOrder.contains((Object[])expectedValues.toArray()));
        }
    }

    private BufferedElementCountingOutputStream createAndWriteValues(List<byte[]> values, OutputStream output) throws Exception {
        BufferedElementCountingOutputStream os = new BufferedElementCountingOutputStream(output, 8);
        for (byte[] value : values) {
            os.markElementStart();
            ByteArrayCoder.of().encode(value, (OutputStream)os);
        }
        return os;
    }
}

