/*
 * Decompiled with CFR 0.152.
 */
package org.apache.beam.sdk.io;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.Reader;
import java.io.Writer;
import java.nio.channels.Channels;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.List;
import org.apache.beam.repackaged.beam_sdks_java_core.com.google.common.collect.FluentIterable;
import org.apache.beam.repackaged.beam_sdks_java_core.com.google.common.collect.ImmutableList;
import org.apache.beam.repackaged.beam_sdks_java_core.com.google.common.collect.Lists;
import org.apache.beam.repackaged.beam_sdks_java_core.com.google.common.io.Files;
import org.apache.beam.repackaged.beam_sdks_java_core.com.google.common.io.LineReader;
import org.apache.beam.repackaged.beam_sdks_java_core.org.apache.commons.lang3.SystemUtils;
import org.apache.beam.sdk.io.LocalFileSystem;
import org.apache.beam.sdk.io.LocalResourceId;
import org.apache.beam.sdk.io.fs.CreateOptions;
import org.apache.beam.sdk.io.fs.MatchResult;
import org.apache.beam.sdk.io.fs.ResolveOptions;
import org.apache.beam.sdk.testing.RestoreSystemProperties;
import org.hamcrest.Matcher;
import org.hamcrest.Matchers;
import org.junit.Assert;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;
import org.junit.rules.TemporaryFolder;
import org.junit.runner.RunWith;
import org.junit.runners.JUnit4;

@RunWith(value=JUnit4.class)
public class LocalFileSystemTest {
    @Rule
    public ExpectedException thrown = ExpectedException.none();
    @Rule
    public TemporaryFolder temporaryFolder = new TemporaryFolder();
    @Rule
    public RestoreSystemProperties restoreSystemProperties = new RestoreSystemProperties();
    private LocalFileSystem localFileSystem = new LocalFileSystem();

    @Test
    public void testCreateWithExistingFile() throws Exception {
        File existingFile = this.temporaryFolder.newFile();
        this.testCreate(existingFile.toPath());
    }

    @Test
    public void testCreateWithinExistingDirectory() throws Exception {
        this.testCreate(this.temporaryFolder.getRoot().toPath().resolve("file.txt"));
    }

    @Test
    public void testCreateWithNonExistentSubDirectory() throws Exception {
        this.testCreate(this.temporaryFolder.getRoot().toPath().resolve("non-existent-dir").resolve("file.txt"));
    }

    private void testCreate(Path path) throws Exception {
        String expected = "my test string";
        this.createFileWithContent(path, expected);
        Assert.assertThat(Files.readLines(path.toFile(), StandardCharsets.UTF_8), (Matcher)Matchers.containsInAnyOrder((Object[])new String[]{expected}));
        Assert.assertTrue((String)("Unable to delete file " + path), (boolean)path.toFile().delete());
        this.createFileWithContent(Paths.get(path.toUri()), expected);
        Assert.assertThat(Files.readLines(path.toFile(), StandardCharsets.UTF_8), (Matcher)Matchers.containsInAnyOrder((Object[])new String[]{expected}));
    }

    @Test
    public void testReadWithExistingFile() throws Exception {
        String data;
        String expected = "my test string";
        File existingFile = this.temporaryFolder.newFile();
        Files.write(expected, existingFile, StandardCharsets.UTF_8);
        try (Reader reader = Channels.newReader(this.localFileSystem.open(LocalResourceId.fromPath((Path)existingFile.toPath(), (boolean)false)), StandardCharsets.UTF_8.name());){
            data = new LineReader(reader).readLine();
        }
        Assert.assertEquals((Object)expected, (Object)data);
    }

    @Test
    public void testReadNonExistentFile() throws Exception {
        this.thrown.expect(FileNotFoundException.class);
        this.localFileSystem.open(LocalResourceId.fromPath((Path)this.temporaryFolder.getRoot().toPath().resolve("non-existent-file.txt"), (boolean)false)).close();
    }

    private void assertContents(List<Path> destFiles, List<String> contents) throws Exception {
        for (int i = 0; i < destFiles.size(); ++i) {
            Assert.assertThat(Files.readLines(destFiles.get(i).toFile(), StandardCharsets.UTF_8), (Matcher)Matchers.containsInAnyOrder((Object[])new String[]{contents.get(i)}));
        }
    }

    @Test
    public void testCopyWithExistingSrcFile() throws Exception {
        Path srcPath1 = this.temporaryFolder.newFile().toPath();
        Path srcPath2 = this.temporaryFolder.newFile().toPath();
        Path destPath1 = this.temporaryFolder.getRoot().toPath().resolve("nonexistentdir").resolve("dest1");
        Path destPath2 = srcPath2.resolveSibling("dest2");
        this.createFileWithContent(srcPath1, "content1");
        this.createFileWithContent(srcPath2, "content2");
        this.localFileSystem.copy(this.toLocalResourceIds(ImmutableList.of(srcPath1, srcPath2), false), this.toLocalResourceIds(ImmutableList.of(destPath1, destPath2), false));
        this.assertContents(ImmutableList.of(destPath1, destPath2), ImmutableList.of("content1", "content2"));
    }

    @Test
    public void testMoveWithExistingSrcFile() throws Exception {
        Path srcPath1 = this.temporaryFolder.newFile().toPath();
        Path srcPath2 = this.temporaryFolder.newFile().toPath();
        Path destPath1 = this.temporaryFolder.getRoot().toPath().resolve("nonexistentdir").resolve("dest1");
        Path destPath2 = srcPath2.resolveSibling("dest2");
        this.createFileWithContent(srcPath1, "content1");
        this.createFileWithContent(srcPath2, "content2");
        this.localFileSystem.rename(this.toLocalResourceIds(ImmutableList.of(srcPath1, srcPath2), false), this.toLocalResourceIds(ImmutableList.of(destPath1, destPath2), false));
        this.assertContents(ImmutableList.of(destPath1, destPath2), ImmutableList.of("content1", "content2"));
        Assert.assertFalse((String)(srcPath1 + "exists"), (boolean)srcPath1.toFile().exists());
        Assert.assertFalse((String)(srcPath2 + "exists"), (boolean)srcPath2.toFile().exists());
    }

    @Test
    public void testDelete() throws Exception {
        File f1 = this.temporaryFolder.newFile("file1");
        File f2 = this.temporaryFolder.newFile("file2");
        File f3 = this.temporaryFolder.newFile("other-file");
        this.localFileSystem.delete(this.toLocalResourceIds(Lists.newArrayList(f1.toPath(), f2.toPath()), false));
        Assert.assertFalse((boolean)f1.exists());
        Assert.assertFalse((boolean)f2.exists());
        Assert.assertTrue((boolean)f3.exists());
    }

    @Test
    public void testMatchExact() throws Exception {
        ImmutableList<String> expected = ImmutableList.of(this.temporaryFolder.newFile("a").toString());
        this.temporaryFolder.newFile("aa");
        this.temporaryFolder.newFile("ab");
        List matchResults = this.localFileSystem.match(ImmutableList.of(this.temporaryFolder.getRoot().toPath().resolve("a").toString()));
        Assert.assertThat(this.toFilenames(matchResults), (Matcher)Matchers.containsInAnyOrder((Object[])expected.toArray(new String[expected.size()])));
    }

    @Test
    public void testMatchPatternNone() throws Exception {
        this.temporaryFolder.newFile("a");
        this.temporaryFolder.newFile("aa");
        this.temporaryFolder.newFile("ab");
        List<MatchResult> matchResults = this.matchGlobWithPathPrefix(this.temporaryFolder.getRoot().toPath().resolve("b"), "*");
        Assert.assertEquals((long)1L, (long)matchResults.size());
        Assert.assertEquals((Object)MatchResult.Status.NOT_FOUND, (Object)matchResults.get(0).status());
    }

    @Test
    public void testMatchForNonExistentFile() throws Exception {
        this.temporaryFolder.newFile("aa");
        List matchResults = this.localFileSystem.match(ImmutableList.of(this.temporaryFolder.getRoot().toPath().resolve("a").toString()));
        Assert.assertEquals((long)1L, (long)matchResults.size());
        Assert.assertEquals((Object)MatchResult.Status.NOT_FOUND, (Object)((MatchResult)matchResults.get(0)).status());
    }

    @Test
    public void testMatchMultipleWithFileExtension() throws Exception {
        ImmutableList<String> expected = ImmutableList.of(this.temporaryFolder.newFile("a.txt").toString(), this.temporaryFolder.newFile("aa.txt").toString(), this.temporaryFolder.newFile("ab.txt").toString());
        this.temporaryFolder.newFile("a.avro");
        this.temporaryFolder.newFile("ab.avro");
        List<MatchResult> matchResults = this.matchGlobWithPathPrefix(this.temporaryFolder.getRoot().toPath().resolve("a"), "*.txt");
        Assert.assertThat(this.toFilenames(matchResults), (Matcher)Matchers.containsInAnyOrder((Object[])expected.toArray(new String[expected.size()])));
    }

    @Test
    public void testMatchInDirectory() throws Exception {
        ImmutableList<String> expected = ImmutableList.of(this.temporaryFolder.newFile("a").toString());
        this.temporaryFolder.newFile("aa");
        this.temporaryFolder.newFile("ab");
        String expectedFile = (String)expected.get(0);
        int slashIndex = expectedFile.lastIndexOf(47);
        if (SystemUtils.IS_OS_WINDOWS) {
            slashIndex = expectedFile.lastIndexOf(92);
        }
        String directory = expectedFile.substring(0, slashIndex);
        String relative = expectedFile.substring(slashIndex + 1);
        System.setProperty("user.dir", directory);
        List results = this.localFileSystem.match(ImmutableList.of(relative));
        Assert.assertThat(this.toFilenames(results), (Matcher)Matchers.containsInAnyOrder((Object[])expected.toArray(new String[expected.size()])));
    }

    @Test
    public void testMatchWithFileSlashPrefix() throws Exception {
        ImmutableList<String> expected = ImmutableList.of(this.temporaryFolder.newFile("a").toString());
        this.temporaryFolder.newFile("aa");
        this.temporaryFolder.newFile("ab");
        String file = "file:/" + this.temporaryFolder.getRoot().toPath().resolve("a").toString();
        List results = this.localFileSystem.match(ImmutableList.of(file));
        Assert.assertThat(this.toFilenames(results), (Matcher)Matchers.containsInAnyOrder((Object[])expected.toArray(new String[expected.size()])));
    }

    @Test
    public void testMatchWithFileThreeSlashesPrefix() throws Exception {
        ImmutableList<String> expected = ImmutableList.of(this.temporaryFolder.newFile("a").toString());
        this.temporaryFolder.newFile("aa");
        this.temporaryFolder.newFile("ab");
        String file = "file:///" + this.temporaryFolder.getRoot().toPath().resolve("a").toString();
        List results = this.localFileSystem.match(ImmutableList.of(file));
        Assert.assertThat(this.toFilenames(results), (Matcher)Matchers.containsInAnyOrder((Object[])expected.toArray(new String[expected.size()])));
    }

    @Test
    public void testMatchMultipleWithoutSubdirectoryExpansion() throws Exception {
        File unmatchedSubDir = this.temporaryFolder.newFolder("aaa");
        File unmatchedSubDirFile = File.createTempFile("sub-dir-file", "", unmatchedSubDir);
        unmatchedSubDirFile.deleteOnExit();
        ImmutableList<String> expected = ImmutableList.of(this.temporaryFolder.newFile("a").toString(), this.temporaryFolder.newFile("aa").toString(), this.temporaryFolder.newFile("ab").toString());
        this.temporaryFolder.newFile("ba");
        this.temporaryFolder.newFile("bb");
        List<MatchResult> matchResults = this.matchGlobWithPathPrefix(this.temporaryFolder.getRoot().toPath().resolve("a"), "*");
        Assert.assertThat(this.toFilenames(matchResults), (Matcher)Matchers.containsInAnyOrder((Object[])expected.toArray(new String[expected.size()])));
    }

    @Test
    public void testMatchMultipleWithSubdirectoryExpansion() throws Exception {
        File matchedSubDir = this.temporaryFolder.newFolder("a");
        File matchedSubDirFile = File.createTempFile("sub-dir-file", "", matchedSubDir);
        matchedSubDirFile.deleteOnExit();
        File unmatchedSubDir = this.temporaryFolder.newFolder("b");
        File unmatchedSubDirFile = File.createTempFile("sub-dir-file", "", unmatchedSubDir);
        unmatchedSubDirFile.deleteOnExit();
        ImmutableList<String> expected = ImmutableList.of(matchedSubDirFile.toString(), this.temporaryFolder.newFile("aa").toString(), this.temporaryFolder.newFile("ab").toString());
        this.temporaryFolder.newFile("ba");
        this.temporaryFolder.newFile("bb");
        List<MatchResult> matchResults = this.matchGlobWithPathPrefix(this.temporaryFolder.getRoot().toPath().resolve("a"), "**");
        Assert.assertThat(this.toFilenames(matchResults), (Matcher)Matchers.hasItems((Object[])expected.toArray(new String[expected.size()])));
    }

    @Test
    public void testMatchWithDirectoryFiltersOutDirectory() throws Exception {
        ImmutableList<String> expected = ImmutableList.of(this.temporaryFolder.newFile("a").toString());
        this.temporaryFolder.newFolder("a_dir_that_should_not_be_matched");
        List<MatchResult> matchResults = this.matchGlobWithPathPrefix(this.temporaryFolder.getRoot().toPath().resolve("a"), "*");
        Assert.assertThat(this.toFilenames(matchResults), (Matcher)Matchers.containsInAnyOrder((Object[])expected.toArray(new String[expected.size()])));
    }

    @Test
    public void testMatchWithoutParentDirectory() throws Exception {
        Path pattern = LocalResourceId.fromPath((Path)this.temporaryFolder.getRoot().toPath(), (boolean)true).resolve("non_existing_dir", (ResolveOptions)ResolveOptions.StandardResolveOptions.RESOLVE_DIRECTORY).resolve("*", (ResolveOptions)ResolveOptions.StandardResolveOptions.RESOLVE_FILE).getPath();
        Assert.assertTrue((boolean)this.toFilenames(this.localFileSystem.match(ImmutableList.of(pattern.toString()))).isEmpty());
    }

    @Test
    public void testMatchNewResource() throws Exception {
        LocalResourceId fileResource = this.localFileSystem.matchNewResource("/some/test/resource/path", false);
        LocalResourceId dirResource = this.localFileSystem.matchNewResource("/some/test/resource/path", true);
        Assert.assertNotEquals((Object)fileResource, (Object)dirResource);
        Assert.assertThat((Object)fileResource.getCurrentDirectory().resolve("path", (ResolveOptions)ResolveOptions.StandardResolveOptions.RESOLVE_DIRECTORY), (Matcher)Matchers.equalTo((Object)dirResource.getCurrentDirectory()));
        Assert.assertThat((Object)fileResource.getCurrentDirectory().resolve("path", (ResolveOptions)ResolveOptions.StandardResolveOptions.RESOLVE_DIRECTORY), (Matcher)Matchers.equalTo((Object)dirResource.getCurrentDirectory()));
        Assert.assertThat((Object)dirResource.toString(), (Matcher)Matchers.equalTo((Object)"/some/test/resource/path/"));
    }

    private void createFileWithContent(Path path, String content) throws Exception {
        try (Writer writer = Channels.newWriter(this.localFileSystem.create(LocalResourceId.fromPath((Path)path, (boolean)false), (CreateOptions)((CreateOptions.StandardCreateOptions.Builder)CreateOptions.StandardCreateOptions.builder().setMimeType("text/plain")).build()), StandardCharsets.UTF_8.name());){
            writer.write(content);
        }
    }

    private List<MatchResult> matchGlobWithPathPrefix(Path pathPrefix, String glob) throws IOException {
        return this.localFileSystem.match(ImmutableList.of(pathPrefix + glob));
    }

    private List<LocalResourceId> toLocalResourceIds(List<Path> paths, boolean isDirectory) {
        return FluentIterable.from(paths).transform(path -> LocalResourceId.fromPath((Path)path, (boolean)isDirectory)).toList();
    }

    private List<String> toFilenames(List<MatchResult> matchResults) {
        return FluentIterable.from(matchResults).transformAndConcat(matchResult -> {
            try {
                return matchResult.metadata();
            }
            catch (IOException e) {
                throw new RuntimeException(e);
            }
        }).transform(metadata -> ((LocalResourceId)metadata.resourceId()).getPath().toString()).toList();
    }
}

