/*
 * Decompiled with CFR 0.152.
 */
package org.apache.beam.sdk.io;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.nio.channels.ReadableByteChannel;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.Random;
import java.util.zip.GZIPOutputStream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;
import javax.annotation.Nullable;
import org.apache.beam.repackaged.beam_sdks_java_core.com.google.common.collect.HashMultiset;
import org.apache.beam.repackaged.beam_sdks_java_core.com.google.common.collect.Lists;
import org.apache.beam.repackaged.beam_sdks_java_core.com.google.common.collect.Sets;
import org.apache.beam.repackaged.beam_sdks_java_core.com.google.common.io.Files;
import org.apache.beam.repackaged.beam_sdks_java_core.com.google.common.primitives.Bytes;
import org.apache.beam.repackaged.beam_sdks_java_core.org.apache.commons.compress.compressors.bzip2.BZip2CompressorOutputStream;
import org.apache.beam.repackaged.beam_sdks_java_core.org.apache.commons.compress.compressors.deflate.DeflateCompressorOutputStream;
import org.apache.beam.repackaged.beam_sdks_java_core.org.apache.commons.compress.compressors.gzip.GzipCompressorOutputStream;
import org.apache.beam.sdk.coders.Coder;
import org.apache.beam.sdk.coders.SerializableCoder;
import org.apache.beam.sdk.io.BoundedSource;
import org.apache.beam.sdk.io.CompressedSource;
import org.apache.beam.sdk.io.FileBasedSource;
import org.apache.beam.sdk.io.fs.MatchResult;
import org.apache.beam.sdk.options.PipelineOptions;
import org.apache.beam.sdk.options.PipelineOptionsFactory;
import org.apache.beam.sdk.options.ValueProvider;
import org.apache.beam.sdk.testing.SourceTestUtils;
import org.apache.beam.sdk.transforms.DoFn;
import org.apache.beam.sdk.transforms.display.DisplayData;
import org.apache.beam.sdk.transforms.display.DisplayDataMatchers;
import org.apache.beam.sdk.transforms.display.HasDisplayData;
import org.apache.beam.sdk.values.KV;
import org.hamcrest.Matcher;
import org.hamcrest.Matchers;
import org.joda.time.Instant;
import org.junit.Assert;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;
import org.junit.rules.TemporaryFolder;
import org.junit.runner.RunWith;
import org.junit.runners.JUnit4;

@RunWith(value=JUnit4.class)
public class CompressedSourceTest {
    @Rule
    public TemporaryFolder tmpFolder = new TemporaryFolder();
    @Rule
    public ExpectedException thrown = ExpectedException.none();

    @Test
    public void testReadGzip() throws Exception {
        byte[] input = this.generateInput(5000);
        this.runReadTest(input, CompressedSource.CompressionMode.GZIP);
    }

    @Test
    public void testAutoSplittable() throws Exception {
        CompressedSource source = CompressedSource.from((FileBasedSource)new ByteSource("input.gz", 1L));
        Assert.assertFalse((boolean)source.isSplittable());
        source = CompressedSource.from((FileBasedSource)new ByteSource("input.GZ", 1L));
        Assert.assertFalse((boolean)source.isSplittable());
        source = CompressedSource.from((FileBasedSource)new ByteSource("input.bz2", 1L));
        Assert.assertFalse((boolean)source.isSplittable());
        source = CompressedSource.from((FileBasedSource)new ByteSource("input.BZ2", 1L));
        Assert.assertFalse((boolean)source.isSplittable());
        source = CompressedSource.from((FileBasedSource)new ByteSource("input.zip", 1L));
        Assert.assertFalse((boolean)source.isSplittable());
        source = CompressedSource.from((FileBasedSource)new ByteSource("input.ZIP", 1L));
        Assert.assertFalse((boolean)source.isSplittable());
        source = CompressedSource.from((FileBasedSource)new ByteSource("input.deflate", 1L));
        Assert.assertFalse((boolean)source.isSplittable());
        source = CompressedSource.from((FileBasedSource)new ByteSource("input.DEFLATE", 1L));
        Assert.assertFalse((boolean)source.isSplittable());
        source = CompressedSource.from((FileBasedSource)new ByteSource("input.txt", 1L));
        Assert.assertTrue((boolean)source.isSplittable());
        source = CompressedSource.from((FileBasedSource)new ByteSource("input.csv", 1L));
        Assert.assertTrue((boolean)source.isSplittable());
    }

    @Test
    public void testGzipSplittable() throws Exception {
        CompressedSource source = CompressedSource.from((FileBasedSource)new ByteSource("input.gz", 1L)).withDecompression((CompressedSource.DecompressingChannelFactory)CompressedSource.CompressionMode.GZIP);
        Assert.assertFalse((boolean)source.isSplittable());
        source = CompressedSource.from((FileBasedSource)new ByteSource("input.GZ", 1L)).withDecompression((CompressedSource.DecompressingChannelFactory)CompressedSource.CompressionMode.GZIP);
        Assert.assertFalse((boolean)source.isSplittable());
        source = CompressedSource.from((FileBasedSource)new ByteSource("input.txt", 1L)).withDecompression((CompressedSource.DecompressingChannelFactory)CompressedSource.CompressionMode.GZIP);
        Assert.assertFalse((boolean)source.isSplittable());
        source = CompressedSource.from((FileBasedSource)new ByteSource("input.csv", 1L)).withDecompression((CompressedSource.DecompressingChannelFactory)CompressedSource.CompressionMode.GZIP);
        Assert.assertFalse((boolean)source.isSplittable());
    }

    @Test
    public void testReadBzip2() throws Exception {
        byte[] input = this.generateInput(5000);
        this.runReadTest(input, CompressedSource.CompressionMode.BZIP2);
    }

    @Test
    public void testReadZip() throws Exception {
        byte[] input = this.generateInput(5000);
        this.runReadTest(input, CompressedSource.CompressionMode.ZIP);
    }

    @Test
    public void testReadDeflate() throws Exception {
        byte[] input = this.generateInput(5000);
        this.runReadTest(input, CompressedSource.CompressionMode.DEFLATE);
    }

    @Test
    public void testEmptyReadGzip() throws Exception {
        byte[] input = this.generateInput(0);
        this.runReadTest(input, CompressedSource.CompressionMode.GZIP);
    }

    private static byte[] compressGzip(byte[] input) throws IOException {
        ByteArrayOutputStream res = new ByteArrayOutputStream();
        try (GZIPOutputStream gzipStream = new GZIPOutputStream(res);){
            gzipStream.write(input);
        }
        return res.toByteArray();
    }

    private static byte[] concat(byte[] first, byte[] second) {
        byte[] res = new byte[first.length + second.length];
        System.arraycopy(first, 0, res, 0, first.length);
        System.arraycopy(second, 0, res, first.length, second.length);
        return res;
    }

    @Test
    public void testReadConcatenatedGzip() throws IOException {
        byte[] header = "a,b,c\n".getBytes(StandardCharsets.UTF_8);
        byte[] body = "1,2,3\n4,5,6\n7,8,9\n".getBytes(StandardCharsets.UTF_8);
        byte[] expected = CompressedSourceTest.concat(header, body);
        byte[] totalGz = CompressedSourceTest.concat(CompressedSourceTest.compressGzip(header), CompressedSourceTest.compressGzip(body));
        File tmpFile = this.tmpFolder.newFile();
        try (FileOutputStream os = new FileOutputStream(tmpFile);){
            os.write(totalGz);
        }
        CompressedSource source = CompressedSource.from((FileBasedSource)new ByteSource(tmpFile.getAbsolutePath(), 1L)).withDecompression((CompressedSource.DecompressingChannelFactory)CompressedSource.CompressionMode.GZIP);
        List actual = SourceTestUtils.readFromSource((BoundedSource)source, (PipelineOptions)PipelineOptionsFactory.create());
        Assert.assertEquals(Bytes.asList(expected), (Object)actual);
    }

    @Test
    public void testReadMultiStreamBzip2() throws IOException {
        CompressedSource.CompressionMode mode = CompressedSource.CompressionMode.BZIP2;
        byte[] input1 = this.generateInput(5, 587973);
        byte[] input2 = this.generateInput(5, 387374);
        ByteArrayOutputStream stream1 = new ByteArrayOutputStream();
        try (OutputStream os = this.getOutputStreamForMode(mode, stream1);){
            os.write(input1);
        }
        ByteArrayOutputStream stream2 = new ByteArrayOutputStream();
        try (OutputStream os = this.getOutputStreamForMode(mode, stream2);){
            os.write(input2);
        }
        File tmpFile = this.tmpFolder.newFile();
        try (FileOutputStream os = new FileOutputStream(tmpFile);){
            ((OutputStream)os).write(stream1.toByteArray());
            ((OutputStream)os).write(stream2.toByteArray());
        }
        byte[] output = Bytes.concat(input1, input2);
        this.verifyReadContents(output, tmpFile, (CompressedSource.DecompressingChannelFactory)mode);
    }

    @Test
    public void testCompressedReadBzip2() throws Exception {
        byte[] input = this.generateInput(0);
        this.runReadTest(input, CompressedSource.CompressionMode.BZIP2);
    }

    @Test
    public void testCompressedAccordingToFilepatternGzip() throws Exception {
        byte[] input = this.generateInput(100);
        File tmpFile = this.tmpFolder.newFile("test.gz");
        this.writeFile(tmpFile, input, CompressedSource.CompressionMode.GZIP);
        this.verifyReadContents(input, tmpFile, null);
    }

    @Test
    public void testCompressedAccordingToFilepatternBzip2() throws Exception {
        byte[] input = this.generateInput(100);
        File tmpFile = this.tmpFolder.newFile("test.bz2");
        this.writeFile(tmpFile, input, CompressedSource.CompressionMode.BZIP2);
        this.verifyReadContents(input, tmpFile, null);
    }

    @Test
    public void testHeterogeneousCompression() throws Exception {
        String baseName = "test-input";
        byte[] generated = this.generateInput(1000);
        ArrayList<Byte> expected = new ArrayList<Byte>();
        File uncompressedFile = this.tmpFolder.newFile(baseName + ".bin");
        generated = this.generateInput(1000, 1);
        Files.write(generated, uncompressedFile);
        expected.addAll(Bytes.asList(generated));
        File gzipFile = this.tmpFolder.newFile(baseName + ".gz");
        generated = this.generateInput(1000, 2);
        this.writeFile(gzipFile, generated, CompressedSource.CompressionMode.GZIP);
        expected.addAll(Bytes.asList(generated));
        File bzip2File = this.tmpFolder.newFile(baseName + ".bz2");
        generated = this.generateInput(1000, 3);
        this.writeFile(bzip2File, generated, CompressedSource.CompressionMode.BZIP2);
        expected.addAll(Bytes.asList(generated));
        String filePattern = new File(this.tmpFolder.getRoot().toString(), baseName + ".*").toString();
        CompressedSource source = CompressedSource.from((FileBasedSource)new ByteSource(filePattern, 1L));
        List actual = SourceTestUtils.readFromSource((BoundedSource)source, (PipelineOptions)PipelineOptionsFactory.create());
        Assert.assertEquals(HashMultiset.create(actual), HashMultiset.create(expected));
    }

    @Test
    public void testUncompressedFileWithAutoIsSplittable() throws Exception {
        String baseName = "test-input";
        File uncompressedFile = this.tmpFolder.newFile(baseName + ".bin");
        Files.write(this.generateInput(10), uncompressedFile);
        CompressedSource source = CompressedSource.from((FileBasedSource)new ByteSource(uncompressedFile.getPath(), 1L));
        Assert.assertTrue((boolean)source.isSplittable());
        SourceTestUtils.assertSplitAtFractionExhaustive((BoundedSource)source, (PipelineOptions)PipelineOptionsFactory.create());
    }

    @Test
    public void testUncompressedFileWithUncompressedIsSplittable() throws Exception {
        String baseName = "test-input";
        File uncompressedFile = this.tmpFolder.newFile(baseName + ".bin");
        Files.write(this.generateInput(10), uncompressedFile);
        CompressedSource source = CompressedSource.from((FileBasedSource)new ByteSource(uncompressedFile.getPath(), 1L)).withDecompression((CompressedSource.DecompressingChannelFactory)CompressedSource.CompressionMode.UNCOMPRESSED);
        Assert.assertTrue((boolean)source.isSplittable());
        SourceTestUtils.assertSplitAtFractionExhaustive((BoundedSource)source, (PipelineOptions)PipelineOptionsFactory.create());
    }

    @Test
    public void testGzipFileIsNotSplittable() throws Exception {
        String baseName = "test-input";
        File compressedFile = this.tmpFolder.newFile(baseName + ".gz");
        this.writeFile(compressedFile, this.generateInput(10), CompressedSource.CompressionMode.GZIP);
        CompressedSource source = CompressedSource.from((FileBasedSource)new ByteSource(compressedFile.getPath(), 1L));
        Assert.assertFalse((boolean)source.isSplittable());
    }

    @Test
    public void testBzip2FileIsNotSplittable() throws Exception {
        String baseName = "test-input";
        File compressedFile = this.tmpFolder.newFile(baseName + ".bz2");
        this.writeFile(compressedFile, this.generateInput(10), CompressedSource.CompressionMode.BZIP2);
        CompressedSource source = CompressedSource.from((FileBasedSource)new ByteSource(compressedFile.getPath(), 1L));
        Assert.assertFalse((boolean)source.isSplittable());
    }

    @Test
    public void testFalseGzipStream() throws Exception {
        byte[] input = this.generateInput(1000);
        File tmpFile = this.tmpFolder.newFile("test.gz");
        Files.write(input, tmpFile);
        this.verifyReadContents(input, tmpFile, (CompressedSource.DecompressingChannelFactory)CompressedSource.CompressionMode.GZIP);
    }

    @Test
    public void testFalseBzip2Stream() throws Exception {
        byte[] input = this.generateInput(1000);
        File tmpFile = this.tmpFolder.newFile("test.bz2");
        Files.write(input, tmpFile);
        this.thrown.expectMessage("Stream is not in the BZip2 format");
        this.verifyReadContents(input, tmpFile, (CompressedSource.DecompressingChannelFactory)CompressedSource.CompressionMode.BZIP2);
    }

    @Test
    public void testEmptyReadGzipUncompressed() throws Exception {
        byte[] input = this.generateInput(0);
        File tmpFile = this.tmpFolder.newFile("test.gz");
        Files.write(input, tmpFile);
        this.verifyReadContents(input, tmpFile, (CompressedSource.DecompressingChannelFactory)CompressedSource.CompressionMode.GZIP);
    }

    @Test
    public void testOneByteReadGzipUncompressed() throws Exception {
        byte[] input = this.generateInput(1);
        File tmpFile = this.tmpFolder.newFile("test.gz");
        Files.write(input, tmpFile);
        this.verifyReadContents(input, tmpFile, (CompressedSource.DecompressingChannelFactory)CompressedSource.CompressionMode.GZIP);
    }

    @Test
    public void testCompressedReadMultipleFiles() throws Exception {
        int numFiles = 3;
        String baseName = "test_input-";
        String filePattern = new File(this.tmpFolder.getRoot().toString(), baseName + "*").toString();
        ArrayList<Byte> expected = new ArrayList<Byte>();
        for (int i = 0; i < numFiles; ++i) {
            byte[] generated = this.generateInput(100);
            File tmpFile = this.tmpFolder.newFile(baseName + i);
            this.writeFile(tmpFile, generated, CompressedSource.CompressionMode.GZIP);
            expected.addAll(Bytes.asList(generated));
        }
        CompressedSource source = CompressedSource.from((FileBasedSource)new ByteSource(filePattern, 1L)).withDecompression((CompressedSource.DecompressingChannelFactory)CompressedSource.CompressionMode.GZIP);
        List actual = SourceTestUtils.readFromSource((BoundedSource)source, (PipelineOptions)PipelineOptionsFactory.create());
        Assert.assertEquals(HashMultiset.create(expected), HashMultiset.create(actual));
    }

    @Test
    public void testDisplayData() {
        ByteSource inputSource = new ByteSource("foobar.txt", 1L){

            public void populateDisplayData(DisplayData.Builder builder) {
                builder.add(DisplayData.item((String)"foo", (String)"bar"));
            }
        };
        CompressedSource compressedSource = CompressedSource.from((FileBasedSource)inputSource);
        CompressedSource gzipSource = compressedSource.withDecompression((CompressedSource.DecompressingChannelFactory)CompressedSource.CompressionMode.GZIP);
        DisplayData compressedSourceDisplayData = DisplayData.from((HasDisplayData)compressedSource);
        DisplayData gzipDisplayData = DisplayData.from((HasDisplayData)gzipSource);
        Assert.assertThat((Object)compressedSourceDisplayData, DisplayDataMatchers.hasDisplayItem("compressionMode"));
        Assert.assertThat((Object)gzipDisplayData, DisplayDataMatchers.hasDisplayItem("compressionMode", CompressedSource.CompressionMode.GZIP.toString()));
        Assert.assertThat((Object)compressedSourceDisplayData, DisplayDataMatchers.hasDisplayItem("source", ((Object)((Object)inputSource)).getClass()));
        Assert.assertThat((Object)compressedSourceDisplayData, DisplayDataMatchers.includesDisplayDataFor("source", (HasDisplayData)inputSource));
    }

    private byte[] generateInput(int size) {
        return this.generateInput(size, 285930);
    }

    private byte[] generateInput(int size, int seed) {
        Random random = new Random(seed);
        byte[] buff = new byte[size];
        random.nextBytes(buff);
        return buff;
    }

    private OutputStream getOutputStreamForMode(CompressedSource.CompressionMode mode, OutputStream stream) throws IOException {
        switch (mode) {
            case GZIP: {
                return new GzipCompressorOutputStream(stream);
            }
            case BZIP2: {
                return new BZip2CompressorOutputStream(stream);
            }
            case ZIP: {
                return new TestZipOutputStream(stream);
            }
            case DEFLATE: {
                return new DeflateCompressorOutputStream(stream);
            }
        }
        throw new RuntimeException("Unexpected compression mode");
    }

    private void writeFile(File file, byte[] input, CompressedSource.CompressionMode mode) throws IOException {
        try (OutputStream os = this.getOutputStreamForMode(mode, new FileOutputStream(file));){
            os.write(input);
        }
    }

    private void runReadTest(byte[] input, CompressedSource.CompressionMode inputCompressionMode, @Nullable CompressedSource.DecompressingChannelFactory decompressionFactory) throws IOException {
        File tmpFile = this.tmpFolder.newFile();
        this.writeFile(tmpFile, input, inputCompressionMode);
        this.verifyReadContents(input, tmpFile, decompressionFactory);
    }

    private void verifyReadContents(byte[] expected, File inputFile, @Nullable CompressedSource.DecompressingChannelFactory decompressionFactory) throws IOException {
        CompressedSource source = CompressedSource.from((FileBasedSource)new ByteSource(inputFile.toPath().toString(), 1L));
        if (decompressionFactory != null) {
            source = source.withDecompression(decompressionFactory);
        }
        ArrayList<KV> actualOutput = Lists.newArrayList();
        try (BoundedSource.BoundedReader reader = source.createReader(PipelineOptionsFactory.create());){
            boolean more = reader.start();
            while (more) {
                actualOutput.add(KV.of((Object)reader.getCurrentTimestamp().getMillis(), (Object)((Byte)reader.getCurrent())));
                more = reader.advance();
            }
        }
        ArrayList<KV> expectedOutput = Lists.newArrayList();
        for (int i = 0; i < expected.length; ++i) {
            expectedOutput.add(KV.of((Object)i, (Object)expected[i]));
        }
        Assert.assertEquals(expectedOutput, actualOutput);
    }

    private void runReadTest(byte[] input, CompressedSource.CompressionMode mode) throws IOException {
        this.runReadTest(input, mode, (CompressedSource.DecompressingChannelFactory)mode);
    }

    @Test
    public void testEmptyGzipProgress() throws IOException {
        File tmpFile = this.tmpFolder.newFile("empty.gz");
        String filename = tmpFile.toPath().toString();
        this.writeFile(tmpFile, new byte[0], CompressedSource.CompressionMode.GZIP);
        PipelineOptions options = PipelineOptionsFactory.create();
        CompressedSource source = CompressedSource.from((FileBasedSource)new ByteSource(filename, 1L));
        try (BoundedSource.BoundedReader readerOrig = source.createReader(options);){
            Assert.assertThat((Object)readerOrig, (Matcher)Matchers.instanceOf(CompressedSource.CompressedReader.class));
            CompressedSource.CompressedReader reader = (CompressedSource.CompressedReader)readerOrig;
            Assert.assertEquals((double)0.0, (double)reader.getFractionConsumed(), (double)1.0E-6);
            Assert.assertEquals((long)0L, (long)reader.getSplitPointsConsumed());
            Assert.assertEquals((long)1L, (long)reader.getSplitPointsRemaining());
            Assert.assertFalse((boolean)reader.start());
            Assert.assertEquals((double)1.0, (double)reader.getFractionConsumed(), (double)1.0E-6);
            Assert.assertEquals((long)0L, (long)reader.getSplitPointsConsumed());
            Assert.assertEquals((long)0L, (long)reader.getSplitPointsRemaining());
        }
    }

    @Test
    public void testGzipProgress() throws IOException {
        int numRecords = 3;
        File tmpFile = this.tmpFolder.newFile("nonempty.gz");
        String filename = tmpFile.toPath().toString();
        this.writeFile(tmpFile, new byte[numRecords], CompressedSource.CompressionMode.GZIP);
        PipelineOptions options = PipelineOptionsFactory.create();
        CompressedSource source = CompressedSource.from((FileBasedSource)new ByteSource(filename, 1L));
        try (BoundedSource.BoundedReader readerOrig = source.createReader(options);){
            Assert.assertThat((Object)readerOrig, (Matcher)Matchers.instanceOf(CompressedSource.CompressedReader.class));
            CompressedSource.CompressedReader reader = (CompressedSource.CompressedReader)readerOrig;
            Assert.assertEquals((double)0.0, (double)reader.getFractionConsumed(), (double)1.0E-6);
            Assert.assertEquals((long)0L, (long)reader.getSplitPointsConsumed());
            Assert.assertEquals((long)1L, (long)reader.getSplitPointsRemaining());
            for (int i = 0; i < numRecords; ++i) {
                if (i == 0) {
                    Assert.assertTrue((boolean)reader.start());
                } else {
                    Assert.assertTrue((boolean)reader.advance());
                }
                Assert.assertEquals((long)0L, (long)reader.getSplitPointsConsumed());
                Assert.assertEquals((long)1L, (long)reader.getSplitPointsRemaining());
            }
            Assert.assertFalse((boolean)reader.advance());
            Assert.assertEquals((double)1.0, (double)reader.getFractionConsumed(), (double)1.0E-6);
            Assert.assertEquals((long)1L, (long)reader.getSplitPointsConsumed());
            Assert.assertEquals((long)0L, (long)reader.getSplitPointsRemaining());
        }
    }

    @Test
    public void testUnsplittable() throws IOException {
        String baseName = "test-input";
        File compressedFile = this.tmpFolder.newFile(baseName + ".gz");
        byte[] input = this.generateInput(10000);
        this.writeFile(compressedFile, input, CompressedSource.CompressionMode.GZIP);
        CompressedSource source = CompressedSource.from((FileBasedSource)new ByteSource(compressedFile.getPath(), 1L));
        ArrayList<Byte> expected = Lists.newArrayList();
        for (byte i : input) {
            expected.add(i);
        }
        PipelineOptions options = PipelineOptionsFactory.create();
        BoundedSource.BoundedReader reader = source.createReader(options);
        ArrayList<Byte> actual = Lists.newArrayList();
        boolean hasNext = reader.start();
        while (hasNext) {
            actual.add((Byte)reader.getCurrent());
            if (actual.size() % 9 == 0) {
                Double fractionConsumed = reader.getFractionConsumed();
                Assert.assertNotNull((Object)fractionConsumed);
                Assert.assertNull((Object)reader.splitAtFraction(fractionConsumed.doubleValue()));
            }
            hasNext = reader.advance();
        }
        Assert.assertEquals((long)expected.size(), (long)actual.size());
        Assert.assertEquals(Sets.newHashSet(expected), Sets.newHashSet(actual));
    }

    @Test
    public void testSplittableProgress() throws IOException {
        File tmpFile = this.tmpFolder.newFile("nonempty.txt");
        String filename = tmpFile.toPath().toString();
        Files.write(new byte[2], tmpFile);
        PipelineOptions options = PipelineOptionsFactory.create();
        CompressedSource source = CompressedSource.from((FileBasedSource)new ByteSource(filename, 1L));
        try (BoundedSource.BoundedReader readerOrig = source.createReader(options);){
            Assert.assertThat((Object)readerOrig, (Matcher)Matchers.not((Matcher)Matchers.instanceOf(CompressedSource.CompressedReader.class)));
            Assert.assertThat((Object)readerOrig, (Matcher)Matchers.instanceOf(FileBasedSource.FileBasedReader.class));
            FileBasedSource.FileBasedReader reader = (FileBasedSource.FileBasedReader)readerOrig;
            Assert.assertEquals((double)0.0, (double)reader.getFractionConsumed(), (double)1.0E-6);
            Assert.assertEquals((long)0L, (long)reader.getSplitPointsConsumed());
            Assert.assertEquals((long)-1L, (long)reader.getSplitPointsRemaining());
            Assert.assertTrue((boolean)reader.start());
            Assert.assertEquals((long)0L, (long)reader.getSplitPointsConsumed());
            Assert.assertEquals((long)-1L, (long)reader.getSplitPointsRemaining());
            Assert.assertTrue((boolean)reader.advance());
            Assert.assertEquals((long)1L, (long)reader.getSplitPointsConsumed());
            Assert.assertEquals((long)1L, (long)reader.getSplitPointsRemaining());
            Assert.assertFalse((boolean)reader.advance());
            Assert.assertEquals((double)1.0, (double)reader.getFractionConsumed(), (double)1.0E-6);
            Assert.assertEquals((long)2L, (long)reader.getSplitPointsConsumed());
            Assert.assertEquals((long)0L, (long)reader.getSplitPointsRemaining());
        }
    }

    private static class ExtractIndexFromTimestamp
    extends DoFn<Byte, KV<Long, Byte>> {
        private ExtractIndexFromTimestamp() {
        }

        @DoFn.ProcessElement
        public void processElement(DoFn.ProcessContext context) {
            context.output((Object)KV.of((Object)context.timestamp().getMillis(), (Object)((Byte)context.element())));
        }
    }

    private static class ByteSource
    extends FileBasedSource<Byte> {
        public ByteSource(String fileOrPatternSpec, long minBundleSize) {
            super((ValueProvider)ValueProvider.StaticValueProvider.of((Object)fileOrPatternSpec), minBundleSize);
        }

        public ByteSource(MatchResult.Metadata metadata, long minBundleSize, long startOffset, long endOffset) {
            super(metadata, minBundleSize, startOffset, endOffset);
        }

        protected ByteSource createForSubrangeOfFile(MatchResult.Metadata metadata, long start, long end) {
            return new ByteSource(metadata, this.getMinBundleSize(), start, end);
        }

        protected FileBasedSource.FileBasedReader<Byte> createSingleFileReader(PipelineOptions options) {
            return new ByteReader(this);
        }

        public Coder<Byte> getOutputCoder() {
            return SerializableCoder.of(Byte.class);
        }

        private static class ByteReader
        extends FileBasedSource.FileBasedReader<Byte> {
            ByteBuffer buff = ByteBuffer.allocate(1);
            Byte current;
            long offset;
            ReadableByteChannel channel;

            public ByteReader(ByteSource source) {
                super((FileBasedSource)source);
                this.offset = source.getStartOffset() - 1L;
            }

            public Byte getCurrent() throws NoSuchElementException {
                return this.current;
            }

            protected boolean isAtSplitPoint() {
                return true;
            }

            protected void startReading(ReadableByteChannel channel) throws IOException {
                this.channel = channel;
            }

            protected boolean readNextRecord() throws IOException {
                this.buff.clear();
                if (this.channel.read(this.buff) != 1) {
                    return false;
                }
                this.current = this.buff.get(0);
                ++this.offset;
                return true;
            }

            protected long getCurrentOffset() {
                return this.offset;
            }

            public Instant getCurrentTimestamp() throws NoSuchElementException {
                return new Instant(this.getCurrentOffset());
            }
        }
    }

    private static class TestZipOutputStream
    extends OutputStream {
        private ZipOutputStream zipOutputStream;
        private long offset = 0L;
        private int entry = 0;

        public TestZipOutputStream(OutputStream stream) throws IOException {
            this.zipOutputStream = new ZipOutputStream(stream);
            this.zipOutputStream.putNextEntry(new ZipEntry(String.format("entry-%05d", this.entry)));
        }

        @Override
        public void write(int b) throws IOException {
            this.zipOutputStream.write(b);
            ++this.offset;
            if (this.offset % 100L == 0L) {
                ++this.entry;
                this.zipOutputStream.putNextEntry(new ZipEntry(String.format("entry-%05d", this.entry)));
            }
        }

        @Override
        public void close() throws IOException {
            this.zipOutputStream.closeEntry();
            super.close();
        }
    }
}

