package security.service;

import org.springframework.util.AntPathMatcher;
import org.springframework.util.StringUtils;
import security.filters.FunctionPermissionHandler;

import java.util.*;

/**
 * 功能路径匹配
 */
public class PathMatcher {

    private final Map<String, FunctionPermissionHandler> fullMatchMap = new HashMap<>();
    private final Map<String, FunctionPermissionHandler> fullMatchMapOfMethod = new HashMap<>();
    private final List<FunctionPermissionHandler> startMatchList = new ArrayList<>();
    private final List<FunctionPermissionHandler> startMatchListOfMethod = new ArrayList<>();
    private final AntPathMatcher antPathMatcher = new AntPathMatcher();

    public PathMatcher(List<FunctionPermissionHandler> functionPermissionHandlers) {
        if (functionPermissionHandlers != null) {
            for (FunctionPermissionHandler handler : functionPermissionHandlers) {
                if (StringUtils.hasText(handler.getFunctionPermission().getPath())) {
                    // 前导匹配
                    if (this.antPathMatcher.isPattern(handler.getFunctionPermission().getPath())) {
                        if (StringUtils.hasText(handler.getFunctionPermission().getMethod())) {
                            startMatchListOfMethod.add(handler);
                        } else {
                            startMatchList.add(handler);
                        }
                    }
                    // 完全匹配
                    else {
                        if (StringUtils.hasText(handler.getFunctionPermission().getMethod())) {
                            fullMatchMapOfMethod.put(handler.getFunctionPermission().getMethod() + ":" + handler.getFunctionPermission().getPath(), handler);
                        } else {
                            fullMatchMap.put(handler.getFunctionPermission().getPath(), handler);
                        }
                    }
                }
            }
        }
        //按路径长度排序
        startMatchList.sort(new Comparator<FunctionPermissionHandler>() {
            @Override
            public int compare(FunctionPermissionHandler o1, FunctionPermissionHandler o2) {
                return Integer.compare(o2.getFunctionPermission().getPath().length(), o1.getFunctionPermission().getPath().length());
            }
        });
        startMatchListOfMethod.sort(new Comparator<FunctionPermissionHandler>() {
            @Override
            public int compare(FunctionPermissionHandler o1, FunctionPermissionHandler o2) {
                return Integer.compare(o2.getFunctionPermission().getPath().length(), o1.getFunctionPermission().getPath().length());
            }
        });
    }

    /**
     * 匹配功能项
     *
     * @param method
     * @param path
     * @return
     */
    public FunctionPermissionHandler match(String method, String path) {
        if (!StringUtils.hasText(path)) return null;
        if (!StringUtils.hasText(method)) return null;
        //
        FunctionPermissionHandler functionPermission;
        // 方法 + 完全路径匹配
        functionPermission = fullMatchMapOfMethod.get(method + ":" + path);
        if (functionPermission != null) return functionPermission;

        // 完全路径匹配
        functionPermission = fullMatchMap.get(path);
        if (functionPermission != null) return functionPermission;

        // 方法 + 前导匹配
        for (FunctionPermissionHandler fp : startMatchListOfMethod) {
            if (method.equals(fp.getFunctionPermission().getMethod()) && this.antPathMatcher.match(fp.getFunctionPermission().getPath(), path)) {
                return fp;
            }
        }

        // 前导匹配
        for (FunctionPermissionHandler fp : startMatchList) {
            if (this.antPathMatcher.match(fp.getFunctionPermission().getPath(), path)) {
                return fp;
            }
        }

        // 未找到
        return null;
    }
}