package security.filters;

import framework.config.SecurityConfig;
import framework.security.Account;
import framework.security.FunctionPermission;
import framework.security.token.AuthTokenInfo;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;
import org.springframework.util.StringUtils;
import security.defined.AuthenticationCheckerResult;
import security.error.AuthenticationCheckerException;
import security.service.FunctionPermissionLoader;
import security.service.PathMatcher;
import security.service.SecurityContext;

import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Base64;
import java.util.Date;
import java.util.List;

@Slf4j
public abstract class AuthenticationBase {

    @Getter
    private final SecurityContext securityContext;
    @Getter
    private final SecurityConfig securityConfig;
    @Getter
    private final PathMatcher authPathMatcher;
    @Getter
    protected boolean corsDisabled = true;

    public AuthenticationBase(SecurityContext securityContext) {
        this.securityContext = securityContext;
        this.securityConfig = securityContext.getSecurityConfig();
        //
        List<FunctionPermission> functionPermissionList = this.loadFunctionPermissionList();
        this.appendSecurityConfig(functionPermissionList);
        this.authPathMatcher = new PathMatcher(this.convertFunctionPermissionHandler(functionPermissionList));
    }

    /**
     * 将功能函数列表转换为处理器列表
     *
     * @param functionPermissionList
     * @return
     */
    protected List<FunctionPermissionHandler> convertFunctionPermissionHandler(List<FunctionPermission> functionPermissionList) {
        List<FunctionPermissionHandler> list = new ArrayList<>();
        for (FunctionPermission functionPermission : functionPermissionList) {
            list.add(new FunctionPermissionHandler(functionPermission));
        }
        return list;
    }

    /**
     * 将配置文件中的配置项附加到功能列表
     *
     * @param functionPermissionList
     */
    protected void appendSecurityConfig(List<FunctionPermission> functionPermissionList) {
        String pathAnonList = getSecurityConfig().getPathAnonList();
        String pathAuthList = getSecurityConfig().getPathAuthList();
        if (pathAnonList != null) {
            for (String path : pathAnonList.split(",")) {
                functionPermissionList.add(new FunctionPermission(path, "ANON", null));
            }
        }
        if (pathAuthList != null) {
            for (String path : pathAuthList.split(",")) {
                functionPermissionList.add(new FunctionPermission(path, "AUTH", null));
            }
        }
    }

    /**
     * 执行授权检查
     *
     * @param handler
     */
    public AuthenticationData authCheck(AuthenticationHandler handler) throws AuthenticationCheckerException {
        if (isCorsDisabled()) {
            if ("OPTIONS".equals(handler.getRequestMethod())) {
                // 放行所有 OPTIONS 请求
                return new AuthenticationData(0L, "", null);
            }
        }

        // 获取token
        String token = handler.getRequestToken();
        Date tokenExpired = null;
        Long id = 0L;

        // token check
        if (StringUtils.hasText(token)) {
            AuthTokenInfo tokenInfo = null;
            // token parse
            if (token.startsWith("BASIC ")) {
                tokenInfo = this.basicParser(token);
            } else if (token.startsWith("TOKEN ")) {
                tokenInfo = this.tokenParser(token);
            }
            // token check
            if (tokenInfo != null && !tokenInfo.isExpired()) {
                id = tokenInfo.getId();
                tokenExpired = tokenInfo.getExpireTime();
            }
        }

        // 加载功能
        FunctionPermissionHandler functionPermissionHandler = this.findFunctionMatcher(handler);
        if (functionPermissionHandler == null) {
            // 请求的路径在配置中不存在，强制要求授权
            throw new AuthenticationCheckerException(AuthenticationCheckerResult.NO_PERMISSION);
        }

        // 匿名授权
        if (this.hasAnonymous(handler, functionPermissionHandler)) {
            return new AuthenticationData(0L, "", tokenExpired);
        }

        // 未登录
        if (id == null || id < 1L) {
            throw new AuthenticationCheckerException(AuthenticationCheckerResult.NO_AUTH);
        }

        // 判定授权要求
        if (!functionPermissionHandler.hasPermission("AUTH")) {
            // 账号权限加载
            List<String> permissions = getSecurityContext().getAccountLoader().loadPermissions(id);
            // 权限验证
            if (!this.hasPermissions(handler, permissions, functionPermissionHandler)) {
                throw new AuthenticationCheckerException(AuthenticationCheckerResult.NO_PERMISSION);
            }
        }

        AuthenticationData data = new AuthenticationData(id, token, tokenExpired);

        // 过期判定
        this.tokenReminder(data, handler);

        // 填充安全上下文
        return data;
    }

    /**
     * 过期提醒判定
     *
     * @param data
     * @param handler
     */
    protected void tokenReminder(AuthenticationData data, AuthenticationHandler handler) {
        Integer tokenExpirationReminderSeconds = getSecurityConfig().getTokenExpirationReminderSeconds();
        // 过期时间
        if (data.getExpireTime() != null && tokenExpirationReminderSeconds != null && tokenExpirationReminderSeconds > 0) {
            long sec = (data.getExpireTime().getTime() - new Date().getTime()) / 1000;
            if (sec < tokenExpirationReminderSeconds) {
                handler.tokenExpiredReminder((int) sec);
            }
        }
    }


    /**
     * 未授权
     *
     * @param handler
     * @param functionPermissionHandler
     */
    protected boolean hasAnonymous(AuthenticationHandler handler, FunctionPermissionHandler functionPermissionHandler) {
        return functionPermissionHandler.hasPermission("ANON");
    }

    /**
     * 权限检查
     *
     * @param handler
     * @param permissions
     * @param functionPermissionHandler
     * @return
     */
    protected boolean hasPermissions(AuthenticationHandler handler, List<String> permissions, FunctionPermissionHandler functionPermissionHandler) {
        if (permissions == null) return false;
        if (permissions.size() == 0) return false;
        //
        for (String permission : permissions) {
            if (permission != null && functionPermissionHandler.hasPermission(permission)) {
                return true;
            }
        }
        //
        return false;
    }

    /**
     * Token解析器
     *
     * @param token
     * @return
     */
    protected AuthTokenInfo tokenParser(String token) {
        AuthTokenInfo tokenInfo;
        try {
            token = token.substring(6);
            tokenInfo = getSecurityContext().getTokenBuilder().decode(token);
        } catch (Exception ex) {
            tokenInfo = null;
        }
        return tokenInfo;
    }

    /**
     * Token解析器
     *
     * @param token
     * @return
     */
    protected AuthTokenInfo basicParser(String token) {
        AuthTokenInfo tokenInfo = null;
        try {
            token = token.substring(6);
            byte[] decode = Base64.getDecoder().decode(token);
            String s = new String(decode, StandardCharsets.UTF_8);
            String[] strings = s.split(":", 2);
            if (strings.length == 2) {
                Account account = null;
                try {
                    account = getSecurityContext().getAccountChecker().authCheck(strings[0], strings[1], null);
                } catch (Exception exception) {
                    // 记录失败记录
                    getSecurityContext().getAccountLoader().loginUnsuccessful(strings[0], exception.getMessage());
                }
                if (account != null) {
                    tokenInfo = new AuthTokenInfo(account.getId(), account.accountExpired());
                }
            }
        } catch (Exception ex) {
            tokenInfo = null;
        }
        return tokenInfo;
    }

    /**
     * 查找功能匹配器
     *
     * @return
     */
    protected FunctionPermissionHandler findFunctionMatcher(AuthenticationHandler handler) {
        String path = handler.getRequestPath();
        String method = handler.getRequestMethod();
        //
        if ("".equals(path)) path = "/";
        // 加载功能列表
        return this.findFunctionMatcher(method, path);
    }

    /**
     * 查找功能匹配器
     *
     * @return
     */
    protected FunctionPermissionHandler findFunctionMatcher(String method, String path) {
        return getAuthPathMatcher().match(method, path);
    }

    /**
     * 加载系统配置的功能权限列表
     *
     * @return
     */
    protected List<FunctionPermission> loadFunctionPermissionList() {
        List<FunctionPermission> functionPermissionList = getSecurityContext().getAccountLoader().loadFunctionPermission();
        if (functionPermissionList == null) {
            functionPermissionList = new ArrayList<>();
        }
        if (getSecurityContext().getFunctionPermissionLoaders() != null) {
            for (FunctionPermissionLoader functionPermissionLoader : getSecurityContext().getFunctionPermissionLoaders()) {
                functionPermissionLoader.loading(functionPermissionList, getSecurityConfig());
            }
        }
        return functionPermissionList;
    }
}
