package security.filters;

import framework.config.SecurityConfig;
import framework.crypto.AESCrypto;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;
import org.springframework.util.StringUtils;
import security.utils.SecurityAuthUtil;
import security.utils.SecurityTokenUtil;

import javax.servlet.*;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;

/**
 * 获取上级传送的Auth Identity
 */
@Slf4j
public class AuthIdentityServletFilter implements Filter {

    @Getter
    private SecurityConfig securityConfig;
    private AESCrypto aesCrypto;

    //
    private final boolean isAuthIdentityHeader;
    public final String authIdentityHeaderName;
    public final String authIdentityHeaderNameSign;

    public AuthIdentityServletFilter(SecurityConfig securityConfig, AESCrypto aesCrypto) {
        this.securityConfig = securityConfig;
        this.aesCrypto = aesCrypto;
        //
        this.isAuthIdentityHeader = StringUtils.hasText(securityConfig.getAuthIdentityHeader());
        this.authIdentityHeaderName = securityConfig.getAuthIdentityHeader();
        this.authIdentityHeaderNameSign = securityConfig.getAuthIdentityHeader() + "-SIGN";
    }

    /**
     * 获取由上级路由传输过来的 Auth Identity
     *
     * @param servletRequest
     * @return
     */
    private Long getAuthIdentity(HttpServletRequest servletRequest) {
        String identity = servletRequest.getHeader(authIdentityHeaderName);
        if (StringUtils.hasText(identity)) {
            return Long.parseLong(identity);
        }
        return 0L;
    }

    /**
     * 获取由上级路由传输过来的 Auth Identity
     *
     * @param servletRequest
     * @return
     */
    private String getAuthSign(HttpServletRequest servletRequest) {
        return servletRequest.getHeader(authIdentityHeaderNameSign);
    }

    @Override
    public void doFilter(ServletRequest servletRequest, ServletResponse servletResponse, FilterChain filterChain) throws IOException, ServletException {
        // 配置了网关授权头
        if (this.isAuthIdentityHeader) {
            HttpServletRequest request = (HttpServletRequest) servletRequest;
            HttpServletResponse response = (HttpServletResponse) servletResponse;
            // 获取由上级传递过来的 auth identity
            Long id = this.getAuthIdentity(request);
            if (id == null) {
                // enable debug
                if (getSecurityConfig().getDebugAccountId() != null) {
                    id = getSecurityConfig().getDebugAccountId();
                }
            } else {
                String authSign = this.getAuthSign(request);
                if (!StringUtils.hasText(authSign)) {
                    response.sendError(403, "illegal access");
                    return;
                }
                String s;
                try {
                    s = this.aesCrypto.decodeBase64AsString(authSign);
                } catch (Exception exception) {
                    s = null;
                }
                if (!StringUtils.hasText(s)) {
                    response.sendError(403, "illegal access");
                    return;
                }
                if (!s.equals(id + "00x")) {
                    response.sendError(403, "illegal access");
                    return;
                }
            }
            // token get
            String token = SecurityTokenUtil.getRequestToken(request);
            // 将获取到的数据填充到安全组件
            SecurityAuthUtil.authorized(id, token);
        }
        // 未配置网关授权头
        else {
            filterChain.doFilter(servletRequest, servletResponse);
        }
    }
}
