package security.filters;

import framework.config.SecurityConfig;
import framework.security.AccountLoader;
import framework.security.FunctionPermission;
import framework.security.token.AuthTokenBuilder;
import framework.security.token.AuthTokenInfo;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;
import org.springframework.util.StringUtils;
import security.defined.AuthenticationCheckerResult;
import security.error.AuthenticationCheckerException;
import security.service.PathMatcher;

import java.util.ArrayList;
import java.util.List;

@Slf4j
public abstract class AuthenticationBase {

    @Getter
    private final PathMatcher authPathMatcher;
    @Getter
    private final AccountLoader accountLoader;
    @Getter
    private final AuthTokenBuilder tokenBuilder;
    @Getter
    private final SecurityConfig securityConfig;
    @Getter
    protected boolean corsDisabled = true;

    public AuthenticationBase(AccountLoader accountLoader, SecurityConfig securityConfig, AuthTokenBuilder tokenBuilder) {
        this.accountLoader = accountLoader;
        this.securityConfig = securityConfig;
        this.tokenBuilder = tokenBuilder;
        //
        List<FunctionPermission> functionPermissionList = this.loadFunctionPermissionList();
        this.appendSecurityConfig(functionPermissionList);
        this.authPathMatcher = new PathMatcher(this.convertFunctionPermissionHandler(functionPermissionList));
    }

    /**
     * 将功能函数列表转换为处理器列表
     *
     * @param functionPermissionList
     * @return
     */
    protected List<FunctionPermissionHandler> convertFunctionPermissionHandler(List<FunctionPermission> functionPermissionList) {
        List<FunctionPermissionHandler> list = new ArrayList<>();
        for (FunctionPermission functionPermission : functionPermissionList) {
            list.add(new FunctionPermissionHandler(functionPermission));
        }
        return list;
    }

    /**
     * 将配置文件中的配置项附加到功能列表
     *
     * @param functionPermissionList
     */
    protected void appendSecurityConfig(List<FunctionPermission> functionPermissionList) {
        String pathAnonList = getSecurityConfig().getPathAnonList();
        String pathAuthList = getSecurityConfig().getPathAuthList();
        if (pathAnonList != null) {
            for (String path : pathAnonList.split(",")) {
                functionPermissionList.add(new FunctionPermission(path, "ANON", null));
            }
        }
        if (pathAuthList != null) {
            for (String path : pathAuthList.split(",")) {
                functionPermissionList.add(new FunctionPermission(path, "AUTH", null));
            }
        }
    }

    /**
     * 执行授权检查
     *
     * @param handler
     */
    public AuthenticationData authCheck(AuthenticationHandler handler) throws AuthenticationCheckerException {
        if (isCorsDisabled()) {
            if ("OPTIONS".equals(handler.getRequestMethod())) {
                // 放行所有 OPTIONS 请求
                return new AuthenticationData(0L, "");
            }
        }

        // 获取token
        String token = handler.getRequestToken();
        Long id = 0L;

        // 解析token
        if (StringUtils.hasText(token)) {
            AuthTokenInfo tokenInfo = this.tokenParser(token);
            if (tokenInfo != null) {
                //token过期
                if (!tokenInfo.isExpired()) {
                    id = tokenInfo.getId();
                }
            }
        }

        // 加载功能
        FunctionPermissionHandler functionPermissionHandler = this.findFunctionMatcher(handler);
        if (functionPermissionHandler == null) {
            // 请求的路径在配置中不存在，强制要求授权
            throw new AuthenticationCheckerException(AuthenticationCheckerResult.NO_PERMISSION);
        }

        // 匿名授权
        if (this.hasAnonymous(handler, functionPermissionHandler)) {
            return new AuthenticationData(0L, "");
        }

        // 未登录
        if (id == null || id == 0L) {
            throw new AuthenticationCheckerException(AuthenticationCheckerResult.NO_AUTH);
        }

        // 账号权限加载
        List<String> permissions = getAccountLoader().loadPermissions(id);
        if (permissions == null) {
            throw new AuthenticationCheckerException(AuthenticationCheckerResult.NO_PERMISSION);
        }

        // 权限验证
        if (!this.hasPermissions(handler, permissions, functionPermissionHandler)) {
            throw new AuthenticationCheckerException(AuthenticationCheckerResult.NO_PERMISSION);
        }

        // 填充安全上下文
        return new AuthenticationData(id, token);
    }

    /**
     * 未授权
     *
     * @param handler
     * @param functionPermissionHandler
     */
    protected boolean hasAnonymous(AuthenticationHandler handler, FunctionPermissionHandler functionPermissionHandler) {
        return functionPermissionHandler.hasPermission("ANON");
    }

    /**
     * 权限检查
     *
     * @param handler
     * @param permissions
     * @param functionPermissionHandler
     * @return
     */
    protected boolean hasPermissions(AuthenticationHandler handler, List<String> permissions, FunctionPermissionHandler functionPermissionHandler) {
        if (permissions == null) return false;
        if (permissions.size() == 0) return false;
        // 判定是否为授权要求
        if ("AUTH".equals(functionPermissionHandler.getFunctionPermission().getPermission())) {
            return true;
        }
        //
        for (String permission : permissions) {
            if (permission != null && functionPermissionHandler.hasPermission(permission)) {
                return true;
            }
        }
        //
        return false;
    }

    /**
     * Token解析器
     *
     * @param token
     * @return
     */
    protected AuthTokenInfo tokenParser(String token) {
        AuthTokenInfo tokenInfo;
        try {
            tokenInfo = getTokenBuilder().decode(token);
        } catch (Exception ex) {
            tokenInfo = null;
        }
        return tokenInfo;
    }

    /**
     * 查找功能匹配器
     *
     * @return
     */
    protected FunctionPermissionHandler findFunctionMatcher(AuthenticationHandler handler) {
        String path = handler.getRequestPath();
        String method = handler.getRequestMethod();
        //
        if ("".equals(path)) path = "/";
        // 加载功能列表
        return this.findFunctionMatcher(method, path);
    }

    /**
     * 查找功能匹配器
     *
     * @return
     */
    protected FunctionPermissionHandler findFunctionMatcher(String method, String path) {
        return getAuthPathMatcher().match(method, path);
    }

    /**
     * 加载系统配置的功能权限列表
     *
     * @return
     */
    protected List<FunctionPermission> loadFunctionPermissionList() {
        List<FunctionPermission> functionPermissionList = getAccountLoader().loadFunctionPermission();
        if (functionPermissionList == null) {
            functionPermissionList = new ArrayList<>();
        }
        return functionPermissionList;
    }
}
