package security.processor;

import framework.config.SecurityConfig;
import framework.security.AccountLoader;
import framework.utils.JsonUtil;
import framework.vo.ResultInfo;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.authentication.LockedException;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.web.authentication.AuthenticationFailureHandler;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;
import security.exceptions.CaptchaException;
import security.service.CaptchaFailCache;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.nio.charset.Charset;

/**
 * 授权失败
 */
@Slf4j
@Component
public class AuthFailedProcessor implements AuthenticationFailureHandler {

    @Autowired
    private AccountLoader accountLoader;
    @Autowired
    private CaptchaFailCache captchaFailCache;

    @Autowired
    private SecurityConfig securityConfig;

    @Override
    public void onAuthenticationFailure(HttpServletRequest request, HttpServletResponse response, AuthenticationException exception) throws IOException, ServletException {
        ResultInfo resultInfo = new ResultInfo().failed(exception.getMessage());
        String username = request.getParameter("username");

        //不记录失败记录的异常类型
        boolean noLog = exception instanceof LockedException || exception instanceof CaptchaException;

        // 记录登录失败记录
        if (!noLog) {
            if (StringUtils.hasText(username)) {
                accountLoader.loginUnsuccessful(username, exception.getMessage());
            }
        }

        // 记录登录异常，要求下次登录使用验证码
        if (StringUtils.hasText(username)) {
            this.captchaFailCache.failed(username);
        }

        //转向
        if (StringUtils.hasLength(securityConfig.getRedirectOnAuthFailed())) {
            response.sendRedirect(securityConfig.getRedirectOnAuthFailed());
            return;
        }

        //
        String s = JsonUtil.toJsonString(resultInfo);
        response.setContentType("application/json");
        response.setCharacterEncoding(Charset.defaultCharset().displayName());
        response.getWriter().write(s);
    }
}
