package security.service;

import framework.exceptions.NotFoundException;
import framework.security.Account;
import framework.security.AccountLoader;
import framework.security.AuthInfo;
import framework.security.FunctionPermission;
import lombok.extern.slf4j.Slf4j;

import java.util.Collections;
import java.util.List;

/**
 * 常规用户安全组件用户的加载器
 */
@Slf4j
public abstract class GeneralAccountLoader implements AccountLoader {

    @Override
    public abstract Account loadUserByUsername(String username);

    @Override
    public abstract Account loadUserById(Long id);

    /**
     * 加载用户的权限列表
     *
     * @param id
     * @return
     */
    @Override
    public List<String> loadPermissions(Long id) {
        //常规用户无权限描述
        return Collections.EMPTY_LIST;
    }

    /**
     * 加载功能权限表(即菜单)
     *
     * @return
     */
    @Override
    public List<FunctionPermission> loadFunctionPermission() {
        //常规用户无功能权限描述
        return Collections.EMPTY_LIST;
    }

    /**
     * 加载所有的权限
     *
     * @return
     */
    @Override
    public List<String> loadAllPermission() {
        //常规用户无
        return Collections.EMPTY_LIST;
    }

    /**
     * 加载登入信息
     *
     * @param id
     * @return
     * @throws NotFoundException
     */
    @Override
    public AuthInfo loadAuthInfo(Long id) {
        Account account = this.loadUserById(id);
        if (account == null)
            return null;
        List<String> permissions = this.loadPermissions(id);
        return new AuthInfo(account, permissions);
    }

    /**
     * 因登录失败次数超限已被限制登录的时间秒数
     *
     * @param username
     * @return
     */
    @Override
    public abstract int loginFailLimit(String username);

    /**
     * 登录成功回调
     *
     * @param id
     * @param username
     */
    @Override
    public abstract void loginSuccessful(String username, Long id);

    /**
     * 登录失败回调
     *
     * @param username
     * @param cause
     */
    @Override
    public abstract void loginUnsuccessful(String username, String cause);

    /**
     * 登出完成回调
     *
     * @param id
     */
    @Override
    public abstract void logoutSuccessful(Long id);

}
