package security.processor;

import framework.config.SecurityConfig;
import framework.security.Account;
import framework.security.AccountLoader;
import framework.security.token.AuthTokenBuilder;
import framework.security.token.AuthTokenInfo;
import framework.utils.JsonUtil;
import framework.vo.ResultInfo;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.userdetails.UsernameNotFoundException;
import org.springframework.security.web.authentication.AuthenticationSuccessHandler;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;
import security.service.CaptchaFailCache;
import security.vo.LoginInfo;
import security.vo.UserDetail;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.nio.charset.Charset;
import java.util.Date;

/**
 * 授权成功
 */
@Slf4j
@Component
@SuppressWarnings("unchecked")
public class AuthSuccessProcessor implements AuthenticationSuccessHandler {

    @Autowired
    private AccountLoader accountLoader;
    @Autowired
    private AuthTokenBuilder authTokenBuilder;
    @Autowired
    private SecurityConfig securityConfig;
    @Autowired
    private CaptchaFailCache captchaFailCache;

    @Override
    public void onAuthenticationSuccess(HttpServletRequest request, HttpServletResponse response, Authentication authentication) throws IOException, ServletException {
        UserDetail detail = (UserDetail) authentication.getPrincipal();
        if (detail == null)
            throw new UsernameNotFoundException("System error, not found ud");

        Account account = detail.getAccount();
        String token = this.createToken(account.getId());

        //更新数据库
        this.accountLoader.loginSuccessful(account.getUsername(), account.getId());

        //输出登录信息
        LoginInfo loginInfo = new LoginInfo();
        loginInfo.setId(account.getId());
        loginInfo.setName(account.getName());
        loginInfo.setUsername(account.getUsername());
        loginInfo.setToken(token);

        //移除可能存在的登录异常验证码要求记录
        this.captchaFailCache.remove(account.getUsername());

        //转向
        if (StringUtils.hasLength(securityConfig.getRedirectOnAuthSuccess())) {
            response.sendRedirect(securityConfig.getRedirectOnAuthSuccess());
            return;
        }

        //
        ResultInfo<LoginInfo> resultInfo = new ResultInfo(loginInfo);
        //
        String s = JsonUtil.toJsonString(resultInfo);
        response.setContentType("application/json");
        response.setCharacterEncoding(Charset.defaultCharset().displayName());
        response.getWriter().write(s);
    }

    /**
     * 创建一个新的token
     *
     * @return
     */
    private String createToken(Long userId) {
        Date now = new Date();
        Date expired = new Date(now.getTime() + this.securityConfig.getTokenSeconds() * 1000);
        //
        String token = this.authTokenBuilder.encode(new AuthTokenInfo(userId, expired));
        return token;
    }

}
