package security.config;

import framework.captcha.Captcha;
import framework.captcha.CaptchaSimple;
import framework.config.SecurityConfig;
import framework.exceptions.ConfigurationException;
import framework.runtime.SystemContext;
import framework.security.AccountLoader;
import framework.security.AuthService;
import framework.security.FunctionPermission;
import framework.security.password.PasswordService;
import framework.security.token.AuthTokenBuilder;
import lombok.extern.slf4j.Slf4j;
import org.springframework.cache.Cache;
import org.springframework.cache.CacheManager;
import org.springframework.http.HttpMethod;
import org.springframework.security.config.annotation.web.builders.HttpSecurity;
import org.springframework.security.config.http.SessionCreationPolicy;
import org.springframework.security.web.AuthenticationEntryPoint;
import org.springframework.security.web.SecurityFilterChain;
import org.springframework.security.web.access.AccessDeniedHandler;
import org.springframework.security.web.authentication.AuthenticationFailureHandler;
import org.springframework.security.web.authentication.AuthenticationSuccessHandler;
import org.springframework.security.web.authentication.UsernamePasswordAuthenticationFilter;
import org.springframework.security.web.authentication.logout.LogoutSuccessHandler;
import org.springframework.util.StringUtils;
import security.filters.TokenAuthenticationFilter;
import security.processor.*;
import security.service.AuthServiceImpl;
import security.service.PasswordServiceImpl;

import javax.servlet.Filter;
import java.util.ArrayList;
import java.util.List;

/**
 * 安全配置
 */
@Slf4j
public abstract class SecurityConfiguration {

    /**
     * 创建新的密码服务
     *
     * @return
     */
    public PasswordService newPasswordService(SecurityConfig securityConfig) {
        return new PasswordServiceImpl();
    }

    /**
     * 创建Token构建器
     *
     * @param securityConfig
     * @return
     */
    public AuthTokenBuilder newTokenBuilder(SecurityConfig securityConfig) {
        String secret = securityConfig.getTokenSecret();
        return new AuthTokenBuilder(secret);
    }

    /**
     * 创建新的授权服务
     *
     * @return
     */
    public AuthService newAuthService(SecurityConfig securityConfig) {
        return new AuthServiceImpl();
    }

    /**
     * 创建新的账号加载服务
     *
     * @return
     */
    public abstract AccountLoader newAccountLoader(SecurityConfig securityConfig);

    /**
     * 创建授权失败处理器
     *
     * @return
     */
    protected AuthenticationFailureHandler loadAuthenticationFailureHandler(SecurityConfig securityConfig) {
        return SystemContext.getBean(AuthFailedProcessor.class);
    }

    /**
     * 创建授权失败处理器
     *
     * @return
     */
    protected AuthenticationSuccessHandler loadAuthenticationSuccessHandler(SecurityConfig securityConfig) {
        return SystemContext.getBean(AuthSuccessProcessor.class);
    }

    /**
     * 创建登出处理器
     *
     * @return
     */
    protected LogoutSuccessHandler loadLogoutSuccessHandler(SecurityConfig securityConfig) {
        return SystemContext.getBean(LogoutSuccessProcessor.class);
    }

    /**
     * 创建授权失败处理器
     *
     * @return
     */
    protected AuthenticationEntryPoint loadAuthenticationEntryPoint(SecurityConfig securityConfig) {
        return SystemContext.getBean(NoAuthProcessor.class);
    }

    /**
     * 创建访问拒绝处理器
     *
     * @return
     */
    protected AccessDeniedHandler loadAccessDeniedHandler(SecurityConfig securityConfig) {
        return SystemContext.getBean(NoPermissionProcessor.class);
    }

    /**
     * 创建新验证服务
     *
     * @param cacheManager
     * @return
     */
    public Captcha newloadCaptcha(CacheManager cacheManager) {
        Cache cache = cacheManager.getCache("Captcha");
        Captcha captcha = new CaptchaSimple(cache);
//        Captcha captcha = new CaptchaBase64(cache);
        //captcha.getGenerator().setNoise(false);
        return captcha;
    }

    /**
     * 返回过滤链
     *
     * @param http
     * @return
     * @throws Exception
     */
    public SecurityFilterChain chain(HttpSecurity http) throws Exception {
        return http.build();
    }

    /**
     * 配置
     *
     * @param http
     * @throws Exception
     */
    public SecurityConfiguration configure(HttpSecurity http) throws Exception {

        SecurityConfig securityConfig = this.loadSecurityConfig();

        //开始一系列配置之前
        this.configure00Before(http, securityConfig);

        //CORS
        this.configure01CORS(http, securityConfig);

        //登入
        this.configure02Login(http, securityConfig);

        //登出
        this.configure03Logout(http, securityConfig);

        //异常
        this.configure04Exception(http, securityConfig);

        //功能权限配置
        this.configure05FunctionPermissions(http, securityConfig);

        //匿名访问
        this.configure06Anonymous(http, securityConfig);

//        //其它登录访问
//        http.authorizeRequests().anyRequest().authenticated();
        //未设置项的默认授权访问
        this.configureNoSetAuth(http, securityConfig);

        //禁止csrf
        this.configure07CSRF(http, securityConfig);

        //IFrame拒绝问题
        this.configure08Frame(http, securityConfig);

        //启用httpBasic
        this.configure09BasicAuth(http, securityConfig);

        //启用Bearer Token
        this.configure10BearerFilter(http, securityConfig);

        //启用Session
        this.configure11Session(http, securityConfig);

        //一系列配置之后
        this.configure99After(http, securityConfig);

        return this;
    }

    /**
     * 未设置项的默认授权机制
     *
     * @param http
     * @param securityConfig
     * @throws Exception
     */
    protected void configureNoSetAuth(HttpSecurity http, SecurityConfig securityConfig) throws Exception {
        http.authorizeRequests().anyRequest().denyAll();
    }

    /**
     * 一系列配置之前
     *
     * @param http
     * @param securityConfig
     */
    protected void configure00Before(HttpSecurity http, SecurityConfig securityConfig) {
    }

    /**
     * 一系列配置之后
     *
     * @param http
     * @param securityConfig
     */
    protected void configure99After(HttpSecurity http, SecurityConfig securityConfig) {
    }

    /**
     * 配置session选项
     *
     * @param http
     * @param securityConfig
     * @throws Exception
     */
    protected void configure11Session(HttpSecurity http, SecurityConfig securityConfig) throws Exception {
        if (securityConfig.getEnableSession()) {
            http.sessionManagement().sessionCreationPolicy(this.sessionPolicy(securityConfig));
        } else {
            http.sessionManagement().sessionCreationPolicy(SessionCreationPolicy.STATELESS);

        }
    }

    /**
     * 配置Bearer token过滤器
     *
     * @param http
     * @param securityConfig
     */
    protected void configure10BearerFilter(HttpSecurity http, SecurityConfig securityConfig) throws Exception {
        if (securityConfig.getEnableBearerToken())
            http.addFilterBefore(this.loadTokenAuthenticationFilter(securityConfig), UsernamePasswordAuthenticationFilter.class);
    }

    /**
     * 配置BasicAuth
     *
     * @param http
     * @param securityConfig
     * @throws Exception
     */
    protected void configure09BasicAuth(HttpSecurity http, SecurityConfig securityConfig) throws Exception {
        if (securityConfig.getEnableBasicAuth())
            http.httpBasic();
    }

    /**
     * 配置frame
     *
     * @param http
     * @param securityConfig
     * @throws Exception
     */
    protected void configure08Frame(HttpSecurity http, SecurityConfig securityConfig) throws Exception {
        http.headers().frameOptions().disable();
    }

    /**
     * 配置CSRF
     *
     * @param http
     * @param securityConfig
     */
    protected void configure07CSRF(HttpSecurity http, SecurityConfig securityConfig) throws Exception {
        if (securityConfig.getEnableCsrf())
            http.csrf();
        else
            http.csrf().disable();
    }

    /**
     * 配置匿名访问清单
     *
     * @param http
     * @param securityConfig
     */
    protected void configure06Anonymous(HttpSecurity http, SecurityConfig securityConfig) throws Exception {

        //得到匿名访问清单
        List<String> anonList = this.loadAnonymousPaths(securityConfig);

        //添加配置文件配置的配置表
        String pathAnonList = securityConfig.getPathAnonList();
        if (StringUtils.hasText(pathAnonList)) {
            String[] items = pathAnonList.split(",");
            for (String item : items) {
                if (StringUtils.hasText(item)) {
                    anonList.add(item);
                }
            }
        }

        //应用
        http.authorizeRequests()
//                .antMatchers("/", "/login", "/register").permitAll()
//                .antMatchers("/favicon.ico","/robots.txt").permitAll()
//                .antMatchers("/js/**","/css/**","/img/**").permitAll()
//                .antMatchers("/files/**","/ui/**").permitAll()
//                .antMatchers("/swagger-ui/**").permitAll()
                .antMatchers(anonList.toArray(new String[0])).permitAll();

        //print
        for (String s : anonList) {
            log.info("anonymous path: {}", s);
        }

    }

    /**
     * 配置异常处理
     *
     * @param http
     * @param securityConfig
     * @throws Exception
     */
    protected void configure04Exception(HttpSecurity http, SecurityConfig securityConfig) throws Exception {
        http.exceptionHandling()
                .accessDeniedHandler(this.loadAccessDeniedHandler(securityConfig))
                .authenticationEntryPoint(this.loadAuthenticationEntryPoint(securityConfig));
    }

    /**
     * 配置登出处理
     *
     * @param http
     * @param securityConfig
     */
    protected void configure03Logout(HttpSecurity http, SecurityConfig securityConfig) throws Exception {
        http.logout()
                .logoutUrl(securityConfig.getLogoutPath())
                .logoutSuccessHandler(this.loadLogoutSuccessHandler(securityConfig));
    }

    /**
     * 配置登入处理
     *
     * @param http
     * @param securityConfig
     * @throws Exception
     */
    protected void configure02Login(HttpSecurity http, SecurityConfig securityConfig) throws Exception {
        http.formLogin()
                .loginPage(securityConfig.getLoginPath())
                .successHandler(this.loadAuthenticationSuccessHandler(securityConfig))
                .failureHandler(this.loadAuthenticationFailureHandler(securityConfig));
    }

    /**
     * 配置 cors
     *
     * @param http
     * @param securityConfig
     */
    protected void configure01CORS(HttpSecurity http, SecurityConfig securityConfig) throws Exception {
        http.cors();
    }

    /**
     * 启用session时的级别
     *
     * @param securityConfig
     * @return
     */
    protected SessionCreationPolicy sessionPolicy(SecurityConfig securityConfig) {
        return SessionCreationPolicy.IF_REQUIRED;
    }

    /**
     * 配置功能权限表
     */
    protected void configure05FunctionPermissions(HttpSecurity http, SecurityConfig securityConfig) throws Exception {

        //获取用户定义的配置表
        List<FunctionPermission> functionPermissions = this.loadFunctionPermissionList(securityConfig);

        //添加配置文件配置的配置表
        String pathAuthList = securityConfig.getPathAuthList();
        if (StringUtils.hasText(pathAuthList)) {
            String[] items = pathAuthList.split(",");
            for (String item : items) {
                if (StringUtils.hasText(item)) {
                    functionPermissions.add(new FunctionPermission(item, "AUTH", null));
                }
            }
        }

        //授权
        int counter = 0;
        if (functionPermissions != null) {
            for (FunctionPermission fp : functionPermissions) {
                //跳过未配置路径的项
                if (!StringUtils.hasText(fp.getPath())) continue;
                //跳过不是本项目的配置
                if (!fp.getPath().startsWith("/")) continue;
                //跳过未配置权限的项
                if (!StringUtils.hasText(fp.getPermission())) continue;

                //指定了方法
                if (StringUtils.hasText(fp.getPermission())) {
                    //
                    String[] permissions = fp.getPermission().split(",");
                    if (StringUtils.hasLength(fp.getMethod())) {
                        HttpMethod method = null;
                        try {
                            method = HttpMethod.valueOf(fp.getMethod());
                        } catch (Exception exception) {
                            throw new ConfigurationException("FunctionPermission invalid by " + fp.getPath(), exception);
                        }
                        if (permissions.length > 1)
                            http.authorizeRequests().antMatchers(method, fp.getPath()).hasAnyAuthority(permissions);
                        else if (fp.getPermission().equals("AUTH"))
                            http.authorizeRequests().antMatchers(fp.getMethod(), fp.getPath()).authenticated();
                        else if (fp.getPermission().equals("ANON"))
                            http.authorizeRequests().antMatchers(fp.getMethod(), fp.getPath()).anonymous();
                        else
                            http.authorizeRequests().antMatchers(method, fp.getPath()).hasAuthority(fp.getPermission());
                    } else {
                        if (permissions.length > 1)
                            http.authorizeRequests().antMatchers(fp.getPath()).hasAnyAuthority(permissions);
                        else if (fp.getPermission().equals("AUTH"))
                            http.authorizeRequests().antMatchers(fp.getPath()).authenticated();
                        else if (fp.getPermission().equals("ANON"))
                            http.authorizeRequests().antMatchers(fp.getPath()).anonymous();
                        else
                            http.authorizeRequests().antMatchers(fp.getPath()).hasAuthority(fp.getPermission());
                    }
                }
                //
                counter++;
            }
        }

        //
        log.info("Load {} function permission", counter);
    }

    /**
     * 加载功能权限表
     *
     * @param securityConfig
     * @return
     */
    protected List<FunctionPermission> loadFunctionPermissionList(SecurityConfig securityConfig) {
        AccountLoader accountLoader = this.loadUserLoader(securityConfig);
        List<FunctionPermission> functionPermissionList = accountLoader.loadFunctionPermission();
        return functionPermissionList;
    }

    /**
     * 加载匿名访问路径表
     *
     * @param securityConfig
     * @return
     */
    protected List<String> loadAnonymousPaths(SecurityConfig securityConfig) {
        List<String> list = new ArrayList<>();
        //index page
        list.add("/");
        //auth page
        //list.add("/login");
//        list.add("/register");
        //site resource
        list.add("/favicon.ico");
        list.add("/robots.txt");
        //static resource
        list.add("/js/**");
        list.add("/css/**");
        list.add("/img/**");
//        //files
//        list.add("/files/**");
        //components
        list.add("/swagger-ui/**");
        list.add("/swagger-resources/**");
        list.add("/v3/api-docs");
        list.add("/v2/api-docs");
        //
        return list;
    }

    /**
     * 加载Token过滤器
     *
     * @param securityConfig
     * @return
     */
    protected Filter loadTokenAuthenticationFilter(SecurityConfig securityConfig) {
        return SystemContext.getBean(TokenAuthenticationFilter.class);
    }

    /**
     * 加载用户加载器
     *
     * @param securityConfig
     * @return
     */
    protected AccountLoader loadUserLoader(SecurityConfig securityConfig) {
        return SystemContext.getBean(AccountLoader.class);
    }


    /**
     * 加载安全配置项
     *
     * @return
     */
    protected SecurityConfig loadSecurityConfig() {
        return SystemContext.getBean(SecurityConfig.class);
    }

}
