package security.filters;

import framework.exceptions.BusinessException;
import framework.security.Account;
import framework.security.AccountLoader;
import framework.security.AuthInfo;
import framework.security.token.AuthTokenBuilder;
import framework.security.token.AuthTokenInfo;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.web.authentication.WebAuthenticationDetailsSource;
import org.springframework.util.StringUtils;
import org.springframework.web.filter.OncePerRequestFilter;
import security.vo.UserAuthority;
import security.vo.UserDetail;

import javax.servlet.FilterChain;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.util.List;
import java.util.stream.Collectors;

/**
 * Token 认证
 */
@Slf4j
public abstract class TokenAuthenticationFilter extends OncePerRequestFilter {

    @Autowired
    private AuthTokenBuilder tokenBuilder;

    @Autowired
    private AccountLoader accountLoader;

    @Override
    protected void doFilterInternal(HttpServletRequest request, HttpServletResponse response, FilterChain filterChain) throws ServletException, IOException {

        Authentication authenticationOfChain = SecurityContextHolder.getContext().getAuthentication();
        if (authenticationOfChain != null && authenticationOfChain.isAuthenticated()) {
            if (authenticationOfChain.getPrincipal() instanceof UserDetail) {
                //已授权，下一个
                filterChain.doFilter(request, response);
                return;
            }

            throw new BusinessException("authentication principal not is UserDetail");
        }

        // get Token
        String token = getRequestToken(request);
        if (!StringUtils.hasLength(token)) {
            //无Token 传递到过滤链的下一个
            filterChain.doFilter(request, response);
            return;
        }

        //解析token
        AuthTokenInfo tokenInfo = this.tokenParser(token);

        //解析失败
        if (tokenInfo == null) {
            filterChain.doFilter(request, response);
            return;
        }

        //TOKEN 过期
        if (tokenInfo.isExpired()) {
            filterChain.doFilter(request, response);
            return;
        }

        //获取授权
        AuthInfo authInfo = this.accountLoader.loadAuthInfo(tokenInfo.getId());
        if (authInfo == null || authInfo.getAccount() == null) {
            filterChain.doFilter(request, response);
            return;
        }

        //检查用户状态
        Account account = authInfo.getAccount();
        try {
            account.statusCheck();
        } catch (Exception exception) {
            filterChain.doFilter(request, response);
            log.warn("Account " + account.getUsername() + "/" + account.getId() + " token check fail: " + exception.getMessage());
            return;
        }

        //获得授权
        List<UserAuthority> authorities = authInfo.getPermissions().stream().map(i -> new UserAuthority(i)).collect(Collectors.toList());

        //构建授权信息
        UserDetail detail = new UserDetail(account, authInfo.getPermissions());
        UsernamePasswordAuthenticationToken authentication = new UsernamePasswordAuthenticationToken(detail, token, authorities);
        authentication.setDetails(new WebAuthenticationDetailsSource().buildDetails(request));

        //把AuthenticationToken放到当前线程,表示认证完成
        SecurityContextHolder.getContext().setAuthentication(authentication);

        //执行下一链
        filterChain.doFilter(request, response);
    }

    /**
     * Token解析器
     *
     * @param token
     * @return
     */
    protected AuthTokenInfo tokenParser(String token) {
        AuthTokenInfo tokenInfo;
        try {
            tokenInfo = this.tokenBuilder.decode(token);
        } catch (Exception ex) {
            tokenInfo = null;
        }
        return tokenInfo;
    }

    /**
     * get token
     *
     * @param request
     * @return
     */
    protected abstract String getRequestToken(HttpServletRequest request);

}
