package security.provider;

import framework.captcha.Captcha;
import framework.config.SecurityConfig;
import framework.security.Account;
import framework.security.AccountLoader;
import framework.security.RegApproval;
import framework.security.password.PasswordService;
import framework.utils.RequestUtil;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.authentication.BadCredentialsException;
import org.springframework.security.authentication.DisabledException;
import org.springframework.security.authentication.LockedException;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.authentication.dao.DaoAuthenticationProvider;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.security.core.userdetails.UserDetailsService;
import org.springframework.stereotype.Component;
import org.springframework.util.StringUtils;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;
import security.exceptions.CaptchaException;
import security.vo.UserDetail;

import javax.servlet.http.HttpServletRequest;

/**
 * 登入处理器(附加验证码，验证)
 */
@Component
public class AuthenticationProvider extends DaoAuthenticationProvider {

    @Autowired
    private Captcha captcha;
    @Autowired
    private SecurityConfig securityConfig;
    @Autowired
    private PasswordService passwordService;
    @Autowired
    private AccountLoader accountLoader;

    public AuthenticationProvider(UserDetailsService userDetailsService) {
        super.setUserDetailsService(userDetailsService);
    }

    /**
     * 重载授权认证，加入验证码校验
     *
     * @param authentication
     * @return
     * @throws AuthenticationException
     */
    @Override
    public Authentication authenticate(Authentication authentication) throws AuthenticationException {
        //captcha check
        if (this.securityConfig.getEnableLoginCaptcha()) {
            HttpServletRequest request = ((ServletRequestAttributes) RequestContextHolder.getRequestAttributes()).getRequest();
            String _captcha = request.getParameter("captcha");
            String _captchaId = request.getParameter("captchaId");
            this.validateCaptcha(_captchaId, _captcha);
        }
        //login fail limit
        String username = (String) authentication.getPrincipal();
        int loginFailLimit = this.accountLoader.loginFailLimit(username);
        if (loginFailLimit > 0) {
            throw new LockedException(RequestUtil.getMessageDefault("security.loginFailLimit"
                    , "Login failure too many times, limit login {0} minutes"
                    , ((int)Math.ceil(loginFailLimit / 60d)) + ""));
        }
        //invoke base
        return super.authenticate(authentication);
    }

    /**
     * 重载附加检查
     *
     * @param userDetails
     * @param authentication
     * @throws AuthenticationException
     */
    @Override
    protected void additionalAuthenticationChecks(UserDetails userDetails, UsernamePasswordAuthenticationToken authentication) throws AuthenticationException {
        Account account = ((UserDetail) userDetails).getAccount();
        //password check
        if (authentication.getCredentials() == null) {
            this.logger.debug("Failed to authenticate since no credentials provided");
            throw new BadCredentialsException(this.messages.getMessage("AbstractUserDetailsAuthenticationProvider.badCredentials", "Bad credentials"));
        } else {

            //check password
            String rawPassword = authentication.getCredentials().toString();
            String salt = account.getPasswordSalt();
            String encodePassword = userDetails.getPassword();
            if (!this.passwordService.matched(rawPassword, salt, encodePassword)) {
                this.logger.debug("Failed to authenticate since password does not match stored value");
                throw new BadCredentialsException(this.messages.getMessage("AbstractUserDetailsAuthenticationProvider.badCredentials", "Bad credentials"));
            }

            //check registration approved
            if (account.getRegApproval() == null) {
            } else if (RegApproval.Agree.equals(account.getRegApproval())) {
            } else if (RegApproval.Reject.equals(account.getRegApproval())) {
                throw new DisabledException(RequestUtil.getMessageDefault("security.userRejectApproved", "Registration approval is rejected, the account is unavailable"));
            } else if (RegApproval.Waiting.equals(account.getRegApproval())) {
                throw new DisabledException(RequestUtil.getMessageDefault("security.userWaitingApproved", "Waiting for registration approval, the account is unavailable"));
            }
        }
    }

    /**
     * 验证码验证
     *
     * @param captchaId
     * @param captcha
     */
    protected void validateCaptcha(String captchaId, String captcha) {
        if (!StringUtils.hasText(captchaId)) {
            throw new CaptchaException("Not set captchaId");
        }

        if (!StringUtils.hasText(captcha)) {
            throw new CaptchaException(RequestUtil.getMessageDefault("security.captcha.empty", "Please input captcha code"));
        }

        boolean checkSuccess = this.captcha.check(captchaId, captcha);
        if (!checkSuccess) {
            throw new CaptchaException(RequestUtil.getMessageDefault("security.captcha.invalid", "Captcha code error"));
        }
    }
}
