package framework.utils;

import framework.client.WebUserAgent;
import framework.runtime.SystemContext;
import org.springframework.context.NoSuchMessageException;
import org.springframework.web.context.request.RequestAttributes;
import org.springframework.web.context.request.RequestContextHolder;

import javax.servlet.http.HttpServletRequest;
import java.util.Locale;

public final class RequestUtil {

    public static RequestAttributes getAttributes() {
        RequestAttributes attributes = RequestContextHolder.currentRequestAttributes();
        return attributes;
    }

    public static void setAttribute(String name, String obj) {
        getAttributes().setAttribute(name, obj,
                RequestAttributes.SCOPE_REQUEST);
    }

    public static Object getAttribute(String name) {
        Object obj = getAttributes().getAttribute(name,
                RequestAttributes.SCOPE_REQUEST);
        if (obj == null)
            return null;
        return obj;
    }

    /**
     * 获取用户代理信息
     *
     * @return
     */
    public static WebUserAgent getUserAgent() {
        HttpServletRequest request = ServletUtil.getRequest();
        return new WebUserAgent(request);
    }

    /**
     * 获取浏览器版本
     *
     * @return
     */
    public static String getBrowser() {
        return getUserAgent().getBrowser();
    }

    /**
     * 获取搭载浏览器的系统版本
     *
     * @return
     */
    public static String getOS() {
        return getUserAgent().getOs();
    }

    /**
     * 获取客户端真实ip
     *
     * @return
     */
    public static String getClientIp() {
        HttpServletRequest request = ServletUtil.getRequest();
        if (request != null) {
            return getClientIp(request);
        }
        return "unknown";
    }

    /**
     * 获取请求头
     *
     * @param name
     * @return
     */
    public static String getHeader(String name) {
        HttpServletRequest request = ServletUtil.getRequest();
        if (request != null) {
            return request.getHeader(name);
        }
        return "unknown";
    }

    /**
     * 获取客户端真实ip
     *
     * @param request
     * @return
     */
    public static String getClientIp(HttpServletRequest request) {
        String ip = request.getHeader("x-forwarded-for");
        if (ip == null || ip.length() == 0 || "unknown".equalsIgnoreCase(ip)) {
            ip = request.getHeader("HTTP_X_FORWARDED_FOR");
        }
        if (ip != null && ip.length() > 0 && !"unknown".equalsIgnoreCase(ip)) {
            return ip.split(",")[0].trim();
        }
        if (ip == null || ip.length() == 0 || "unknown".equalsIgnoreCase(ip)) {
            ip = request.getHeader("Proxy-Client-IP");
        }
        if (ip == null || ip.length() == 0 || "unknown".equalsIgnoreCase(ip)) {
            ip = request.getHeader("WL-Proxy-Client-IP");
        }
        if (ip == null || ip.length() == 0 || "unknown".equalsIgnoreCase(ip)) {
            ip = request.getHeader("HTTP_CLIENT_IP");
        }
        if (ip == null || ip.length() == 0 || "unknown".equalsIgnoreCase(ip)) {
            ip = request.getRemoteAddr();
        }
        return ip;
    }

    /**
     * 是否是Ajax异步请求
     *
     * @param request
     */
    public static boolean isAjaxRequest(HttpServletRequest request) {
        String accept = request.getHeader("accept");
        if (accept != null && accept.contains("application/json")) {
            return true;
        }

        String xRequestedWith = request.getHeader("X-Requested-With");
        if (xRequestedWith != null && xRequestedWith.contains("XMLHttpRequest")) {
            return true;
        }

        String uri = request.getRequestURI();
        if (uri.endsWith(".json")) {
            return true;
        }
        if (uri.endsWith(".xml")) {
            return true;
        }

        return false;
    }

    /**
     * 以浏览器请求区域语言获取一个语言文本
     *
     * @param code
     * @param args
     * @return
     * @throws NoSuchMessageException
     */
    public static String getMessage(String code, Object... args) throws NoSuchMessageException {
        HttpServletRequest request = ServletUtil.getRequest();
        Locale locale = request.getLocale();
        if (locale == null)
            locale = Locale.getDefault();
        String message = SystemContext.getMessage(code, args, locale);
        return message;
    }

    /**
     * 以浏览器请求区域语言获取一个语言文本
     *
     * @param code
     * @param args
     * @param defaultMessage
     * @return
     * @throws NoSuchMessageException
     */
    public static String getMessageDefault(String code, String defaultMessage, Object... args) throws NoSuchMessageException {
        HttpServletRequest request = ServletUtil.getRequest();
        Locale locale = request.getLocale();
        if (locale == null)
            locale = Locale.getDefault();
        String message = SystemContext.getMessage(code, args, defaultMessage, locale);
        return message;
    }

}
