package framework.utils;

import framework.exceptions.*;
import org.springframework.util.StringUtils;

import java.util.ArrayList;
import java.util.List;

/**
 * 错误处理工具
 */
public class ErrorUtil {

    /**
     * 获取最顶部异常
     *
     * @param e
     * @return
     */
    public static Throwable getCause(Throwable e) {
        Throwable cause1 = e;
        Throwable cause2 = e;
        while (true) {
            cause2 = cause1.getCause();
            if (cause2 == null)
                return cause1;
            cause1 = cause2;
        }
    }


    /**
     * 获取空引用顶部追踪
     *
     * @param e
     * @return
     */
    public static String getNullTrace(NullPointerException e) {
        return getNullTrace(e, 5);
    }

    /**
     * 获取空引用顶部追踪
     *
     * @param e
     * @param level
     * @return
     */
    public static String getNullTrace(NullPointerException e, int level) {
        List<String> items = new ArrayList<>();
        StackTraceElement[] stackTrace = e.getStackTrace();
        if (stackTrace != null && stackTrace.length > 0) {
            for (int i = 0; i < level && stackTrace.length > i; i++) {
                items.add(stackTrace[i].getFileName() + ":" + stackTrace[i].getLineNumber());
            }
        }
        return String.join(",", items);
    }

    /**
     * 请求错误
     *
     * @throws Error400Exception
     */
    public static void badRequest() throws Error400Exception {
        String message = RequestUtil.getMessageDefault("error.e400", "Bad Request");
        throw new Error400Exception(message);
    }

    /**
     * 请求错误
     *
     * @param message
     * @throws Error400Exception
     */
    public static void badRequest(String message) throws Error400Exception {
        throw new Error400Exception(message);
    }

    /**
     * 未授权
     *
     * @throws Error401Exception
     */
    public static void noAuthorization() throws Error401Exception {
        String message = RequestUtil.getMessageDefault("error.e401", "Unauthorized");
        throw new Error401Exception(message);
    }

    /**
     * 未授权
     *
     * @param message
     * @throws Error401Exception
     */
    public static void noAuthorization(String message) throws Error401Exception {
        throw new Error401Exception(message);
    }

    /**
     * 禁止访问
     *
     * @throws Error403Exception
     */
    public static void forbidden() throws Error403Exception {
        String message = RequestUtil.getMessageDefault("error.e403", "Forbidden");
        throw new Error403Exception(message);
    }

    /**
     * 禁止访问
     *
     * @param message
     * @throws Error403Exception
     */
    public static void forbidden(String message) throws Error403Exception {
        throw new Error403Exception(message);
    }

    /**
     * 未找到操作主体
     *
     * @throws Error404Exception
     */
    public static void notFound() throws Error404Exception {
        String message = RequestUtil.getMessageDefault("error.e404", "Not found");
        throw new Error404Exception(message);
    }

    /**
     * 未找到操作主体
     *
     * @throws Error404Exception
     */
    public static void notFound(String message) throws Error404Exception {
        throw new Error404Exception(message);
    }

    /**
     * 无权限
     *
     * @throws Error405Exception
     */
    public static void noAccess() throws Error405Exception {
        String message = RequestUtil.getMessageDefault("error.e405", "No Access");
        throw new Error405Exception(message);
    }

    /**
     * 无权限
     *
     * @param message
     * @throws Error405Exception
     */
    public static void noAccess(String message) throws Error405Exception {
        throw new Error405Exception(message);
    }

    /**
     * 服务器错误
     *
     * @throws WebException
     */
    public static void serverError() throws WebException {
        String message = RequestUtil.getMessageDefault("error.e500", "Server error");
        throw new WebException(message);
    }

    /**
     * 服务器错误
     *
     * @param message
     * @throws WebException
     */
    public static void serverError(String message) throws WebException {
        throw new WebException(message);
    }

    /**
     * 业务错误
     *
     * @param message
     * @throws BusinessException
     */
    public static void error(String message) throws BusinessException {
        throw new BusinessException(message);
    }

    /**
     * 业务错误
     *
     * @param i18n
     * @param defaultMessage
     * @throws BusinessException
     */
    public static void error(String i18n, String defaultMessage) throws BusinessException {
        String message = RequestUtil.getMessageDefault(i18n, defaultMessage);
        throw new BusinessException(message);
    }

    /**
     * 业务错误
     *
     * @param i18n
     * @param defaultMessage allow null
     * @param args
     * @throws BusinessException
     */
    public static void error(String i18n, String defaultMessage, Object... args) throws BusinessException {
        String message;
        if (StringUtils.hasText(defaultMessage))
            message = RequestUtil.getMessageDefault(i18n, defaultMessage, args);
        else
            message = RequestUtil.getMessage(i18n, args);
        throw new BusinessException(message);
    }
}
