package framework.snowflake;

/*
  * 19位长度
  * 1 - 41位 - 10位 - 12位
  * 0 - 41位 - 10位 - 12位
  *
    41 位的时间序列（精确到毫秒，41 位的长度可以使用 69 年）；
    10 位的机器标识（10 位的长度最多支持部署 1024 个节点）；
    12 位的计数顺序号（12 位的计数顺序号支持每个节点每毫秒产生 4096 个 ID 序号）最高位是符号位，始终为 0。
* */


/*
  * 16位长度
  * 1 - 31位 - 10位 - 12位
  * 0 - 31位 - 10位 - 12位
  *
    31 位的时间序列（精确到秒，31 位的长度可以使用 60 年）；
    10 位的机器标识（10 位的长度最多支持部署 1024 个节点）；
    12 位的计数顺序号（12 位的计数顺序号支持每个节点每毫秒产生 4096 个 ID 序号）最高位是符号位，始终为 0。
* */

/**
 * 雪花算法
 */
public class Snowflake16 implements NextId {
    /**
     * 工作id
     */
    private long workerId;
    /**
     * 数据id
     */
    private long datacenterId;
    /**
     * 12位的序列号
     */
    private long sequence;
    /**
     * 初始时间戳
     */
    private long twepoch = 1430413261L;
    //长度为5位
    private long workerIdBits = 5L;
    private long datacenterIdBits = 5L;
    /**
     * 最大值
     */
    private long maxWorkerId = -1L ^ (-1L << workerIdBits);
    private long maxDatacenterId = -1L ^ (-1L << datacenterIdBits);
    /**
     * 序列号id长度
     */
    private long sequenceBits = 12L;
    /**
     * 序列号最大值
     */
    private long sequenceMask = -1L ^ (-1L << sequenceBits);
    /**
     * 工作id需要左移的位数，12位
     */
    private long workerIdShift = sequenceBits;
    /**
     * 数据id需要左移位数 12+5=17位
     */
    private long datacenterIdShift = sequenceBits + workerIdBits;
    /**
     * 时间戳需要左移位数 12+5+5=22位
     */
    private long timestampLeftShift = sequenceBits + workerIdBits + datacenterIdBits;
    /**
     * 时间最大值
     */
    private long maxTimeValue = -1L ^ (-1L << 31);

    /**
     * 上次时间戳，初始值为负数
     */
    private long lastTimestamp = -1L;


    public Snowflake16(long workerId, long datacenterId) {
        long sequence = 0;
        this.init(workerId, datacenterId, sequence);
    }

    /**
     * 21, 10, 0
     *
     * @param workerId     21
     * @param datacenterId 10
     * @param sequence     0
     */
    public Snowflake16(long workerId, long datacenterId, long sequence) {
        this.init(workerId, datacenterId, sequence);
    }

    /**
     * 21, 10, 0
     *
     * @param workerId     21
     * @param datacenterId 10
     * @param sequence     0
     */
    private void init(long workerId, long datacenterId, long sequence) {
        // sanity check for workerId
        if (workerId > maxWorkerId || workerId < 0) {
            throw new IllegalArgumentException(String.format("worker Id can't be greater than %d or less than 0", maxWorkerId));
        }
        if (datacenterId > maxDatacenterId || datacenterId < 0) {
            throw new IllegalArgumentException(String.format("datacenter Id can't be greater than %d or less than 0", maxDatacenterId));
        }
        System.out.printf(
                "worker starting. time length %d, timestamp left shift %d, datacenter id bits %d, worker id bits %d, sequence bits %d, workerid %d, datacenterId %d",
                (64 - (timestampLeftShift + datacenterIdBits + workerIdBits + sequenceBits)),
                timestampLeftShift, datacenterIdBits, workerIdBits, sequenceBits, workerId, datacenterId
        );

        System.out.println("");

        this.workerId = workerId;
        this.datacenterId = datacenterId;
        this.sequence = sequence;
    }

    /**
     * ID生成
     *
     * @return
     */
    public synchronized long nextId() {
        long timestamp = timeGen();

        //获取当前时间戳如果小于上次时间戳，则表示时间戳获取出现异常
        if (timestamp < lastTimestamp) {
            System.err.printf("clock is moving backwards.  Rejecting requests until %d.", lastTimestamp);
            throw new RuntimeException(String.format("Clock moved backwards.  Refusing to generate id for %d milliseconds",
                    lastTimestamp - timestamp));
        }

        //获取当前时间戳如果等于上次时间戳
        //说明：还处在同一毫秒内，则在序列号加1；否则序列号赋值为0，从0开始。
        // 0  - 4095
        if (lastTimestamp == timestamp) {
            sequence = (sequence + 1) & sequenceMask;
            if (sequence == 0) {
                timestamp = tilNextMillis(lastTimestamp);
            }
        } else {
            sequence = 0;
        }

        //将上次时间戳值刷新
        lastTimestamp = timestamp;

        long t = timestamp - twepoch;
        if (t > maxTimeValue) {
            throw new RuntimeException(String.format("time value is reached maximum value %d",
                    maxTimeValue));
        }

        /*
         * 返回结果：
         * (timestamp - twepoch) << timestampLeftShift) 表示将时间戳减去初始时间戳，再左移相应位数
         * (datacenterId << datacenterIdShift) 表示将数据id左移相应位数
         * (workerId << workerIdShift) 表示将工作id左移相应位数
         * | 是按位或运算符，例如：x | y，只有当x，y都为0的时候结果才为0，其它情况结果都为1。
         * 因为个部分只有相应位上的值有意义，其它位上都是0，所以将各部分的值进行 | 运算就能得到最终拼接好的id
         */
        return (t << timestampLeftShift) |
                (datacenterId << datacenterIdShift) |
                (workerId << workerIdShift) |
                sequence;
    }

    /**
     * 获取时间戳，并与上次时间戳比较
     *
     * @param lastTimestamp
     * @return
     */
    private long tilNextMillis(long lastTimestamp) {
        long timestamp = timeGen();
        while (timestamp <= lastTimestamp) {
            timestamp = timeGen();
        }
        return timestamp;
    }

    /**
     * 获取系统时间戳
     *
     * @return
     */
    private long timeGen() {
        return System.currentTimeMillis() / 1000;
    }

    public static void main(String[] args) {
        SnowflakeWorker worker = new SnowflakeWorker(2, 2, 0);
        System.out.println(worker.nextId());

        Snowflake16 worker1 = new Snowflake16(2, 2, 0);
        System.out.println(worker1.nextId());
    }
}