package framework.base;

import com.github.pagehelper.PageHelper;
import com.github.pagehelper.PageInfo;
import framework.proxy.MapProperty;

import java.util.List;
import java.util.function.Function;

/**
 * 服务基础类型
 *
 * @param <T>
 * @param <K>
 */
public interface BaseService<T, K> {

    /**
     * 返回数据操作对象
     *
     * @return
     */
    BaseMapper<T, K> getMapper();

    /**
     * 依据查询参数分页查询数据
     *
     * @param param
     * @param pageIndex
     * @param pageSize
     * @return
     */
    default PageInfo<T> list(T param, Integer pageIndex, Integer pageSize) {
        PageHelper.startPage(pageIndex, pageSize);
        List<T> list = this.getMapper().list(param);
        return new PageInfo<>(list);
    }

    /**
     * 依据对象参数返回第一个匹配结果
     *
     * @param param
     * @return
     */
    default T first(T param) {
        PageHelper.startPage(1, 1, false);
        T result = this.one(param);
        return result;
    }

    /**
     * 依据对象参数返回结果只有唯一一条的对象数据
     *
     * @param param
     * @return
     */
    default T one(T param) {
        T result = this.getMapper().one(param);
        return result;
    }

    /**
     * 依据对象参数统计数据量
     *
     * @param param
     * @return
     */
    default Integer count(T param) {
        Integer result = this.getMapper().count(param);
        return result;
    }

    /**
     * 依据查询参数返回数据列表
     *
     * @param param
     * @return
     */
    default List<T> load(T param) {
        List<T> list = this.getMapper().load(param);
        return list;
    }

    /**
     * 依据查询参数返回数据列表
     *
     * @param param
     * @param pageIndex
     * @param pageSize
     * @return
     */
    default PageInfo<T> load(T param, Integer pageIndex, Integer pageSize) {
        PageHelper.startPage(pageIndex, pageSize);
        List<T> list = this.getMapper().load(param);
        return new PageInfo<>(list);
    }

    /**
     * 添加新数据
     *
     * @param param
     * @return
     */
    default Integer add(T param) {
        Integer result = this.getMapper().add(param);
        return result;
    }

    /**
     * 依据对象参数更新数据
     *
     * @param param
     * @return
     */
    default Integer update(T param) {
        Integer result = this.getMapper().update(param);
        return result;
    }

    /**
     * 依据对象参数删除匹配的数据
     *
     * @param param
     * @return
     */
    default Integer delete(T param) {
        return getMapper().delete(param);
    }

    /**
     * 依据对象参数查找数据，找到后调用 deleteById 删除数据
     *
     * @param param
     * @param finder ID值查找
     * @return
     */
    default Integer delete(T param, Function<T, K> finder) {
        int result = 0;
        List<T> list = this.getMapper().select(param);
        for (T item : list) {
            K k = finder.apply(item);
            result += this.deleteById(k);
        }
        return result;
    }

    /**
     * 依据主键删除数据
     *
     * @param id
     * @return
     */
    default Integer deleteById(K id) {
        Integer result = this.getMapper().deleteById(id);
        return result;
    }

    /**
     * 依据主键(联合主键)删除数据
     *
     * @param param
     * @return
     */
    default Integer deleteByPK(T param) {
        Integer result = this.getMapper().deleteByPrimaryKey(param);
        return result;
    }

    /**
     * 依据主键ID获取对象
     *
     * @param id
     * @return
     */
    default T loadById(K id) {
        T result = this.getMapper().loadById(id);
        return result;
    }

    /**
     * 依据主键(联合主键)获取对象
     *
     * @param param
     * @return
     */
    default T loadByIdPK(T param) {
        T result = this.getMapper().selectByPrimaryKey(param);
        return result;
    }

    /**
     * 依据ID列表返回数据
     *
     * @param ids
     * @return
     */
    default List<T> loadByIds(List<K> ids) {
        List<T> result = this.getMapper().loadByIds(ids);
        return result;
    }

    /**
     * 依据主键ID列表删除数据
     *
     * @param ids
     * @return
     */
    default Integer deleteByIds(List<K> ids) {
        int rows = 0;
        if (ids == null) return rows;
        for (K id : ids) {
            rows += this.deleteById(id);
        }
        return rows;
    }

    default int dynamicUpdateById(T obj) {
        if (!(obj instanceof MapProperty)) {
            throw new RuntimeException("parameter obj must use MapperUtil.dynamicEntity() created");
        }
        return this.getMapper().dynamicUpdateById(obj);
    }

    default int dynamicUpdate(T obj, T where) {
        if (!(obj instanceof MapProperty)) {
            throw new RuntimeException("parameter obj must use MapperUtil.dynamicEntity() created");
        }
        if (!(where instanceof MapProperty)) {
            throw new RuntimeException("parameter obj must use MapperUtil.dynamicEntity() created");
        }
        return this.getMapper().dynamicUpdate(obj, where);
    }

    default boolean exists(T param) {
        return loadPrimaryKeyFirst(param) != null;
    }

    default boolean existsByPK(T param) {
        return this.getMapper().loadPrimaryKeyByPK(param) != null;
    }

    default List<T> loadPrimaryKey(T param) {
        return this.getMapper().loadPrimaryKey(param);
    }

    default T loadPrimaryKeyFirst(T param) {
        PageHelper.startPage(1, 1, false);
        List<T> list = this.getMapper().loadPrimaryKey(param);
        if (list != null && !list.isEmpty()) {
            return list.get(0);
        }
        return null;
    }

}