package io.cucumber.junit;

import cucumber.runner.ThreadLocalRunnerSupplier;
import cucumber.runtime.CucumberException;
import cucumber.runtime.filter.Filters;
import cucumber.runtime.model.CucumberFeature;
import gherkin.ast.Feature;
import gherkin.events.PickleEvent;
import io.cucumber.junit.PickleRunners.PickleRunner;
import org.junit.runner.Description;
import org.junit.runner.notification.RunNotifier;
import org.junit.runners.ParentRunner;
import org.junit.runners.model.InitializationError;

import java.io.Serializable;
import java.net.URI;
import java.util.ArrayList;
import java.util.List;

import static io.cucumber.junit.PickleRunners.withNoStepDescriptions;
import static io.cucumber.junit.PickleRunners.withStepDescriptions;

final class FeatureRunnerPartitioned extends ParentRunner<PickleRunner> {
    private final List<PickleRunner> children = new ArrayList<>();

    private final CucumberFeature cucumberFeature;
    private Description description;
    private int cucumberPartitionQty;
    private int cucumberPartitionId;
    private boolean debug;

    FeatureRunnerPartitioned(CucumberFeature cucumberFeature, Filters filters, ThreadLocalRunnerSupplier runnerSupplier, JUnitOptions jUnitOptions) throws InitializationError {
        super(null);
        setPartitionInfo();
        this.cucumberFeature = cucumberFeature;
        buildFeatureElementRunners(filters, runnerSupplier, jUnitOptions);
    }

    private void setPartitionInfo() {
        try {
            cucumberPartitionQty = Integer.valueOf(System.getProperty("cucumber.partition.qty"));
            cucumberPartitionId = Integer.valueOf(System.getProperty("cucumber.partition.id"));
            debug = Boolean.valueOf(System.getProperty("cucumber.partition.debug","false"));
        } catch (RuntimeException e) {
            printPartitionInstructions();
            e.printStackTrace();
            throw new RuntimeException("CucumberPartition Not Properly Configured");
        }

        if(cucumberPartitionId < 1 ||
                cucumberPartitionQty < 1 ||
                cucumberPartitionId > cucumberPartitionQty) {
            printPartitionInstructions();
            throw new RuntimeException("CucumberPartition Not Properly Configured");
        }
    }

    private void printPartitionInstructions() {
        System.out.println("To use CucumberPartitioned, you must pass two system variables. For example:");
        System.out.println("-Dcucumber.partition.id=2 -Dcucumber.partition.qty=5");
        System.out.println("cucumber.partition.qty is the total number of partitions");
        System.out.println("cucumber.partition.id is the id of the particular partition");
        System.out.println("There should be an environment with in id from 1 through the qty of partitions.");
        System.out.println("cucumber.partition.id and cucumber.partition.qty must be present and greater than 0");
        System.out.println("cucumber.partition.id cannot be more than cucumber.partition.qty");
    }
    @Override
    protected String getName() {
        Feature feature = cucumberFeature.getGherkinFeature().getFeature();
        return feature.getKeyword() + ": " + feature.getName();
    }

    @Override
    public Description getDescription() {
        if (description == null) {
            description = Description.createSuiteDescription(getName(), new FeatureId(cucumberFeature));
            for (PickleRunner child : getChildren()) {
                description.addChild(describeChild(child));
            }
        }
        return description;
    }

    public boolean isEmpty() {
        return children.isEmpty();
    }

    @Override
    protected List<PickleRunner> getChildren() {
        return children;
    }

    @Override
    protected Description describeChild(PickleRunner child) {
        return child.getDescription();
    }

    @Override
    protected void runChild(PickleRunner child, RunNotifier notifier) {
        child.run(notifier);
    }

    private void buildFeatureElementRunners(Filters filters, ThreadLocalRunnerSupplier runnerSupplier, JUnitOptions jUnitOptions) {
        for (PickleEvent pickleEvent : cucumberFeature.getPickles()) {
            if (filters.matchesFilters(pickleEvent)) {
                try {
                    int assignToPartition = Math.abs(pickleEvent.pickle.getName().hashCode()) % cucumberPartitionQty + 1;
                    if(debug) {
                        System.out.print(cucumberPartitionId == assignToPartition ? "*" : " ");
                        System.out.print("Partition " + assignToPartition);
                        System.out.println(" assigned to " + pickleEvent.pickle.getName());
                    }

                    if(cucumberPartitionId == assignToPartition) {
                        if (jUnitOptions.stepNotifications()) {
                            PickleRunner picklePickleRunner;
                            picklePickleRunner = withStepDescriptions(runnerSupplier, pickleEvent, jUnitOptions);
                            children.add(picklePickleRunner);
                        } else {
                            PickleRunner picklePickleRunner;
                            picklePickleRunner = withNoStepDescriptions(cucumberFeature.getName(), runnerSupplier, pickleEvent, jUnitOptions);
                            children.add(picklePickleRunner);
                        }
                    }
                } catch (InitializationError e) {
                    throw new CucumberException("Failed to create scenario runner", e);
                }
            }
        }
    }

    private static final class FeatureId implements Serializable {
        private static final long serialVersionUID = 1L;
        private final URI uri;

        FeatureId(CucumberFeature feature) {
            this.uri = feature.getUri();
        }

        @Override
        public boolean equals(Object o) {
            if (this == o) return true;
            if (o == null || getClass() != o.getClass()) return false;
            FeatureId featureId = (FeatureId) o;
            return uri.equals(featureId.uri);

        }

        @Override
        public int hashCode() {
            return uri.hashCode();
        }

        @Override
        public String toString() {
            return uri.toString();
        }
    }

}
