/*
 * **********************************************************************
 * Copyright (c) 2022 .
 * All rights reserved.
 * 项目名称：common-spring-autoconfig
 * 项目描述：公共的工具集
 * 版权说明：本软件属andy.zhou(rjzjh@163.com)所有。
 * ***********************************************************************
 */
package net.wicp.tams.common.spring.autoconfig.service.websocket;

import java.io.IOException;
import java.io.OutputStream;
import java.lang.reflect.ParameterizedType;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import org.springframework.web.socket.TextMessage;
import org.springframework.web.socket.WebSocketSession;

import com.fasterxml.jackson.databind.ObjectMapper;

import lombok.extern.slf4j.Slf4j;
import net.wicp.tams.common.exception.ExceptAll;
import net.wicp.tams.common.exception.ProjectExceptionRuntime;
import net.wicp.tams.common.spring.autoconfig.beans.SocketParamBean;
import net.wicp.tams.common.spring.autoconfig.constant.ConstantWebSocket;

/***
 * 
 * @author Andy
 *
 * @param <P> 连接的参数
 * @param <C> 连接的信息
 */
@Slf4j
public abstract class WebServiceCommon<P extends SocketParamBean, C> implements IWebSSHService {

	@SuppressWarnings("unchecked")
	public Class<P> getPClass() {
		Class<P> tClass = (Class<P>) ((ParameterizedType) getClass().getGenericSuperclass())
				.getActualTypeArguments()[0];
		return tClass;
	}

	@SuppressWarnings("unchecked")
	public Class<C> getCClass() {
		Class<C> tClass = (Class<C>) ((ParameterizedType) getClass().getGenericSuperclass())
				.getActualTypeArguments()[0];
		return tClass;
	}

	// 存放ssh连接信息的map
	protected static Map<String, Object> sshMap = new ConcurrentHashMap<>();

	@Override
	public final void initConnection(WebSocketSession session) {
		String uuid = String.valueOf(session.getAttributes().get(ConstantWebSocket.USER_UUID_KEY));
		// 将这个ssh连接信息放入map中
		sshMap.put(uuid, initConnectionSub(session));
	}

	/***
	 * 处理客户端发送的数据
	 */
	@Override
	public final void recvHandle(String buffer, WebSocketSession session) {
		P msgInfo = packMsgInfo(buffer, session);
		C connectInfo = packConnectInfo(session);
		if (msgInfo == null || msgInfo.getOperate() == null) {
			close(session);
			throw new ProjectExceptionRuntime(ExceptAll.param_error, "Operate需要传值");
		}
		switch (msgInfo.getOperate()) {
		case connect:
			doConnect(msgInfo, connectInfo, session);
			break;
		case command:
			doCommand(msgInfo, connectInfo, session);
			break;
		default:
			log.error("不支持的操作");
			close(session);
			break;
		}
	}

	protected abstract void doConnect(P param, C connector, WebSocketSession session);

	protected abstract void doCommand(P param, C connector, WebSocketSession session);

	@Override
	public void sendMessage(WebSocketSession session, byte[] buffer) throws IOException {
		session.sendMessage(new TextMessage(buffer));
	}

	@Override
	public final void close(WebSocketSession session) {
		String userId = String.valueOf(session.getAttributes().get(ConstantWebSocket.USER_UUID_KEY));
		C connectInfo = packConnectInfo(session);
		if (connectInfo != null) {
			closeSub(connectInfo);
			// map中移除
			sshMap.remove(userId);
		}
	}

	protected P packMsgInfo(String buffer, WebSocketSession session) {
		ObjectMapper objectMapper = new ObjectMapper();
		P params = null;
		try {
			params = objectMapper.readValue(buffer, getPClass());
		} catch (IOException e) {
			String msg = String.format("转换参数，传过来的参数：%s ,参数类型：%s ,异常信息:%s", buffer, getPClass().getName(),
					e.getMessage());
			throw new ProjectExceptionRuntime(ExceptAll.param_error, msg, e);
		}
		return params;
	}

	@SuppressWarnings("unchecked")
	protected C packConnectInfo(WebSocketSession session) {
		String userId = String.valueOf(session.getAttributes().get(ConstantWebSocket.USER_UUID_KEY));
		C connectInfo = (C) sshMap.get(userId);
		return connectInfo;
	}

	protected void transCmd(OutputStream outputStream, String command) {
		if (outputStream != null) {
			try {
				outputStream.write(command.getBytes());
				outputStream.flush();
			} catch (Exception e) {
				log.error("发送命令失败:" + command, e);
			}
		}
	}

	public abstract C initConnectionSub(WebSocketSession session);

	public abstract void closeSub(C connectInfo);

}
