/*
 * **********************************************************************
 * Copyright (c) 2022 .
 * All rights reserved.
 * 项目名称：common
 * 项目描述：公共的工具集
 * 版权说明：本软件属andy.zhou(rjzjh@163.com)所有。
 * ***********************************************************************
 */
package net.wicp.tams.common.paimon;

import java.lang.reflect.Type;
import java.math.BigDecimal;
import java.sql.Timestamp;
import java.sql.Types;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.util.Date;

import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.flink.table.types.logical.DecimalType;
import org.apache.flink.table.types.logical.LocalZonedTimestampType;
import org.apache.flink.table.types.logical.LogicalType;
import org.apache.flink.table.types.logical.RowType.RowField;
import org.apache.paimon.data.BinaryString;
import org.apache.paimon.data.Decimal;
import org.apache.paimon.data.GenericRow;
import org.apache.paimon.types.DataType;
import org.apache.paimon.types.DataTypes;

import net.wicp.tams.common.apiext.DateUtil;
import net.wicp.tams.common.apiext.StringUtil;
import net.wicp.tams.common.binlog.alone.ListenerConf.ColumnType;
import net.wicp.tams.common.constant.DateFormatCase;
import net.wicp.tams.common.constant.Middleware;
import net.wicp.tams.common.constant.dic.intf.IEnumCombobox;
import net.wicp.tams.common.exception.ExceptAll;
import net.wicp.tams.common.exception.ProjectExceptionRuntime;

/***
 * 返回空值的暂不支持 参看：org.apache.flink.table.runtime.typeutils.InternalSerializers
 * 
 * @author Andy
 *
 */
//根类型
public enum FlinkTypeEnum implements IEnumCombobox {
	BOOLEAN("boolean型", ColumnType.BIT, new String[] {}, new String[] { "bit" }, Boolean.class,
			new IFlinkTypeCallback() {
				@Override
				public DataType getObj() {
					return DataTypes.BOOLEAN();
				}

				@Override
				public Object getDefaultValue(LogicalType logicalType) {
					return getValue(FlinkTypeEnum.BOOLEAN, "false", logicalType);
				}

				@Override
				public String getDataTypeStr() {
					return "BOOLEAN";
				}
			}),

	CHAR("CHAR", ColumnType.VARCHAR, new String[] {}, new String[] { "char" }, String.class, new IFlinkTypeCallback() {
		@Override
		public DataType getObj() {
			return DataTypes.STRING();
		}

		@Override
		public Object getDefaultValue(LogicalType logicalType) {
			return getValue(FlinkTypeEnum.CHAR, "N", logicalType);
		}

		@Override
		public String getDataTypeStr() {
			return "STRING";
		}
	}),

	VARCHAR("VARCHAR", ColumnType.VARCHAR, new String[] { "string" },
			new String[] { "varchar", "json", "text", "tinytext", "longtext" }, String.class, new IFlinkTypeCallback() {
				@Override
				public DataType getObj() {
					return DataTypes.STRING();
				}

				@Override
				public Object getDefaultValue(LogicalType logicalType) {
					return getValue(FlinkTypeEnum.VARCHAR, "NA", logicalType);
				}

				@Override
				public String getDataTypeStr() {
					return "STRING";
				}
			}),

	BINARY("BINARY", ColumnType.DATETIME, new String[] {}, new String[] { "binary" }, byte[].class,
			new IFlinkTypeCallback() {
				@Override
				public DataType getObj() {
					return DataTypes.BYTES();
				}

				@Override
				public Object getDefaultValue(LogicalType logicalType) {
					return getValue(FlinkTypeEnum.BINARY, "NA", logicalType);
				}

				@Override
				public String getDataTypeStr() {
					return "BYTES";
				}
			}),

	VARBINARY("VARBINARY", ColumnType.DATETIME, new String[] {}, new String[] { "blob" }, byte[].class,
			new IFlinkTypeCallback() {
				@Override
				public DataType getObj() {
					return DataTypes.BYTES();
				}

				@Override
				public Object getDefaultValue(LogicalType logicalType) {
					return getValue(FlinkTypeEnum.VARBINARY, "NA", logicalType);
				}

				@Override
				public String getDataTypeStr() {
					return "BYTES";
				}
			}),

	// FLINK-18580 ,官方建议在Flink 的建维表时将 BIGINT 定义为 DECIMAL(20,0)
	DECIMAL("DECIMAL", ColumnType.DECIMAL, new String[] {}, new String[] { "decimal" }, BigDecimal.class,
			new IFlinkTypeCallback() {
				@Override
				public DataType getObj(int precision, int scale) {
					return DataTypes.DECIMAL(precision, scale);
				}

				@Override
				public DataType getObj() {
//					return getObj(27, 4);
					throw new ProjectExceptionRuntime(ExceptAll.param_error, "decimal必须提供precision和scale");
				}

				@Override
				public Object getDefaultValue(LogicalType logicalType) {
					return getValue(FlinkTypeEnum.DECIMAL, "0", logicalType);
				}

				@Override
				public String getDataTypeStr() {
					return "DECIMAL";
				}
			}),

	// "tinyint"
	TINYINT("短整形（<255）", ColumnType.TINY, new String[] {}, new String[] {}, Integer.class, new IFlinkTypeCallback() {
		@Override
		public DataType getObj() {
			return DataTypes.TINYINT();
		}

		@Override
		public Object getDefaultValue(LogicalType logicalType) {
			return getValue(FlinkTypeEnum.TINYINT, "0", logicalType);
		}

		@Override
		public String getDataTypeStr() {
			return "TINYINT";
		}
	}),

//"smallint"	
	SMALLINT("小整型（<65535）", ColumnType.SHORT, new String[] {}, new String[] {}, Integer.class,
			new IFlinkTypeCallback() {
				@Override
				public DataType getObj() {
					return DataTypes.SMALLINT();
				}

				@Override
				public Object getDefaultValue(LogicalType logicalType) {
					return getValue(FlinkTypeEnum.SMALLINT, "0", logicalType);
				}

				@Override
				public String getDataTypeStr() {
					return "SMALLINT";
				}
			}),

	// 表中定义为tinyint,注意不要生成tinyint类型，因为写sql时flink默认转为INTEGER类型，意味着tinyint类型插入不会成功，报语法错误
	INTEGER("整型", ColumnType.INT24, new String[] { "int" }, new String[] { "integer", "int", "smallint", "tinyint" },
			Integer.class, new IFlinkTypeCallback() {
				@Override
				public DataType getObj() {
					return DataTypes.INT();
				}

				@Override
				public Object getDefaultValue(LogicalType logicalType) {
					return getValue(FlinkTypeEnum.INTEGER, "0", logicalType);
				}

				@Override
				public String getDataTypeStr() {
					return "INT";
				}
			}),
	// "bigint"见DECIMAL类型
	BIGINT("长整型", ColumnType.LONG, new String[] {}, new String[] { "bigint" }, Long.class, new IFlinkTypeCallback() {
		@Override
		public DataType getObj() {
			return DataTypes.BIGINT();
		}

		@Override
		public Object getDefaultValue(LogicalType logicalType) {
			return getValue(FlinkTypeEnum.BIGINT, "0", logicalType);
		}

		@Override
		public String getDataTypeStr() {
			return "BIGINT";
		}
	}),

	FLOAT("单精度浮点型", ColumnType.FLOAT, new String[] {}, new String[] { "float" }, Float.class, new IFlinkTypeCallback() {
		@Override
		public DataType getObj() {
			return DataTypes.FLOAT();
		}

		@Override
		public Object getDefaultValue(LogicalType logicalType) {
			return getValue(FlinkTypeEnum.FLOAT, "0", logicalType);
		}

		@Override
		public String getDataTypeStr() {
			return "FLOAT";
		}
	}),

	DOUBLE("双精度浮点型", ColumnType.DOUBLE, new String[] {}, new String[] { "double" }, Double.class,
			new IFlinkTypeCallback() {
				@Override
				public DataType getObj() {
					return DataTypes.DOUBLE();
				}

				@Override
				public Object getDefaultValue(LogicalType logicalType) {
					return getValue(FlinkTypeEnum.DOUBLE, "0", logicalType);
				}

				@Override
				public String getDataTypeStr() {
					return "DOUBLE";
				}
			}),

	DATE("日期", ColumnType.DATE, new String[] {}, new String[] {}, Integer.class, new IFlinkTypeCallback() {
		@Override
		public DataType getObj() {
			return DataTypes.DATE();
		}

		@Override
		public Object getDefaultValue(LogicalType logicalType) {
			return getValue(FlinkTypeEnum.DATE, "0", logicalType);
		}

		@Override
		public String getDataTypeStr() {
			return "DATE";
		}
	}),

	TIME_WITHOUT_TIME_ZONE("时间", ColumnType.DATE, new String[] {}, new String[] {}, Integer.class,
			new IFlinkTypeCallback() {
				@Override
				public DataType getObj() {
					return DataTypes.TIME();
				}

				@Override
				public Object getDefaultValue(LogicalType logicalType) {
					return getValue(FlinkTypeEnum.TIME_WITHOUT_TIME_ZONE, "0", logicalType);
				}

				@Override
				public String getDataTypeStr() {
					return "TIME";
				}
			}),

	// process()类型也是它。
	TIMESTAMP_WITH_LOCAL_TIME_ZONE("TIMESTAMP_WITH_LOCAL_TIME_ZONE", ColumnType.DATETIME, new String[] {},
			new String[] { "date", "datetime", "timestamp" }, Date.class, new IFlinkTypeCallback() {
				@Override
				public DataType getObj() {
					// DataTypes.TIMESTAMP()需要用org.apache.flink.table.data.TimestampData类型，转为Long
					// 使用TIMESTAMP_WITH_TIME_ZONE报SQL validation failed. Type is not supported:
					// TIMESTAMP_WITH_TIME_ZONE
					return DataTypes.TIMESTAMP(3);
				}

				@Override
				public Object getDefaultValue(LogicalType logicalType) {
					return getValue(FlinkTypeEnum.TIMESTAMP_WITH_LOCAL_TIME_ZONE, "1970-01-01", logicalType);
				}

				@Override
				public String getDataTypeStr() {
					return "TIMESTAMP_LTZ";
				}
			}),

	TIMESTAMP_WITH_TIME_ZONE("TIMESTAMP_WITH_TIME_ZONE", ColumnType.DATETIME, new String[] {}, new String[] {},
			Date.class, new IFlinkTypeCallback() {
				@Override
				public DataType getObj() {
					return DataTypes.TIMESTAMP(3);
				}

				@Override
				public Object getDefaultValue(LogicalType logicalType) {
					return getValue(FlinkTypeEnum.TIMESTAMP_WITH_TIME_ZONE, "1970-01-01", logicalType);
				}

				@Override
				public String getDataTypeStr() {
					return "TIMESTAMP";
				}
			}),

	TIMESTAMP_WITHOUT_TIME_ZONE("TIMESTAMP_WITHOUT_TIME_ZONE", ColumnType.DATETIME, new String[] {}, new String[] {},
			Date.class, new IFlinkTypeCallback() {
				@Override
				public DataType getObj() {
					return DataTypes.TIMESTAMP(3);
				}

				@Override
				public Object getDefaultValue(LogicalType logicalType) {
					return getValue(FlinkTypeEnum.TIMESTAMP_WITHOUT_TIME_ZONE, "1970-01-01", logicalType);
				}

				@Override
				public String getDataTypeStr() {
					return "TIMESTAMP";
				}
			}),

	INTERVAL_DAY_TIME("INTERVAL_DAY_TIME", ColumnType.LONG, new String[] {}, new String[] {}, Long.class,
			new IFlinkTypeCallback() {
				@Override
				public DataType getObj() {
					return DataTypes.BIGINT();
				}

				@Override
				public Object getDefaultValue(LogicalType logicalType) {
					return getValue(FlinkTypeEnum.INTERVAL_DAY_TIME, "0", logicalType);
				}

				@Override
				public String getDataTypeStr() {
					return "BIGINT";
				}
			}),

	INTERVAL_YEAR_MONTH("INTERVAL_YEAR_MONTH", ColumnType.SHORT, new String[] {}, new String[] {}, Integer.class,
			new IFlinkTypeCallback() {
				@Override
				public DataType getObj() {
					return DataTypes.INT();
				}

				@Override
				public Object getDefaultValue(LogicalType logicalType) {
					return getValue(FlinkTypeEnum.INTERVAL_YEAR_MONTH, "0", logicalType);
				}

				@Override
				public String getDataTypeStr() {
					return "INT";
				}
			}),

	ROW("ROW", ColumnType.VARCHAR, new String[] {}, new String[] {}, String.class, new IFlinkTypeCallback() {
		@Override
		public DataType getObj() {
			return null;
		}

		@Override
		public Object getDefaultValue(LogicalType logicalType) {
			return getValue(FlinkTypeEnum.ROW, "0", logicalType);
		}

		@Override
		public String getDataTypeStr() {
			return "ROW";
		}
	}),

	ARRAY("ARRAY", null, new String[] {}, new String[] {}, String.class, new IFlinkTypeCallback() {
		@Override
		public DataType getObj() {
			return null;
			// return DataTypes.ARRAY(elementDataType);
		}

		@Override
		public Object getDefaultValue(LogicalType logicalType) {
			return getValue(FlinkTypeEnum.ARRAY, "0", logicalType);
		}

		@Override
		public String getDataTypeStr() {
			// TODO Auto-generated method stub
			return null;
		}
	}),

	MAP("MAP", null, new String[] {}, new String[] {}, String.class, new IFlinkTypeCallback() {
		@Override
		public DataType getObj() {
			return null;// DataTypes.MAP(keyDataType, valueDataType);
		}

		@Override
		public Object getDefaultValue(LogicalType logicalType) {
			return getValue(FlinkTypeEnum.MAP, "0", logicalType);
		}

		@Override
		public String getDataTypeStr() {
			// TODO Auto-generated method stub
			return null;
		}
	}),

	MULTISET("MULTISET", null, new String[] {}, new String[] {}, String.class, new IFlinkTypeCallback() {
		@Override
		public DataType getObj() {
			return null;// DataTypes.MAP(keyDataType, valueDataType);
		}

		@Override
		public Object getDefaultValue(LogicalType logicalType) {
			return getValue(FlinkTypeEnum.MULTISET, "0", logicalType);
		}

		@Override
		public String getDataTypeStr() {
			// TODO Auto-generated method stub
			return null;
		}
	}),

	RAW("RAW", null, new String[] {}, new String[] {}, String.class, new IFlinkTypeCallback() {
		@Override
		public DataType getObj() {
			return null;// DataTypes.RAW(clazz);
		}

		@Override
		public Object getDefaultValue(LogicalType logicalType) {
			return getValue(FlinkTypeEnum.RAW, "0", logicalType);
		}

		@Override
		public String getDataTypeStr() {
			// TODO Auto-generated method stub
			return null;
		}
	}),

	STRUCTURED_TYPE("STRUCTURED_TYPE", null, new String[] {}, new String[] {}, String.class, new IFlinkTypeCallback() {
		@Override
		public DataType getObj() {
			return null;// DataTypes.RAW(clazz);
		}

		@Override
		public Object getDefaultValue(LogicalType logicalType) {
			return getValue(FlinkTypeEnum.STRUCTURED_TYPE, "0", logicalType);
		}

		@Override
		public String getDataTypeStr() {
			// TODO Auto-generated method stub
			return null;
		}
	}),

	DISTINCT_TYPE("DISTINCT_TYPE", null, new String[] {}, new String[] {}, String.class, new IFlinkTypeCallback() {
		@Override
		public DataType getObj() {
			return null;// DataTypes.RAW(clazz);
		}

		@Override
		public Object getDefaultValue(LogicalType logicalType) {
			return getValue(FlinkTypeEnum.DISTINCT_TYPE, "0", logicalType);
		}

		@Override
		public String getDataTypeStr() {
			// TODO Auto-generated method stub
			return null;
		}
	}),

	SYMBOL("SYMBOL", null, new String[] {}, new String[] {}, String.class, new IFlinkTypeCallback() {
		@Override
		public DataType getObj() {
			return null;// DataTypes.RAW(clazz);
		}

		@Override
		public Object getDefaultValue(LogicalType logicalType) {
			return getValue(FlinkTypeEnum.SYMBOL, "0", logicalType);
		}

		@Override
		public String getDataTypeStr() {
			// TODO Auto-generated method stub
			return null;
		}
	}),

	UNRESOLVED("UNRESOLVED", null, new String[] {}, new String[] {}, String.class, new IFlinkTypeCallback() {
		@Override
		public DataType getObj() {
			return null;// DataTypes.RAW(clazz);
		}

		@Override
		public Object getDefaultValue(LogicalType logicalType) {
			return getValue(FlinkTypeEnum.UNRESOLVED, "0", logicalType);
		}

		@Override
		public String getDataTypeStr() {
			// TODO Auto-generated method stub
			return null;
		}
	});

	private final String desc;
	private final String[] mysqlTypes;
	private final String[] logicType;// flink的逻辑类型
	private final ColumnType columnType;// 用于转为duckulaevent序列化

	private final int defaultPrecision = 38;// 默认的精度 decemal使用

	private final int defaultScale = 4;// 默认的精度decemal使用

	private final Type javaType;// java类型
	private final IFlinkTypeCallback convert;

	public IFlinkTypeCallback getConvert() {
		return convert;
	}

	public String[] getMysqlTypes() {

		return mysqlTypes;
	}

	private FlinkTypeEnum(String desc, ColumnType columnType, String[] logicType, String[] mysqlTypes, Type javaType,
			IFlinkTypeCallback convert) {
		this.desc = desc;
		this.mysqlTypes = mysqlTypes;
		this.javaType = javaType;
		this.convert = convert;
		this.logicType = logicType;
		this.columnType = columnType;
	}

	@Override
	public String getDesc() {
		return desc;
	}

	@Override
	public String getName() {
		return this.name();
	}

	@Override
	public String getDesc_zh() {
		return this.desc;
	}

	@Override
	public String getDesc_en() {
		return this.name();
	}

	public String getStr(Object obj) {
//		Type javaType = this.getJavaType();
		// 先不区分
		return String.valueOf(obj);
	}

	/***
	 * 通过传过来的字符类型得到DataTypeEnum， 注意：TIMESTAMP WITH LOCAL TIME_ZONE 匹配
	 * TIMESTAMP_WITH_LOCAL_TIME_ZONE
	 * TIMESTAMP(3)匹配TIMESTAMP_WITHOUT_TIME_ZONE，那如何匹配TIMESTAMP_WITH_LOCAL_TIME_ZONE？待解决
	 * 
	 * @param flinkRowType 字符类型
	 * @return
	 */
	public static FlinkTypeEnum findByFlinkRowType(String flinkRowType) {

		if (StringUtils.isEmpty(flinkRowType)) {
			return null;
		}
		for (FlinkTypeEnum ele : FlinkTypeEnum.values()) {
			if (StringUtil.blankToChart(flinkRowType, "_", true).equalsIgnoreCase(ele.name())) {
				return ele;
			}
		}
		return null;
	}

	public static FlinkTypeEnum findByMysqlType(String datatype, String columnType, Middleware middleware) {
		if (StringUtil.isNull(datatype)) {
			return FlinkTypeEnum.VARCHAR;
		}
		// 特殊处理
		// https://nightlies.apache.org/flink/flink-docs-release-1.13/zh/docs/connectors/table/jdbc/#%E6%95%B0%E6%8D%AE%E7%B1%BB%E5%9E%8B%E6%98%A0%E5%B0%84
		String mysqltypelower = datatype.toLowerCase();
		if (middleware == Middleware.mysql && "tinyint(1)".equalsIgnoreCase(columnType)) {// 在mysql里
																							// TINYINT(1)也是Boolean型
			return FlinkTypeEnum.BOOLEAN;
		}
//203230223  不支持旧版本
//		else if (middleware == Middleware.doris
//				&& ("date".equals(mysqltypelower) || "datetime".equals(mysqltypelower))) {// connector0.15是varchar的，但到了connector1.1后面就是datetime类型了。暂时保持吧。
//			return FlinkTypeEnum.VARCHAR;
//		}
		for (FlinkTypeEnum ele : FlinkTypeEnum.values()) {
			if (ArrayUtils.contains(ele.getMysqlTypes(), mysqltypelower)) {
				return ele;
			}
		}
		return FlinkTypeEnum.VARCHAR;
	}

	// 通过逻辑类型定位
	public static FlinkTypeEnum findByLogicType(String logicType) {
		if (StringUtil.isNull(logicType)) {
			return null;
		}
		for (FlinkTypeEnum ele : FlinkTypeEnum.values()) {
			if (ArrayUtils.contains(ele.getLogicType(), logicType.toLowerCase())) {
				return ele;
			}
		}
		return null;
	}

	public Type getJavaType() {
		return javaType;
	}

	// private static DateTimeFormatter dtf =
	// DateTimeFormatter.ofPattern(DateFormatCase.YYYY_MM_DD_hhmmss.getPattern());

	/****
	 * 精度一定要算对了，所以引入logicalType，否则会出现141.00变为14100.00的情况，因为进去的时候是 (38,4)即为：141.0000
	 * ，出来时变为（20,2）则会变为14100.00
	 * 
	 * @param <T>
	 * @param value
	 * @param logicalType
	 * @return
	 */
	@SuppressWarnings("unchecked")
	public static <T> T getValue(FlinkTypeEnum flinkTypeEnum, String value, LogicalType logicalType) {
		Object retobj = null;
		if (StringUtil.isNull(value)) {
			return null;
		}
		if ("java.lang.String".equals(flinkTypeEnum.getJavaType().getTypeName())) {
			retobj =   BinaryString.fromString(value);
			return (T) retobj;
		}
		switch (flinkTypeEnum) {
		case CHAR:
		case VARCHAR:
			retobj = value;
		case BOOLEAN:
			value = "1".equals(value) ? "true" : "false";
			retobj = Boolean.valueOf(value);
			break;
		case BINARY:
		case VARBINARY:
			retobj = value.getBytes();
			break;
		case DECIMAL:
			BigDecimal bd = new BigDecimal(value);
			DecimalType typeTrue = (DecimalType) logicalType;
			retobj =  Decimal.fromBigDecimal(bd, typeTrue.getPrecision(), typeTrue.getScale());
			break;
		case TINYINT:
			retobj = Byte.valueOf(value);
			break;
		case SMALLINT:
			retobj = Short.valueOf(value);
			break;
		case INTEGER:
		case TIME_WITHOUT_TIME_ZONE:
		case INTERVAL_YEAR_MONTH:
			retobj = Integer.valueOf(value);
			break;
		case DATE:
			retobj = DateUtil.objToDate(value);
			// (int) (((Date) val).toLocalDate().toEpochDay());

		case BIGINT:
		case INTERVAL_DAY_TIME:
			retobj = Long.valueOf(value);
			break;
		case FLOAT:
			retobj = Float.valueOf(value);
			break;
		case DOUBLE:
			retobj = Double.valueOf(value);
			break;
		case TIMESTAMP_WITHOUT_TIME_ZONE:// 标准时间，把东8区转为标准时间
//			Timestamp time = new Timestamp(DateUtil
//					.changeTimeZone(DateUtil.objToDate(value), DateUtil.getBeijingTimeZone(), DateUtil.getTZZone())
//					.getTime());
//			retobj = TimestampData.fromEpochMillis(time.getTime());
			Timestamp time = new Timestamp(DateUtil.objToDate(value).getTime());
			retobj =  org.apache.paimon.data.Timestamp.fromEpochMillis(time.getTime());
			break;
		case TIMESTAMP_WITH_LOCAL_TIME_ZONE:
			// LocalDateTime只是在本地select显示 2022-09-09 02:12:29.305 但实际上是2022-09-08 18:12:29
			// 使用了TIMESTAMP_LTZ类型就不需要加28800000，系统会自动换算，这样与TIMESTAMP_WITHOUT_TIME_ZONE处理逻辑一样了
			Date objToDate = DateUtil.objToDate(value);
			// 这是flink的标准输出
			// 会丢失精度
//			LocalDateTime localDateTime = LocalDateTime
//					.parse(DateFormatCase.YYYY_MM_DD_hhmmss.getInstanc().format(objToDate), dtf);

			// 20221201前使用
			LocalDateTime localDateTime = objToDate.toInstant().atZone(ZoneId.systemDefault()).toLocalDateTime();
			retobj = org.apache.paimon.data.Timestamp.fromLocalDateTime(localDateTime);

//			retobj = TimestampData.fromInstant(objToDate.toInstant());

//			retobj = TimestampData.fromEpochMillis(objToDate.getTime());	//会导致时间多8小时		
//			LocalDateTime.ofEpochSecond(epochSecond, nanoOfSecond, offset)
//			TimestampData.fromLocalDateTime(DateUtil.objToDate(value));
//			retobj = TimestampData.fromEpochMillis(time2.getTime());// + 28800000
			break;
		case TIMESTAMP_WITH_TIME_ZONE:
		case ARRAY:
		case MULTISET:
		case MAP:
		case ROW:
		case STRUCTURED_TYPE:
		case DISTINCT_TYPE:
		case RAW:
		case SYMBOL:
		case UNRESOLVED:
			throw new UnsupportedOperationException();
		default:
			retobj = value;
			break;
		}
		T retobj2 = (T) retobj;
		return retobj2;
	}

	public String[] getLogicType() {
		return logicType;
	}

	public ColumnType getColumnType() {
		return this.columnType;
	}

	public int getSqlType() {
		String dataTypeStr = this.getConvert().getDataTypeStr();
		if (dataTypeStr == null) {
			return Types.VARCHAR;
		}
		int ret = Types.VARCHAR;
		switch (dataTypeStr) {
		case "STRING":
			ret = Types.VARCHAR;
			break;
		case "DECIMAL":
			ret = Types.DECIMAL;
			break;
		case "BYTES":
			ret = Types.BINARY;
			break;
		case "BOOLEAN":
			ret = Types.BOOLEAN;
			break;
		case "TINYINT":
			ret = Types.TINYINT;
			break;
		case "SMALLINT":
			ret = Types.SMALLINT;
			break;
		case "FLOAT":
			ret = Types.FLOAT;
			break;
		case "DATE":
			ret = Types.DATE;
			break;
		case "TIME":
			ret = Types.TIME;
			break;
		case "TIMESTAMP_LTZ":
			ret = Types.TIMESTAMP_WITH_TIMEZONE;
			break;
		case "TIMESTAMP":
			ret = Types.TIMESTAMP;
			break;
		case "INT":
			ret = Types.INTEGER;
			break;
		case "ROW":
			ret = Types.VARCHAR;
			break;

		default:
			break;
		}
		return ret;
	}

	// **
	public static FlinkTypeEnum getByDataType(String dataTypeStr) {
		if (StringUtils.isEmpty(dataTypeStr)) {
			return FlinkTypeEnum.VARCHAR;
		}
		for (FlinkTypeEnum ele : FlinkTypeEnum.values()) {
			if (dataTypeStr.equalsIgnoreCase(ele.getConvert().getDataTypeStr())) {
				return ele;
			}
		}
		return FlinkTypeEnum.VARCHAR;
	}

	public static FlinkTypeEnum getByColumnType(ColumnType columnType) {
		FlinkTypeEnum retojb = null;
		switch (columnType) {
		case VARCHAR:
			retojb = FlinkTypeEnum.VARCHAR;
			break;
		case DECIMAL:
			retojb = FlinkTypeEnum.DECIMAL;
			break;
		case TINY:
			retojb = FlinkTypeEnum.TINYINT;
			break;
		case INT24:
			retojb = FlinkTypeEnum.INTEGER;
			break;
		case SHORT:
			retojb = FlinkTypeEnum.SMALLINT;
			break;
		case LONG:
		case LONGLONG:
			retojb = FlinkTypeEnum.BIGINT;
			break;
		case FLOAT:
			retojb = FlinkTypeEnum.FLOAT;
			break;
		case DOUBLE:
			retojb = FlinkTypeEnum.DOUBLE;
			break;
		case TIMESTAMP:
			retojb = FlinkTypeEnum.TIMESTAMP_WITHOUT_TIME_ZONE;
			break;
		case DATE:
			retojb = FlinkTypeEnum.DATE;
			break;
		case TIME:
			retojb = FlinkTypeEnum.TIME_WITHOUT_TIME_ZONE;
			break;
		case DATETIME:
			retojb = FlinkTypeEnum.TIMESTAMP_WITH_LOCAL_TIME_ZONE;
			break;
		case YEAR:
			retojb = FlinkTypeEnum.INTEGER;
			break;
		case NEWDATE:
			retojb = FlinkTypeEnum.DATE;
			break;
		case BIT:
			retojb = FlinkTypeEnum.VARCHAR;
			break;
		case TIMESTAMP2:
			retojb = FlinkTypeEnum.VARCHAR;
			break;
		case DATETIME2:
			retojb = FlinkTypeEnum.VARCHAR;
			break;
		case TIME2:
			retojb = FlinkTypeEnum.VARCHAR;
			break;
		case JSON:
			retojb = FlinkTypeEnum.VARCHAR;
			break;
		case NEWDECIMAL:
			retojb = FlinkTypeEnum.VARCHAR;
			break;
		case ENUM:
			retojb = FlinkTypeEnum.VARCHAR;
			break;
		case SET:
			retojb = FlinkTypeEnum.VARCHAR;
			break;
		case TINY_BLOB:
			retojb = FlinkTypeEnum.VARCHAR;
			break;
		case MEDIUM_BLOB:
			retojb = FlinkTypeEnum.VARCHAR;
			break;
		case LONG_BLOB:
			retojb = FlinkTypeEnum.VARCHAR;
			break;
		case BLOB:
			retojb = FlinkTypeEnum.VARCHAR;
			break;
		case VAR_STRING:
			retojb = FlinkTypeEnum.VARCHAR;
			break;
		case STRING:
			retojb = FlinkTypeEnum.VARCHAR;
			break;
		default:
			retojb = FlinkTypeEnum.VARCHAR;
			break;
		}
		return retojb;
	}

	// 参考：getValue(FlinkTypeEnum flinkTypeEnum, String value, LogicalType
	// logicalType)
	public Object getRowDataValue(GenericRow record, int pos, LogicalType logicalType) {
		if (record.isNullAt(pos)) {
			return null;
		} else {
			// LogicalType logicalType = this.getConvert().getObj().getLogicalType();
			switch (this) {
			case BOOLEAN:
				return record.getBoolean(pos) ? 1L : 0L;
			case TINYINT:
				return record.getByte(pos);
			case SMALLINT:
				return record.getShort(pos);
			case INTEGER:
				return record.getInt(pos);
			case BIGINT:
				return record.getLong(pos);
			case FLOAT:
				return record.getFloat(pos);
			case DOUBLE:
				return record.getDouble(pos);
			case CHAR:
			case VARCHAR:
				return record.getString(pos).toString();
			case DATE:
				return DateFormatCase.TZyyyyMMddHHmmss.getInstanc()
						.format(java.sql.Date.valueOf(LocalDate.ofEpochDay(record.getInt(pos))));
			case TIMESTAMP_WITHOUT_TIME_ZONE:
				int timestampPrecision = 3;//((TimestampType) this.getConvert().getObj().getLogicalType()).getPrecision();
				return DateFormatCase.TZyyyyMMddHHmmss.getInstanc()
						.format(new Date(record.getTimestamp(pos, timestampPrecision).getMillisecond()));
			case TIMESTAMP_WITH_LOCAL_TIME_ZONE:
				LocalZonedTimestampType zonedTimestampType = (LocalZonedTimestampType) logicalType;
				return record.getTimestamp(pos, zonedTimestampType.getPrecision());
			case DECIMAL:
				DecimalType decimalType = (DecimalType) logicalType;
				return record.getDecimal(pos, decimalType.getPrecision(), decimalType.getScale()).toBigDecimal();
			default:
				throw new UnsupportedOperationException("还没有支持此类型的获取值:" + this.name());
			}
		}
	}

	public static String getStr(RowField rowField, GenericRow value, int pos) {
		FlinkTypeEnum flinkTypeEnum = FlinkTypeEnum.findByFlinkRowType(rowField.getType().getTypeRoot().toString());
		if (flinkTypeEnum == null) {
			throw new ProjectExceptionRuntime(ExceptAll.param_notfit,
					"列：【" + rowField.getName() + "】不支持的类型【" + rowField.getType().getTypeRoot().toString() + "】");
		}
		return flinkTypeEnum.getStringValue(value, pos, rowField.getType());
	}

	public static FlinkTypeEnum getBySqlExpression(String sqlExpression, FlinkTypeEnum oriColType) {
		FlinkTypeEnum retobj = oriColType;
		if (sqlExpression.startsWith("COUNT(")) {
			retobj = FlinkTypeEnum.BIGINT;
		} else if (sqlExpression.startsWith("PROCTIME(")) {
			retobj = FlinkTypeEnum.TIMESTAMP_WITH_LOCAL_TIME_ZONE;
		} else if (sqlExpression.startsWith("LAST(")) {// 取最后一个列的类型
			retobj = oriColType;
		}
		if (retobj == null) {
			retobj = FlinkTypeEnum.VARCHAR;
		}
		return retobj;
	}

	public String getStringValue(GenericRow record, int pos, LogicalType logicalType) {
		Object rowDataValue = getRowDataValue(record, pos, logicalType);
		String retstr = null;
		if (rowDataValue == null) {
			return retstr;
		}
		if (rowDataValue instanceof Date) {
			retstr = DateFormatCase.TZyyyyMMddHHmmss.getInstanc().format(rowDataValue);
		} else {
			retstr = String.valueOf(rowDataValue);
		}
		return retstr;
	}

	public int getDefaultPrecision() {
		return defaultPrecision;
	}

	public int getDefaultScale() {
		return defaultScale;
	}

}