package net.wicp.tams.common.others.kafka;

import java.lang.reflect.ParameterizedType;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Properties;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.apache.kafka.clients.consumer.ConsumerRecords;
import org.apache.kafka.clients.consumer.KafkaConsumer;

import lombok.extern.slf4j.Slf4j;
import net.wicp.tams.common.Conf;
import net.wicp.tams.common.Result;

/***
 * 一个consumer多个线程
 * 
 * @author 偏锋书生
 *
 *         2018年5月11日
 */
@Slf4j
public abstract class KafkaConsumerThread<T> {
	private final KafkaConsumer<String, T> consumer;
	private final String topic;
	private final IConsumer<T> doConsumer;
	private ExecutorService executor;
	private int batchNum = Integer.parseInt(Conf.get("common.others.kafka.consumer.batch.num"));
	private long timeout = Long.parseLong(Conf.get("common.others.kafka.consumer.batch.timeout"));

	public KafkaConsumerThread(String groupId, String topic, IConsumer<T> doConsumer) {
		Properties props = KafkaTools.getProps(false);
		props.put("group.id", groupId);
		props.put("key.deserializer", "org.apache.kafka.common.serialization.StringDeserializer");
		props.put("value.deserializer", KafkaTools.getValueProp(getTClass(), false));
		this.consumer = new KafkaConsumer<String, T>(props);
		this.topic = topic;
		this.doConsumer = doConsumer;
		this.consumer.subscribe(Arrays.asList(this.topic));
	}

	public KafkaConsumerThread(String topic, IConsumer<T> doConsumer) {
		this(Conf.get("common.others.kafka.consumer.group.id"), topic, doConsumer);
	}

	@SuppressWarnings("unchecked")
	public Class<T> getTClass() {
		Class<T> tClass = (Class<T>) ((ParameterizedType) this.getClass().getGenericSuperclass())
				.getActualTypeArguments()[0];
		return tClass;
	}

	public void start(int threadNumber) {
		executor = new ThreadPoolExecutor(threadNumber, threadNumber, 0L, TimeUnit.MILLISECONDS,
				new ArrayBlockingQueue<Runnable>(1000), new ThreadPoolExecutor.CallerRunsPolicy());
		final List<ConsumerRecord<String, T>> buffer = new ArrayList<>();
		long maxTime = timeout * 3;
		long startTime = System.currentTimeMillis();
		while (true) {
			final ConsumerRecords<String, T> consumerRecords = consumer.poll(timeout);
			for (ConsumerRecord<String, T> consumerRecord : consumerRecords) {
				buffer.add(consumerRecord);
			}
			long time2 = System.currentTimeMillis();
			if (buffer.size() >= batchNum || time2 - startTime > maxTime) {
				executor.submit(new Runnable() {
					@Override
					public void run() {
						Result doWithRecords = doConsumer.doWithRecords(buffer);
						KafkaTools.errorlog(consumerRecords, doWithRecords, log);
					}
				});
				consumer.commitSync();
				buffer.clear();
				log.info("从kafka取数据用时：{},数量：{}", time2 - startTime, buffer.size());
				startTime = System.currentTimeMillis();
			}
		}
	}

}
