package net.wicp.tams.common.others;

import java.io.Serializable;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import net.wicp.tams.commons.Conf;
import net.wicp.tams.commons.apiext.ReflectAssist;
import net.wicp.tams.commons.apiext.StringUtil;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;
import redis.clients.jedis.JedisPoolConfig;

import com.alibaba.fastjson.JSON;

@SuppressWarnings({ "rawtypes", "unchecked" })
public abstract class RedisClient {
	private static Logger logger = LoggerFactory.getLogger(RedisClient.class);
	private static JedisPool jedisPool;// 非切片连接池
	private static int defautlDb = 0;// 默认数据库
	private static boolean initPool = false;
	private final static Object lockObj = new Object();
	// public final static Gson gson = new
	// GsonBuilder().setDateFormat("yyyy-MM-dd HH:mm:ss").create();// gson的格式化

	/****
	 * 通过配置得到 Jedis
	 * 
	 * @return Jedis实例
	 */
	public static Jedis getConnection() {
		init();
		Jedis retJedis = jedisPool.getResource();
		retJedis.select(defautlDb);
		return retJedis;
	}

	/***
	 * 得到链接池对象
	 * 
	 * @return
	 */
	public static JedisPool getJedisPool() {
		init();
		return jedisPool;
	}

	private static void init() {
		if (jedisPool == null || initPool) {
			synchronized (lockObj) {
				initPool = false;
				if (jedisPool != null) {
					jedisPool.destroy();
				}
				String name = "common.others.redisserver";
				Map<String, String> confMap = Conf.getPre(name);
				JedisPoolConfig config = new JedisPoolConfig();
				config.setMaxTotal(Integer.parseInt(confMap.get(name + ".maxTotal")));
				config.setMaxIdle(Integer.parseInt(confMap.get(name + ".maxIdle")));
				// 20170616
				config.setMinIdle(Integer.parseInt(confMap.get(name + ".minIdle")));
				config.setMinEvictableIdleTimeMillis(
						Integer.parseInt(confMap.get(name + ".minEvictableIdleTimeMillis")));// 连接空闲最小时间
				config.setSoftMinEvictableIdleTimeMillis(
						Integer.parseInt(confMap.get(name + ".softMinEvictableIdleTimeMillis")));// 连接空闲的最小时间，达到此值后空闲链接将会被移除，且保留“minIdle”个空闲连接数。默认为-1.
				config.setNumTestsPerEvictionRun(Integer.parseInt(confMap.get(name + ".numTestsPerEvictionRun")));// 对于“空闲链接”检测线程而言，每次检测的链接资源的个数。默认为3.
				config.setTestOnReturn(Boolean.parseBoolean(confMap.get(name + ".testOnReturn")));// 向连接池“归还”链接时，是否检测“链接”对象的有效性。默认为false。建议保持默认值.
				config.setTimeBetweenEvictionRunsMillis(
						Integer.parseInt(confMap.get(name + ".timeBetweenEvictionRunsMillis")));// “空闲链接”检测线程，检测的周期，毫秒数。如果为负值，表示不运行“检测线程”。默认为-1.
				config.setBlockWhenExhausted(Boolean.parseBoolean(confMap.get(name + ".blockWhenExhausted")));// 当链接满时是否阻塞

				config.setMaxWaitMillis(Integer.parseInt(confMap.get(name + ".maxWaitMillis")));
				config.setTestOnBorrow(Boolean.parseBoolean(confMap.get(name + ".testOnBorrow")));// 向调用者输出“链接”资源时，是否检测是有有效，如果无效则从连接池中移除，并尝试获取继续获取。默认为false。建议保持默认值.
				// end 20170616
				config.setTestWhileIdle(Boolean.parseBoolean(confMap.get(name + ".testWhileIdle")));// 向调用者输出“链接”对象时，是否检测它的空闲超时；默认为false。如果“链接”空闲超时，将会被移除。建议保持默认值.
				defautlDb = confMap.get(name + ".defaultDb") == null ? 0
						: Integer.parseInt(confMap.get(name + ".defaultDb"));
				String password = confMap.get(name + ".password");
				if (StringUtils.isBlank(password)) {// 有设置密码
					jedisPool = new JedisPool(config, confMap.get(name + ".host"),
							Integer.parseInt(confMap.get(name + ".port")));
				} else {
					jedisPool = new JedisPool(config, confMap.get(name + ".host"),
							Integer.parseInt(confMap.get(name + ".port")),
							Integer.parseInt(confMap.get(name + ".maxWaitMillis")), password);
				}
				logger.info("初始化池成功");
			}
		}
	}

	/***
	 * 释放资源
	 *
	 * @param jedis
	 *            要释放的连接
	 */
	public static void returnResource(Jedis jedis) {
		if (jedis != null) {
			try {
				jedis.close();
			} catch (Exception e) {

			}
		}
		/*
		 * if (jedisPool != null) { jedisPool.returnResource(jedis); }
		 */
	}

	/*****
	 * 把对象做为Map存放到Redis
	 * 
	 * @param jedis
	 *            连接的引用，外部传入
	 * @param key
	 *            要放入的key值
	 * @param obj
	 *            要放入的对象
	 * @param expire
	 *            超时时间，单位（秒）
	 */
	public static <T extends Serializable> void putObjByMap(Jedis jedis, String key, T obj, Integer expire) {
		Map<String, String> inpumap = ReflectAssist.convertMapFromBean(obj);
		jedis.hmset(key, inpumap);
		if (expire != null) {
			jedis.expire(key, expire);
		}
	}

	/***
	 * 池废弃
	 */
	public static void destroy() {
		jedisPool.destroy();
		initPool = true;
	}

	/***
	 * 把对象做为Map存放到Redis，没有超时时间，永久性放入
	 * 
	 * @param jedis
	 *            连接的引用，外部传入
	 * @param key
	 *            要放入的key值
	 * @param obj
	 *            要放入的对象
	 */
	public static <T extends Serializable> void putObjByMap(Jedis jedis, String key, T obj) {
		putObjByMap(jedis, key, obj, null);
	}

	/***
	 * 把对象做为Json存放到Redis
	 * 
	 * @param jedis
	 *            连接的引用，外部传入
	 * @param obj
	 *            要放入的对象
	 * @param key
	 *            要放入的key值
	 * @param expire
	 *            超时时间，单位（秒）
	 */
	public final static <T extends Serializable> void putObjByJson(Jedis jedis, T obj, String key, Integer expire) {
		String json = JSON.toJSONString(obj);
		jedis.set(key, json);
		if (expire != null) {
			jedis.expire(key, expire);
		}
	}

	/****
	 * 把对象做为Json存放到Redis
	 * 
	 * @param jedis
	 *            连接的引用，外部传入
	 * @param obj
	 *            要放入的对象
	 * @param key
	 *            要放入的key值
	 */
	public final static <T extends Serializable> void putObjByJson(Jedis jedis, T obj, String key) {
		putObjByJson(jedis, obj, key, null);
	}

	/***
	 * 枚举类型的map放到缓存
	 * 
	 * @param jedis
	 *            连接的引用，外部传入
	 * @param key
	 *            要放入的key值
	 * @param inputMap
	 *            要存放的map，它的key是一个枚举值
	 */

	public final static <T extends Enum> void putEnumMap(Jedis jedis, String key, Map<T, String> inputMap) {
		if (jedis == null || MapUtils.isEmpty(inputMap) || StringUtil.isNull(key)) {
			return;
		}
		Map<String, String> input = new HashMap<>();
		for (Enum ele : inputMap.keySet()) {
			input.put(ele.name(), inputMap.get(ele));
		}
		jedis.hmset(key, input);
	}

	/****
	 * 得到枚举类的缓存对象
	 * 
	 * @param jedis
	 *            连接的引用，外部传入
	 * @param key
	 *            要放入的key值
	 * @param clazz
	 *            枚举所对应的类
	 * @return
	 */

	public final static <T extends Enum> Map<T, String> getEnumMap(Jedis jedis, String key, Class clazz) {
		Object[] objs = clazz.getEnumConstants();
		String[] fields = new String[objs.length];
		for (int i = 0; i < fields.length; i++) {
			Enum tempobj = (Enum) objs[i];
			fields[i] = tempobj.name();
		}
		List<String> rets = jedis.hmget(key, fields);
		Map<T, String> retobj = new HashMap<>();
		for (int i = 0; i < fields.length; i++) {
			Enum curobj = null;
			for (Object object : objs) {
				Enum tempobj = (Enum) object;
				if (fields[i].equals(tempobj.name())) {
					curobj = tempobj;
					break;
				}
			}
			retobj.put((T) curobj, rets.get(i));
		}
		return retobj;
	}

	/***
	 * 取指定列的值
	 * 
	 * @param jedis
	 *            连接的引用，外部传入
	 * @param key
	 *            要放入的key值
	 * @param fields
	 *            要取的列名
	 * @return
	 */
	public static Map<String, String> getMapByField(Jedis jedis, String key, String... fields) {
		Map<String, String> retobj = new HashMap<String, String>();
		if (jedis == null || StringUtil.isNull(key) || ArrayUtils.isEmpty(fields)) {
			return retobj;
		}
		List<String> values = jedis.hmget(key, fields);
		if (ArrayUtils.isNotEmpty(fields) && CollectionUtils.isNotEmpty(values)) {
			for (int i = 0; i < fields.length; i++) {
				retobj.put(fields[i], values.get(i));
			}
		}
		return retobj;
	}

	/***
	 * Redis上的值为Map,取对象的值，没有指定字段就取全部
	 * 
	 * @param clazz
	 *            要返回的对象的类
	 * @param jedis
	 *            连接的引用，外部传入
	 * @param key
	 *            要放入的key值
	 * @param fields
	 *            要取的列名
	 * @return
	 */
	public static <T extends Serializable> T getObjByMapValue(Class clazz, Jedis jedis, String key, String... fields) {
		if (ArrayUtils.isEmpty(fields)) {
			List<String> classfields = ReflectAssist.findGetField(clazz);
			fields = classfields.toArray(new String[classfields.size()]);
		}
		Map<String, String> retmap = getMapByField(jedis, key, fields);
		T rett = (T) ReflectAssist.convertMapToBean(clazz, retmap);
		return rett;
	}

	/***
	 * Redis上的值为Json,取对象的值
	 * 
	 * @param clazz
	 *            要返回的对象的类
	 * @param jedis
	 *            连接的引用，外部传入
	 * @param key
	 *            要放入的key值
	 * @return
	 */
	public static <T extends Serializable> T getObjByJsonValue(Class clazz, Jedis jedis, String key) {
		String jsonstr = jedis.get(String.valueOf(key));
		T retobj = (T) JSON.toJavaObject(JSON.parseObject(jsonstr), clazz);
		return retobj;
	}

}
