package net.wicp.tams.common.others;

import java.io.Serializable;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import net.wicp.tams.commons.Conf;
import net.wicp.tams.commons.apiext.ReflectAssist;
import net.wicp.tams.commons.apiext.StringUtil;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import redis.clients.jedis.Jedis;
import redis.clients.jedis.JedisPool;
import redis.clients.jedis.JedisPoolConfig;

import com.alibaba.fastjson.JSON;

@SuppressWarnings({ "rawtypes", "unchecked" })
public class RedisClient {
	private static Logger logger = LoggerFactory.getLogger(RedisClient.class);
	private static JedisPool jedisPool;// 非切片连接池
	private static int defautlDb = 0;// 默认数据库
	private static boolean initPool = false;
	private final static Object lockObj = new Object();
	// public final static Gson gson = new
	// GsonBuilder().setDateFormat("yyyy-MM-dd HH:mm:ss").create();// gson的格式化

	/****
	 * 通过配置得到 Jedis
	 * 
	 * @return Jedis实例
	 */
	public static Jedis getConnection() {
		if (jedisPool == null || initPool) {
			synchronized (lockObj) {
				initPool = false;
				if (jedisPool != null) {
					jedisPool.destroy();
				}
				String name = "common.others.redisserver";
				Map<String, String> confMap = Conf.getPre(name);
				JedisPoolConfig config = new JedisPoolConfig();
				config.setMaxTotal(Integer.parseInt(confMap.get(name + ".maxTotal")));
				config.setMaxIdle(Integer.parseInt(confMap.get(name + ".maxIdle")));
				config.setMaxWaitMillis(Integer.parseInt(confMap.get(name + ".maxWaitMillis")));
				config.setTestOnBorrow(Boolean.parseBoolean(confMap.get(name + ".testOnBorrow")));
				defautlDb = confMap.get(name + ".defaultDb") == null ? 0
						: Integer.parseInt(confMap.get(name + ".defaultDb"));
				String password = confMap.get(name + ".password");
				if (StringUtils.isBlank(password)) {// 有设置密码
					jedisPool = new JedisPool(config, confMap.get(name + ".host"),
							Integer.parseInt(confMap.get(name + ".port")));
				} else {
					jedisPool = new JedisPool(config, confMap.get(name + ".host"),
							Integer.parseInt(confMap.get(name + ".port")),
							Integer.parseInt(confMap.get(name + ".maxWaitMillis")), password);
				}
				logger.info("初始化池成功");
			}
		}
		Jedis retJedis = jedisPool.getResource();
		retJedis.select(defautlDb);
		return retJedis;
	}

	/***
	 * 释放资源
	 *
	 * @param jedis
	 *            要释放的连接
	 */
	public static void returnResource(Jedis jedis) {
		if (jedisPool != null) {
			jedisPool.returnResource(jedis);
		}
	}

	/*****
	 * 把对象做为Map存放到Redis
	 * 
	 * @param jedis
	 *            连接的引用，外部传入
	 * @param key
	 *            要放入的key值
	 * @param obj
	 *            要放入的对象
	 * @param expire
	 *            超时时间，单位（秒）
	 */
	public static <T extends Serializable> void putObjByMap(Jedis jedis, String key, T obj, Integer expire) {
		Map<String, String> inpumap = ReflectAssist.convertMapFromBean(obj);
		jedis.hmset(key, inpumap);
		if (expire != null) {
			jedis.expire(key, expire);
		}
	}

	/***
	 * 把对象做为Map存放到Redis，没有超时时间，永久性放入
	 * 
	 * @param jedis
	 *            连接的引用，外部传入
	 * @param key
	 *            要放入的key值
	 * @param obj
	 *            要放入的对象
	 */
	public static <T extends Serializable> void putObjByMap(Jedis jedis, String key, T obj) {
		putObjByMap(jedis, key, obj, null);
	}

	/***
	 * 把对象做为Json存放到Redis
	 * 
	 * @param jedis
	 *            连接的引用，外部传入
	 * @param obj
	 *            要放入的对象
	 * @param key
	 *            要放入的key值
	 * @param expire
	 *            超时时间，单位（秒）
	 */
	public final static <T extends Serializable> void putObjByJson(Jedis jedis, T obj, String key, Integer expire) {
		String json = JSON.toJSONString(obj);
		jedis.set(key, json);
		if (expire != null) {
			jedis.expire(key, expire);
		}
	}

	/****
	 * 把对象做为Json存放到Redis
	 * 
	 * @param jedis
	 *            连接的引用，外部传入
	 * @param obj
	 *            要放入的对象
	 * @param key
	 *            要放入的key值
	 */
	public final static <T extends Serializable> void putObjByJson(Jedis jedis, T obj, String key) {
		putObjByJson(jedis, obj, key, null);
	}

	/***
	 * 枚举类型的map放到缓存
	 * 
	 * @param jedis
	 *            连接的引用，外部传入
	 * @param key
	 *            要放入的key值
	 * @param inputMap
	 *            要存放的map，它的key是一个枚举值
	 */

	public final static <T extends Enum> void putEnumMap(Jedis jedis, String key, Map<T, String> inputMap) {
		if (jedis == null || MapUtils.isEmpty(inputMap) || StringUtil.isNull(key)) {
			return;
		}
		Map<String, String> input = new HashMap<>();
		for (Enum ele : inputMap.keySet()) {
			input.put(ele.name(), inputMap.get(ele));
		}
		jedis.hmset(key, input);
	}

	/****
	 * 得到枚举类的缓存对象
	 * 
	 * @param jedis
	 *            连接的引用，外部传入
	 * @param key
	 *            要放入的key值
	 * @param clazz
	 *            枚举所对应的类
	 * @return
	 */

	public final static <T extends Enum> Map<T, String> getEnumMap(Jedis jedis, String key, Class clazz) {
		Object[] objs = clazz.getEnumConstants();
		String[] fields = new String[objs.length];
		for (int i = 0; i < fields.length; i++) {
			Enum tempobj = (Enum) objs[i];
			fields[i] = tempobj.name();
		}
		List<String> rets = jedis.hmget(key, fields);
		Map<T, String> retobj = new HashMap<>();
		for (int i = 0; i < fields.length; i++) {
			Enum curobj = null;
			for (Object object : objs) {
				Enum tempobj = (Enum) object;
				if (fields[i].equals(tempobj.name())) {
					curobj = tempobj;
					break;
				}
			}
			retobj.put((T) curobj, rets.get(i));
		}
		return retobj;
	}

	/***
	 * 取指定列的值
	 * 
	 * @param jedis
	 *            连接的引用，外部传入
	 * @param key
	 *            要放入的key值
	 * @param fields
	 *            要取的列名
	 * @return
	 */
	public static Map<String, String> getMapByField(Jedis jedis, String key, String... fields) {
		Map<String, String> retobj = new HashMap<String, String>();
		if (jedis == null || StringUtil.isNull(key) || ArrayUtils.isEmpty(fields)) {
			return retobj;
		}
		List<String> values = jedis.hmget(key, fields);
		if (ArrayUtils.isNotEmpty(fields) && CollectionUtils.isNotEmpty(values)) {
			for (int i = 0; i < fields.length; i++) {
				retobj.put(fields[i], values.get(i));
			}
		}
		return retobj;
	}

	/***
	 * Redis上的值为Map,取对象的值，没有指定字段就取全部
	 * 
	 * @param clazz
	 *            要返回的对象的类
	 * @param jedis
	 *            连接的引用，外部传入
	 * @param key
	 *            要放入的key值
	 * @param fields
	 *            要取的列名
	 * @return
	 */
	public static <T extends Serializable> T getObjByMapValue(Class clazz, Jedis jedis, String key, String... fields) {
		if (ArrayUtils.isEmpty(fields)) {
			List<String> classfields = ReflectAssist.findGetField(clazz);
			fields = classfields.toArray(new String[classfields.size()]);
		}
		Map<String, String> retmap = getMapByField(jedis, key, fields);
		T rett = (T) ReflectAssist.convertMapToBean(clazz, retmap);
		return rett;
	}

	/***
	 * Redis上的值为Json,取对象的值
	 * 
	 * @param clazz
	 *            要返回的对象的类
	 * @param jedis
	 *            连接的引用，外部传入
	 * @param key
	 *            要放入的key值
	 * @return
	 */
	public static <T extends Serializable> T getObjByJsonValue(Class clazz, Jedis jedis, String key) {
		String jsonstr = jedis.get(String.valueOf(key));
		T retobj = (T) JSON.toJavaObject(JSON.parseObject(jsonstr), clazz);
		return retobj;
	}

}
