package net.wicp.tams.common.others;

import java.util.List;

import org.apache.curator.framework.CuratorFramework;
import org.apache.curator.framework.CuratorFrameworkFactory;
import org.apache.curator.retry.ExponentialBackoffRetry;
import org.apache.zookeeper.data.Stat;

import com.alibaba.fastjson.JSONObject;

import lombok.extern.slf4j.Slf4j;
import net.wicp.tams.commons.Conf;
import net.wicp.tams.commons.Result;
import net.wicp.tams.commons.apiext.StringUtil;

@Slf4j
public class ZkClient {
	private static ZkClient inst = new ZkClient();
	private final CuratorFramework curator;

	private ZkClient() {
		curator = CuratorFrameworkFactory.newClient(Conf.get("common.others.zookeeper.constr"),
				Integer.parseInt(Conf.get("common.others.zookeeper.sleepTimeMs")),
				Integer.parseInt(Conf.get("common.others.zookeeper.sessionTimeoutMs")),
				new ExponentialBackoffRetry(Integer.parseInt(Conf.get("common.others.zookeeper.sleepTimeMs")),
						Integer.parseInt(Conf.get("common.others.zookeeper.maxRetries"))));
		curator.start();
	}

	public static ZkClient getInst() {
		return ZkClient.inst;
	}

	public Stat exists(String path) {
		try {
			return curator.checkExists().forPath(path);
		} catch (Exception e) {
			log.error("查找task配置错误", e);
			return null;
		}
	}

	public List<String> getChildren(String path) {
		try {
			List<String> colsTables = curator.getChildren().forPath(path);
			return colsTables;
		} catch (Exception e) {
			log.error("查看子节点错误", e);
			return null;
		}
	}

	public CuratorFramework getCurator() {
		return curator;
	}

	public String createNode(String path, String value) {
		path = StringUtil.trimSpace(path);
		if (StringUtil.isNull(path)) {
			return null;
		}
		value = StringUtil.trimSpace(value);
		value = StringUtil.isNull(value) ? null : value;
		try {
			String pathret = null;
			if (value == null) {
				pathret = curator.create().forPath(path);
			} else {
				pathret = curator.create().forPath(path, value.getBytes("UTF-8"));
			}
			return pathret;
		} catch (Exception e) {
			log.error("创建结点错误", e);
			return null;
		}
	}

	public Stat updateNode(String path, String data) {
		try {
			return curator.setData().forPath(path, data.getBytes("UTF-8"));
		} catch (Exception e) {
			log.error("更新数据错误", e);
			return null;
		}
	}

	public Result createOrUpdateNode(String path, String data) {
		try {
			Stat stat = curator.checkExists().forPath(path);
			if (stat == null) {
				createNode(path, data);
			} else {
				updateNode(path, data);
			}
			return Result.getSuc();
		} catch (Exception e) {
			log.error("更新数据错误", e);
			return Result.getError(e.getMessage());
		}
	}

	public <T> Result createOrUpdateNodeForJson(String path, T obj) {
		String jsonstr = JSONObject.toJSONString(obj);
		return createOrUpdateNode(path, jsonstr);
	}

	/***
	 * 创建多路径节点
	 * 
	 * @param path
	 */
	public void createMultilevelNode(String path) {
		if (StringUtil.isNull(path) || !path.startsWith("/")) {
			throw new IllegalArgumentException("path错误");
		}
		path = path.replace("\\", "/");
		String[] paths = path.split("/");

		try {
			for (String ele : paths) {
				if (StringUtil.isNull(ele)) {
					continue;
				}
				String tempstr = "/" + ele;
				int idx = path.indexOf(tempstr);
				String tempPath = path.substring(0, idx + tempstr.length());
				Stat stat = curator.checkExists().forPath(tempPath);
				if (stat == null) {
					curator.create().forPath(tempPath);
				}
			}
		} catch (Exception e) {
			throw new RuntimeException(e);
		}
	}

	public String getZkDataStr(String path) {
		try {
			byte[] databin = curator.getData().forPath(path);
			return new String(databin, "UTF-8");
		} catch (Exception e) {
			log.error("查找task配置错误", e);
			return null;
		}
	}

	/***
	 * 从zk上得到java对象
	 * 
	 * @param path
	 * @param clazz
	 * @return
	 */
	public <T> T getDateObj(String path, Class<T> clazz) {
		String data = getZkDataStr(path);
		return JSONObject.parseObject(StringUtil.isNull(data) ? "{}" : data, clazz);
	}

	public Result deleteNode(String path) {
		try {
			curator.delete().forPath(path);
			return Result.getSuc();
		} catch (Exception e) {
			log.error("删除数据错误", e);
			return Result.getError("删除数据错误:" + e.getMessage());
		}
	}

	/***
	 * 得到json数据
	 * 
	 * @param path
	 * @return
	 */
	public JSONObject getZkData(String path) {
		return JSONObject.parseObject(getZkDataStr(path));
	}
}
