/*
 * **********************************************************************
 * Copyright (c) 2022 .
 * All rights reserved.
 * 项目名称：common
 * 项目描述：公共的工具集
 * 版权说明：本软件属andy.zhou(rjzjh@163.com)所有。
 * ***********************************************************************
 */
package net.wicp.tams.common.metrics.core;

import com.codahale.metrics.MetricFilter;
import com.codahale.metrics.MetricRegistry;
import com.codahale.metrics.Snapshot;
import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.core.Version;
import com.fasterxml.jackson.databind.JsonSerializer;
import com.fasterxml.jackson.databind.Module;
import com.fasterxml.jackson.databind.SerializerProvider;
import com.fasterxml.jackson.databind.module.SimpleSerializers;
import com.fasterxml.jackson.databind.ser.std.StdSerializer;

import java.io.IOException;
import java.util.Arrays;
import java.util.Locale;
import java.util.concurrent.TimeUnit;

import net.wicp.tams.common.metrics.entity.statisticbean.*;


public class TsMetricsSerializer extends Module{
    static final Version VERSION = new Version(3, 0, 0, "", "com.codahale.metrics", "metrics-json");

    private static class StatisticGaugeBeanSerializer extends StdSerializer<StatisticGaugeBean> {
        private StatisticGaugeBeanSerializer() {
            super(StatisticGaugeBean.class);
        }

        @Override
        public void serialize(StatisticGaugeBean statisticGaugeBean,
                              JsonGenerator json,
                              SerializerProvider provider) throws IOException {
            json.writeStartObject();
            json.writeStringField("name", statisticGaugeBean.getName());
            json.writeStringField("type", "GAUGE");
            final Object value;
            try {
                value = statisticGaugeBean.getGauge().getValue();
                json.writeObjectField("value", value);
            } catch (RuntimeException e) {
                json.writeObjectField("error", e.toString());
            }
            json.writeEndObject();
        }
    }

    private static class StatisticCounterBeanSerializer extends StdSerializer<StatisticCounterBean> {
        private StatisticCounterBeanSerializer() {
            super(StatisticCounterBean.class);
        }

        @Override
        public void serialize(StatisticCounterBean statisticCounterBean,
                              JsonGenerator json,
                              SerializerProvider provider) throws IOException {
            json.writeStartObject();
            json.writeStringField("name", statisticCounterBean.getName());
            json.writeStringField("type", "COUNTER");
            json.writeNumberField("count", statisticCounterBean.getCounter().getCount());
            json.writeEndObject();
        }
    }

    private static class StatisticHistogramBeanSerializer extends StdSerializer<StatisticHistogramBean> {
        private final boolean showSamples;

        private StatisticHistogramBeanSerializer(boolean showSamples) {
            super(StatisticHistogramBean.class);
            this.showSamples = showSamples;
        }

        @Override
        public void serialize(StatisticHistogramBean statisticHistogramBean,
                              JsonGenerator json,
                              SerializerProvider provider) throws IOException {
            json.writeStartObject();
            final Snapshot snapshot = statisticHistogramBean.getHistogram().getSnapshot();
            json.writeStringField("name", statisticHistogramBean.getName());
            json.writeStringField("type", "HISTOGRAM");
            json.writeNumberField("count", statisticHistogramBean.getHistogram().getCount());
            json.writeNumberField("max", snapshot.getMax());
            json.writeNumberField("mean", snapshot.getMean());
            json.writeNumberField("min", snapshot.getMin());
            json.writeNumberField("p50", snapshot.getMedian());
            json.writeNumberField("p75", snapshot.get75thPercentile());
            json.writeNumberField("p95", snapshot.get95thPercentile());
            json.writeNumberField("p98", snapshot.get98thPercentile());
            json.writeNumberField("p99", snapshot.get99thPercentile());
            json.writeNumberField("p999", snapshot.get999thPercentile());

            if (showSamples) {
                json.writeObjectField("values", snapshot.getValues());
            }

            json.writeNumberField("stddev", snapshot.getStdDev());
            json.writeEndObject();
        }
    }

    private static class StatisticMeterBeanSerializer extends StdSerializer<StatisticMeterBean> {
        private final String rateUnit;
        private final double rateFactor;

        public StatisticMeterBeanSerializer(TimeUnit rateUnit) {
            super(StatisticMeterBean.class);
            this.rateFactor = rateUnit.toSeconds(1);
            this.rateUnit = calculateRateUnit(rateUnit, "events");
        }

        @Override
        public void serialize(StatisticMeterBean statisticMeterBean,
                              JsonGenerator json,
                              SerializerProvider provider) throws IOException {
            json.writeStartObject();
            json.writeStringField("name", statisticMeterBean.getName());
            json.writeStringField("type", "METER");
            json.writeNumberField("count", statisticMeterBean.getMeter().getCount());
            json.writeNumberField("m15_rate", statisticMeterBean.getMeter().getFifteenMinuteRate() * rateFactor);
            json.writeNumberField("m1_rate", statisticMeterBean.getMeter().getOneMinuteRate() * rateFactor);
            json.writeNumberField("m5_rate", statisticMeterBean.getMeter().getFiveMinuteRate() * rateFactor);
            json.writeNumberField("mean_rate", statisticMeterBean.getMeter().getMeanRate() * rateFactor);
            json.writeStringField("units", rateUnit);
            json.writeEndObject();
        }
    }

    private static class StatisticTimerBeanSerializer extends StdSerializer<StatisticTimerBean> {
        private final String rateUnit;
        private final double rateFactor;
        private final String durationUnit;
        private final double durationFactor;
        private final boolean showSamples;

        private StatisticTimerBeanSerializer(TimeUnit rateUnit,
                                             TimeUnit durationUnit,
                                             boolean showSamples) {
            super(StatisticTimerBean.class);
            this.rateUnit = calculateRateUnit(rateUnit, "calls");
            this.rateFactor = rateUnit.toSeconds(1);
            this.durationUnit = durationUnit.toString().toLowerCase(Locale.US);
            this.durationFactor = 1.0 / durationUnit.toNanos(1);
            this.showSamples = showSamples;
        }

        @Override
        public void serialize(StatisticTimerBean statisticTimerBean,
                              JsonGenerator json,
                              SerializerProvider provider) throws IOException {
            json.writeStartObject();
            final Snapshot snapshot = statisticTimerBean.getTimer().getSnapshot();
            json.writeStringField("name", statisticTimerBean.getName());
            json.writeStringField("type", "TIMER");
            json.writeNumberField("count", statisticTimerBean.getTimer().getCount());
            json.writeNumberField("max", snapshot.getMax() * durationFactor);
            json.writeNumberField("mean", snapshot.getMean() * durationFactor);
            json.writeNumberField("min", snapshot.getMin() * durationFactor);

            json.writeNumberField("p50", snapshot.getMedian() * durationFactor);
            json.writeNumberField("p75", snapshot.get75thPercentile() * durationFactor);
            json.writeNumberField("p95", snapshot.get95thPercentile() * durationFactor);
            json.writeNumberField("p98", snapshot.get98thPercentile() * durationFactor);
            json.writeNumberField("p99", snapshot.get99thPercentile() * durationFactor);
            json.writeNumberField("p999", snapshot.get999thPercentile() * durationFactor);

            if (showSamples) {
                final long[] values = snapshot.getValues();
                final double[] scaledValues = new double[values.length];
                for (int i = 0; i < values.length; i++) {
                    scaledValues[i] = values[i] * durationFactor;
                }
                json.writeObjectField("values", scaledValues);
            }

            json.writeNumberField("stddev", snapshot.getStdDev() * durationFactor);
            json.writeNumberField("m15_rate", statisticTimerBean.getTimer().getFifteenMinuteRate() * rateFactor);
            json.writeNumberField("m1_rate", statisticTimerBean.getTimer().getOneMinuteRate() * rateFactor);
            json.writeNumberField("m5_rate", statisticTimerBean.getTimer().getFiveMinuteRate() * rateFactor);
            json.writeNumberField("mean_rate", statisticTimerBean.getTimer().getMeanRate() * rateFactor);
            json.writeStringField("duration_units", durationUnit);
            json.writeStringField("rate_units", rateUnit);
            json.writeEndObject();
        }
    }

    private static class MetricRegistrySerializer extends StdSerializer<MetricRegistry> {

        private final MetricFilter filter;

        private MetricRegistrySerializer(MetricFilter filter) {
            super(MetricRegistry.class);
            this.filter = filter;
        }

        @Override
        public void serialize(MetricRegistry registry,
                              JsonGenerator json,
                              SerializerProvider provider) throws IOException {
            json.writeStartObject();
            json.writeStringField("version", VERSION.toString());
            json.writeObjectField("gauges", registry.getGauges(filter));
            json.writeObjectField("counters", registry.getCounters(filter));
            json.writeObjectField("histograms", registry.getHistograms(filter));
            json.writeObjectField("meters", registry.getMeters(filter));
            json.writeObjectField("timers", registry.getTimers(filter));
            json.writeEndObject();
        }
    }

    private final TimeUnit rateUnit;
    private final TimeUnit durationUnit;
    private final boolean showSamples;
    private final MetricFilter filter;

    public TsMetricsSerializer(TimeUnit rateUnit, TimeUnit durationUnit, boolean showSamples) {
        this(rateUnit, durationUnit, showSamples, MetricFilter.ALL);
    }

    public TsMetricsSerializer(TimeUnit rateUnit, TimeUnit durationUnit, boolean showSamples, MetricFilter filter) {
        this.rateUnit = rateUnit;
        this.durationUnit = durationUnit;
        this.showSamples = showSamples;
        this.filter = filter;
    }

    @Override
    public String getModuleName() {
        return "metrics";
    }

    @Override
    public Version version() {
        return VERSION;
    }

    @Override
    public void setupModule(Module.SetupContext context) {
        context.addSerializers(new SimpleSerializers(Arrays.<JsonSerializer<?>>asList(
                new TsMetricsSerializer.StatisticGaugeBeanSerializer(),
                new TsMetricsSerializer.StatisticCounterBeanSerializer(),
                new TsMetricsSerializer.StatisticHistogramBeanSerializer(showSamples),
                new TsMetricsSerializer.StatisticMeterBeanSerializer(rateUnit),
                new TsMetricsSerializer.StatisticTimerBeanSerializer(rateUnit, durationUnit, showSamples),
                new TsMetricsSerializer.MetricRegistrySerializer(filter)
        )));
    }

    private static String calculateRateUnit(TimeUnit unit, String name) {
        final String s = unit.toString().toLowerCase(Locale.US);
        return name + '/' + s.substring(0, s.length() - 1);
    }
}