/*
 * **********************************************************************
 * Copyright (c) 2022 .
 * All rights reserved.
 * 项目名称：common
 * 项目描述：公共的工具集
 * 版权说明：本软件属andy.zhou(rjzjh@163.com)所有。
 * ***********************************************************************
 */
package net.wicp.tams.common.kudu;

import java.math.BigDecimal;
import java.sql.Connection;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;

import com.google.common.collect.Lists;
import net.wicp.tams.common.binlog.alone.ListenerConf;
import net.wicp.tams.common.constant.DbType;
import net.wicp.tams.common.exception.ExceptAll;
import net.wicp.tams.common.exception.ProjectExceptionRuntime;
import net.wicp.tams.common.jdbc.MySqlAssitExt;
import net.wicp.tams.common.kudu.plugin.JdbcConnection;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.kudu.ColumnSchema;
import org.apache.kudu.ColumnTypeAttributes;
import org.apache.kudu.Schema;
import org.apache.kudu.Type;
import org.apache.kudu.client.*;
import org.apache.kudu.client.KuduClient.KuduClientBuilder;

import lombok.extern.slf4j.Slf4j;
import net.wicp.tams.common.Conf;
import net.wicp.tams.common.Result;
import net.wicp.tams.common.apiext.DateUtil;
import net.wicp.tams.common.apiext.StringUtil;
import net.wicp.tams.common.apiext.jdbc.MySqlColBean;
import net.wicp.tams.common.constant.FieldFormart;

@Slf4j
public class KuduAssit {

	private final KuduClient client;

	private final Map<String, String> configmap;

	public KuduAssit(String ds) {
		Map<String, String> preAddDefault = Conf.getPreAddDefault("common.kudu." + StringUtil.hasNull(ds, "default"),
				true);
		KuduClientBuilder clientBuilder = new KuduClient.KuduClientBuilder(preAddDefault.get("master.addresses"));
		KuduClient client = clientBuilder.build();
		this.client = client;
		this.configmap = preAddDefault;
	}

	public void close() {
		try {
			client.close();
		} catch (KuduException e) {
		}
	}

	/**
	 * 通过mysql字段创建表
	 * 
	 * @param tb        要创建的表名
	 * @param mysqlCols 要创建的mysql字段
	 * @param addCols   附加字段
	 * @throws Exception
	 */
	public void createTable(String tb, List<MySqlColBean> mysqlCols, MySqlColBean... addCols) throws Exception {
		boolean tableExists = this.client.tableExists(tb);
		if (tableExists) {
			log.warn("表[{}]已存在", tb);
			return;
		}
		if (ArrayUtils.isNotEmpty(addCols)) {
			mysqlCols.addAll(Arrays.asList(addCols));
		}
		List<ColumnSchema> columns = new ArrayList<ColumnSchema>();
		List<String> partions = new ArrayList<String>();
		for (MySqlColBean mysqlCol : mysqlCols) {
			Type kuduType = convertMysqlToKuduType(mysqlCol.getDataType());
			ColumnTypeAttributes columnTypeAttributes = null;
			switch (kuduType) {
			case DECIMAL:
				columnTypeAttributes = (new ColumnTypeAttributes.ColumnTypeAttributesBuilder())
						.scale(mysqlCol.getNumericScale())
						.precision(Integer.parseInt(String.valueOf(mysqlCol.getMaxlength()))).build();
				break;
			default:
				break;
			}
			columns.add(new ColumnSchema.ColumnSchemaBuilder(mysqlCol.getColumnNameStandard(FieldFormart.dbstandard),
					kuduType).key(mysqlCol.isPri()).typeAttributes(columnTypeAttributes)
							.nullable(mysqlCol.isPri() ? false : true).build());
			if (mysqlCol.isPri()) {
				partions.add(mysqlCol.getColumnNameStandard(FieldFormart.dbstandard));
			}
		}
		Schema schema = new Schema(columns);
		CreateTableOptions options = new CreateTableOptions()
				.addHashPartitions(partions, Integer.parseInt(configmap.get("partions")))
				.setNumReplicas(Integer.parseInt(configmap.get("replicas")));
		client.createTable(tb, schema, options);
	}

	/**
	 * 新增字段
	 * @param tableName
	 * @param addColNames
	 * @throws Exception
	 */
	public void addTableColumn(String tableName , ListenerConf.ColHis colHis, String[] addColNames ) throws Exception {
		Boolean isExiste = client.tableExists("kudu::" + tableName);
		if(!isExiste){
			log.info("upsert kudu表不存在,table={}",tableName);
			return;
		}
		AlterTableOptions alterTableOptions = new AlterTableOptions();
		for(String columnName : addColNames){
			Type type = convertMysqlToKuduType(colHis.getColTypes(colHis.getColsList().indexOf(columnName)));
			alterTableOptions.addNullableColumn(columnName,type);
		}
		AlterTableResponse alterTableResponse = client.alterTable("kudu::"+tableName,alterTableOptions);
		//重建impala外表
		String impalaUrl = DbType.impala.geturl(configmap.get("impala.addresses"), Integer.parseInt(configmap.get("impala.port")));
		Connection impalaConn = JdbcConnection.getConnection("com.cloudera.impala.jdbc41.Driver", impalaUrl,
				configmap.get("impala.user"), configmap.get("impala.password"));

		String impalaTbName = tableName;

		//删impala外表
		String dropSql = "DROP TABLE IF EXISTS " + impalaTbName;
		MySqlAssitExt.execSql(impalaConn, dropSql);
		//建impala外表
		String sql = "CREATE EXTERNAL TABLE IF NOT EXISTS " + impalaTbName
				+ " STORED AS KUDU TBLPROPERTIES('kudu.table_name' = 'kudu::" + tableName
				+ "','kudu.master_addresses' = 'bigdata-01:7051,bigdata-02:7051,bigdata-03:7051')";
		MySqlAssitExt.execSql(impalaConn, sql);
	}

	//	脚本
//	public static void main(String[] args) throws KuduException {
//		KuduClientBuilder clientBuilder = new KuduClient.KuduClientBuilder("10.81.83.47:7051,10.81.80.65:7051,10.81.80.173:7051");
//		KuduClient client1 = clientBuilder.build();
//
//		String tableName = "kudu::ods_tps.tps_tradeapp_purchase_receipt_sku_row";
//		//client_type,source_type
//		String columnName = "header_id";
//
//		String columnType = "varchar";
//
//		Boolean isExiste = client1.tableExists(tableName);
//		if(!isExiste){
//			log.info("upsert kudu表不存在,table={}",tableName);
//			return;
//		}
////		AlterTableOptions alterTableOptions = new AlterTableOptions();
////		Type type = convertMysqlToKuduType(columnType);
////		alterTableOptions.addNullableColumn(columnName,type);
////		AlterTableResponse alterTableResponse = client1.alterTable("kudu::"+tableName,alterTableOptions);
//		DeleteTableResponse deleteTableResponse = client1.deleteTable(tableName);
//		client1.close();
//	}

	public void dropTable(String tb) throws Exception {
		boolean tableExists = this.client.tableExists(tb);
		if (!tableExists) {
			log.warn("表[{}]不存在", tb);
			return;
		}
		this.client.deleteTable(tb);
	}

	/***
	 * 更新数据
	 * 
	 * @param tb
	 * @param datas
	 * @throws Exception
	 */
	public void upsertRow(String tb, List<Map<String, String>> datas) throws Exception {
		Boolean isExiste = client.tableExists("kudu::" + tb);
		if(!isExiste){
			log.info("upsert kudu表不存在,table={}",tb);
			return;
		}
		KuduTable table = client.openTable("kudu::" + tb);
		if(table != null){
			KuduSession session = this.client.newSession();
			for (Map<String, String> data : datas) {
				Upsert upsert = table.newUpsert();
				PartialRow row = upsert.getRow();
				for (String key : data.keySet()) {
					ColumnSchema column = table.getSchema().getColumn(key);
					Type colType = column.getType();
					String value = data.get(key);
					putValueByType(row, key, value, colType);
				}
				OperationResponse operationResponse = session.apply(upsert);
				if(operationResponse.hasRowError()){
					log.error("upsert kudu 异常,rowErrors={}",operationResponse.getRowError());
					throw new ProjectExceptionRuntime(ExceptAll.Project_default,"upsert kudu 异常！");
				}
			}
			session.close();
		}
	}

	/**
	 * 删除数据
	 * 
	 * @throws Exception
	 */
	public Result dataDelete(String tb, List<Map<String, String>> keysValues) throws Exception {
		KuduTable table = this.client.openTable(tb);
		List<ColumnSchema> keyColumns = table.getSchema().getPrimaryKeyColumns();
		KuduSession session = this.client.newSession();
		for (Map<String, String> keysValue : keysValues) {
			Delete delete = table.newDelete();
			PartialRow row = delete.getRow();
			for (ColumnSchema keyColumn : keyColumns) {
				String value = keysValue.get(keyColumn.getName());
				putValueByType(row, keyColumn.getName(), value, keyColumn.getType());
			}
			session.apply(delete);
		}
		session.flush();
		session.close();
		return Result.getSuc();
	}

	/**
	 * 条件查询 select * from person where id=1
	 * 
	 */
//	public static void query() throws Exception {
//		// 1、创建kudu client
//		KuduClient client = new KuduClientBuilder("").build();
//		// 2、打开表
//		KuduTable table = client.openTable("person");
//		// 3、创建scanner扫描器
//		KuduScanner.KuduScannerBuilder kuduScannerBuilder = client.newScannerBuilder(table);
//		// 4、创建查询条件
//		KuduPredicate filter = KuduPredicate.newComparisonPredicate(table.getSchema().getColumn("id"),
//				KuduPredicate.ComparisonOp.EQUAL, 1);
//		// 5、将查询条件加入到scanner中
//		KuduScanner scanner = kuduScannerBuilder.addPredicate(filter).build();
//		// 6、获取查询结果
//		while (scanner.hasMoreRows()) {
//			RowResultIterator rows = scanner.nextRows();
//			while (rows.hasNext()) {
//				RowResult row = rows.next();
//				Integer id = row.getInt("id");
//				String name = row.getString("name");
//				int age = row.getInt("age");
//				System.out.println(id + "---" + name + "---" + age);
//			}
//		}
//		// 7、关闭client
//		client.close();
//	}

	public static void putValueByType(PartialRow row, String key, String value, Type colType) {
		switch (colType) {
		case STRING:
			row.addString(key, value);
			break;
		case INT8:
			row.addByte(key, Byte.parseByte(value));
			break;
		case INT16:
			row.addShort(key, Short.parseShort(value));
			break;
		case INT32:
			row.addInt(key, Integer.parseInt(value));
			break;
		case INT64:
			row.addLong(key, Long.parseLong(value));
			break;
		case FLOAT:
			row.addFloat(key, Float.parseFloat(value));
			break;
		case DOUBLE:
			row.addDouble(key, Double.parseDouble(value));
			break;
		case BOOL:
			row.addBoolean(key, Boolean.parseBoolean(value));
			break;
		case DECIMAL:
			row.addDecimal(key, new BigDecimal(value));
			break;
		case UNIXTIME_MICROS:
//			Timestamp time = new Timestamp(DateUtil.objToDate(value).getTime());
			Timestamp time = new Timestamp(DateUtil.changeTimeZone(DateUtil.objToDate(value),DateUtil.getTZZone(),DateUtil.getBeijingTimeZone()).getTime());
			row.addTimestamp(key, time);
			break;
		case BINARY:
			row.addBinary(key, value.getBytes());
			break;
		default:
			break;
		}
	}


	/***
	 * 
	 * @param mySqlDataType mysql数据类型，参考：MySqlColBean.DataType
	 * @return
	 */
	public static Type convertMysqlToKuduType(String mySqlDataType) {
		// 不限长度
		Type retColType = Type.STRING;
		switch (mySqlDataType) {
		case "varchar":
			retColType = Type.STRING;
			break;
		case "binary":
			retColType = Type.STRING;
			break;
		case "bool":
			retColType = Type.BOOL;
			break;
		case "boolean":
			retColType = Type.BOOL;
			break;
		case "longblob":
			retColType = Type.STRING;
			break;
		case "int":
			retColType = Type.INT32;
			break;
		case "integer":
			retColType = Type.INT32;
			break;
		case "mediumint":
			retColType = Type.INT32;
			break;
		case "smallint":
			retColType = Type.INT16;
			break;
		case "char":
			retColType = Type.STRING;
			break;
		case "json":
			retColType = Type.STRING;
			break;
		case "datetime":
			retColType = Type.UNIXTIME_MICROS;
			break;
		case "bit":
			retColType = Type.INT8;
			break;
		case "decimal":
			retColType = Type.DECIMAL;
			break;
		case "longtext":
			retColType = Type.STRING;
			break;
		case "tinyint":
			retColType = Type.INT8;
			break;
		case "mediumtext":
			retColType = Type.STRING;
			break;
		case "bigint":
			retColType = Type.INT64;
			break;
		case "time":
			retColType = Type.UNIXTIME_MICROS;
			break;
		case "float":
			retColType = Type.FLOAT;
			break;
		case "timestamp":
			retColType = Type.UNIXTIME_MICROS;
			break;
		case "text":
			retColType = Type.STRING;
			break;
		case "double":
			retColType = Type.DOUBLE;
			break;
		case "date":
			retColType = Type.UNIXTIME_MICROS;
			break;
		case "blob":
			retColType = Type.STRING;
			break;
		default:
			break;
		}
		return retColType;
	}

}
