package net.wicp.tams.common.kubernetes.helm;

import java.util.Arrays;
import java.util.Comparator;
import java.util.Map;

import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.ArrayUtils;

import net.wicp.tams.common.apiext.StringUtil;
import net.wicp.tams.common.callback.IConvertValue;
import net.wicp.tams.common.constant.PathType;
import net.wicp.tams.common.constant.dic.intf.IEnumCombobox;
import net.wicp.tams.common.exception.ExceptAll;
import net.wicp.tams.common.exception.ProjectExceptionRuntime;

/***
 * Helm参数配置项枚举
 * 
 * @author andy.zhou
 * 
 */
public enum HelmFlags implements IEnumCombobox {
	name("名称", "name", "name", true, new IConvertValue<String>() {

		@Override
		public String getStr(String keyObj) {
			// rfc1123规则
			return StringUtil.formatRfc1123(keyObj);
		}

	}, new HelmCmd[] { HelmCmd.install, HelmCmd.delete, HelmCmd.reposearch, HelmCmd.repoadd }, 0), // 参数 -name

	namespace("名称空间", "namespace", "n", false, null, new HelmCmd[] { HelmCmd.install, HelmCmd.list, HelmCmd.delete },
			Integer.MAX_VALUE), // 必须放到最后面

	namespace_all("所有名称空间", "all-namespaces", "A", false, null,
			new HelmCmd[] { HelmCmd.install, HelmCmd.list, HelmCmd.delete }, Integer.MAX_VALUE, true), // 必须放到最后面，它不需要值。

	kubeconfig("k8s配置文件", "kubeconfig", null, true, new IConvertValue<String>() {

		@Override
		public String getStr(String keyObj) {
			return PathType.getPath(keyObj);
		}

	}, new HelmCmd[] { HelmCmd.install, HelmCmd.list, HelmCmd.delete }, 0),

	kubecontext("k8s context", "kube-context", false, new HelmCmd[] { HelmCmd.install, HelmCmd.list, HelmCmd.delete }),

	url("chartrepo地址", null, null, true, null, new HelmCmd[] { HelmCmd.repoadd }, 1),

	filter("过滤", "filter", "f", false, null, new HelmCmd[] { HelmCmd.list }, 0),

	// 不是真正的flags
	reponame("本地仓库名", null, null, true, null, new HelmCmd[] { HelmCmd.chartpush }, Integer.MAX_VALUE - 1),

	// char包本地路径,有了它就不需要chartFilePath+version
	// <Chart 名称>-<Chart 版本>.tgz 或 <chart目录 名称>
	chartLocalpath("chart包路径", null, null, false, new IConvertValue<String>() {

		@Override
		public String getStr(String keyObj) {
			return PathType.getPath(keyObj);
		}

	}, new HelmCmd[] { HelmCmd.install }, Integer.MAX_VALUE - 2),

	// 安装的版本，如果chartFilePath是本地上传的chart目录则不需要传入，如果是本地仓库则必须传入
	version("chart版本", "version", null, false, null, new HelmCmd[] { HelmCmd.install, HelmCmd.chartfetch },
			Integer.MAX_VALUE - 1),

	// chart文件名（用于chart仓库）+version就可以惟一定位于chart包了。
	chartFilePath("本地文件名或目录名", null, null, false, null,
			new HelmCmd[] { HelmCmd.chartpush, HelmCmd.install, HelmCmd.chartfetch }, Integer.MAX_VALUE - 2),

	// 修改少量参数,中间使用逗号分隔，并设置到value里
	set("覆盖参数（少量）", "set", null, false, null, new HelmCmd[] { HelmCmd.install }, Integer.MAX_VALUE - 3),

	values("用户自定义的参数（大量）", "values", "f", false, new IConvertValue<String>() {

		@Override
		public String getStr(String keyObj) {
			return PathType.getPath(keyObj);
		}

	}, new HelmCmd[] { HelmCmd.install }, Integer.MAX_VALUE - 4),

	destination("package打包的目的目录，不需要k8s配置", "destination", "d", false, null, new HelmCmd[] { HelmCmd.chartPackage }, 2),

	dependency_update("package打包前更新依赖到charts/目录", "dependency-update", "u", false, null,
			new HelmCmd[] { HelmCmd.chartPackage }, Integer.MAX_VALUE - 2, true);

	private final String desc;
	private final String paramName;
	private final String paramShortName;
	private final boolean notnull;
	private final boolean notvalue;// 不需要传值的参数，如 -A

	private final IConvertValue<String> formart;
	private final HelmCmd[] cmds;
	private final int orderby;

	/**
	 * 产生参数
	 * 
	 * @param values
	 * @return
	 */
	public static String proCommonParmStr(HelmCmd cmd, Map<HelmFlags, String> values) {
		if (MapUtils.isEmpty(values)) {
			return cmd.getCmd();
		}
		HelmFlags[] flags = HelmFlags.values();
		// 检查不为空
		for (HelmFlags helmFlags : flags) {
			if (!helmFlags.notvalue && ArrayUtils.contains(helmFlags.getCmds(), cmd) && helmFlags.isNotnull()
					&& (values != null && StringUtil.isNull(values.get(helmFlags)))) {
				throw new ProjectExceptionRuntime(ExceptAll.param_error, helmFlags.getDesc() + "不允许为空");
			}
		}
		// 一些特殊检查
		switch (cmd) {
		case install:
			String chartLocalpath = values.get(HelmFlags.chartLocalpath);
			String chartFilePath = values.get(HelmFlags.chartFilePath);
			String version = values.get(HelmFlags.version);
			if (StringUtil.isNull(chartLocalpath) && (StringUtil.isNull(chartFilePath) || StringUtil.isNull(version))) {
				throw new ProjectExceptionRuntime(ExceptAll.param_error,
						"安装chart需要传入chartLocalpath或 chartFilePath和version的组合");
			}
			break;
		case chartfetch:
			if (StringUtil.isNull(values.get(HelmFlags.chartFilePath))
					|| StringUtil.isNull(values.get(HelmFlags.version))) {
				throw new ProjectExceptionRuntime(ExceptAll.param_error, "chartfetch需要chartFilePath和version的组合");
			}
			break;
		default:
			break;
		}
		// 冲突参数
		if (StringUtil.isNotNull(values.get(HelmFlags.namespace)) && values.containsKey(HelmFlags.namespace_all)) {// 如果有namespace，那么需要删除namespace_all
			values.remove(HelmFlags.namespace_all);
		}
		Arrays.parallelSort(flags, new Comparator<HelmFlags>() {
			@Override
			public int compare(HelmFlags o1, HelmFlags o2) {
				return o1.getOrderby() - o2.getOrderby();
			}
		});
		StringBuffer buff = new StringBuffer(cmd.getCmd());
		if (MapUtils.isNotEmpty(values)) {
			for (int i = 0; i < flags.length; i++) {
				if (!ArrayUtils.contains(flags[i].getCmds(), cmd)) {
					continue;// 不是此命令的flag
				}
				if (values.containsKey(flags[i])) {// 有传配置项
					if (flags[i].isNotvalue()) {// 不需要值
						if (flags[i].getParamShortName() != null) {
							buff.append(String.format(" -%s", flags[i].getParamShortName()));
						} else {
							buff.append(String.format(" --%s", flags[i].getParamName()));
						}
					} else if (StringUtil.isNotNull(values.get(flags[i]))) {// 需要值且传过来的值不为空才处理
						String value = flags[i].getFormart() == null ? values.get(flags[i])
								: flags[i].getFormart().getStr(values.get(flags[i]));
						if (StringUtil.isNull(flags[i].getParamName()) && StringUtil.isNull(flags[i].paramShortName)) {// 需要特殊处理
							buff.append(" " + value);
							continue;
						}
						String[] valueAry = flags[i] == HelmFlags.set ? value.split(",") : new String[] { value };// set是以数组的形式出现，逗号分隔
						for (String valueEle : valueAry) {
							if (flags[i].getParamShortName() != null) {
								buff.append(String.format(" -%s %s", flags[i].getParamShortName(), valueEle));
							} else {
								buff.append(String.format(" --%s %s", flags[i].getParamName(), valueEle));
							}
						}
					}
				}
			}
		}
		return buff.toString();
	}

	private HelmFlags(String desc, String paramName, String paramShortName, boolean notnull,
			IConvertValue<String> formart, HelmCmd[] cmds, int orderby, boolean notvalue) {
		this.desc = desc;
		this.paramName = paramName;
		this.paramShortName = paramShortName;
		this.notnull = notnull;
		this.formart = formart;
		this.cmds = cmds;
		this.orderby = orderby;
		this.notvalue = notvalue;
	}

	private HelmFlags(String desc, String paramName, String paramShortName, boolean notnull,
			IConvertValue<String> formart, HelmCmd[] cmds, int orderby) {
		this(desc, paramName, paramShortName, notnull, formart, cmds, orderby, false);
	}

	private HelmFlags(String desc, String paramName, boolean notnull, HelmCmd[] cmds) {
		this(desc, paramName, null, notnull, null, cmds, 0);
	}

	public String getDesc() {
		return desc;
	}

	public String getName() {
		return this.name();
	}

	@Override
	public String getDesc_zh() {
		return this.desc;
	}

	@Override
	public String getDesc_en() {
		return this.name();
	}

	public String getParamName() {
		return paramName;
	}

	public String getParamShortName() {
		return paramShortName;
	}

	public boolean isNotnull() {
		return notnull;
	}

	public IConvertValue<String> getFormart() {
		return formart;
	}

	public HelmCmd[] getCmds() {
		return cmds;
	}

	public int getOrderby() {
		return orderby;
	}

	public boolean isNotvalue() {
		return notvalue;
	}

}
