/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.wicp.tams.common.kubernetes.driver;

import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.SQLFeatureNotSupportedException;
import java.sql.SQLWarning;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.apache.calcite.sql.SqlDrop;
import org.apache.calcite.sql.SqlInsert;
import org.apache.calcite.sql.SqlUpdate;
import org.apache.calcite.sql.ddl.SqlCreateSchema;
import org.apache.calcite.sql.ddl.SqlDropSchema;
import org.slf4j.MDC;

import net.wicp.tams.common.apiext.CollectionUtil;
import net.wicp.tams.common.apiext.StringUtil;
import net.wicp.tams.common.beans.TbSimple;
import net.wicp.tams.common.kubernetes.apiserver.KubeClientTams;
import net.wicp.tams.common.kubernetes.beans.KubernetesResultCol;
import net.wicp.tams.common.kubernetes.beans.KubernetesResultPo;
import net.wicp.tams.common.kubernetes.constant.ResourcesType;

/**
 * Flink JDBC statement.beeline
 */
public class KubernetesStatement implements Statement {

	private final KubeClientTams clientTams;
	private final KubernetesConnection kubernetesConnection;
	private int fetchSize;
	private long maxRows;
	private KubernetesResultSet kubernetesResultSet;// execQuery的结果,影响的数量也在这里拿
	private SqlParserAssit sqlParserTams;

	public KubernetesStatement(KubernetesConnection kubernetesConnection) {
		this.clientTams = kubernetesConnection.getClientTams();
		this.kubernetesConnection = kubernetesConnection;
		this.fetchSize = 0;
		this.maxRows = 0;
	}

	/***
	 * 注意：select在flink里拿到的结果不一定正确，sqlgate接口这块非常不稳定
	 */
	@Override
	public ResultSet executeQuery(String sql) throws SQLException {
		checkClosed();
		chekSqlAndInit(sql);
		MDC.put("execMethod", "executeQuery");
		if (!this.sqlParserTams.isQuery()) {
			throw new SQLException("此方法只接受查询的SQL");
		}
		this.kubernetesResultSet = null;
		execute(sql);
		MDC.remove("execMethod");
		return this.kubernetesResultSet;
	}

	@Override
	public int executeUpdate(String sql) throws SQLException {
		checkClosed();
		chekSqlAndInit(sql);
		MDC.put("execMethod", "executeUpdate");
		if (this.sqlParserTams.isQuery()) {
			throw new SQLException("此方法只接受非查询的SQL");
		}
		this.kubernetesResultSet = null;
		execute(sql);
		MDC.remove("execMethod");
		String[] simpleData = this.kubernetesResultSet.getKubernetesResultPo().getSimpleData();
		if ("OK".equals(simpleData[0])) {
			return 0;
		}
		return 0;
	}

	// true:是查询，false:不是查询
	@Override
	public boolean execute(String sql) throws SQLException {
		checkClosed();
		TbSimple tbSimple = chekSqlAndInit(sql);
		KubernetesResultPo po = null;
		if (!this.sqlParserTams.isQuery()) {
			this.sqlParserTams.doDdl(this.clientTams);// 做动作
			List<Map<String, String>> curData = new ArrayList<Map<String, String>>();
			curData.add(CollectionUtil.newMapStr("result", "ok"));
			po = new KubernetesResultPo(new String[] { "result" }, curData);
		} else {
			po = new KubernetesResultPo(this.clientTams, ResourcesType.find(tbSimple.getTb()), sqlParserTams);
		}
		this.kubernetesResultSet = new KubernetesResultSet(po, fetchSize);// 调用查询
		return this.sqlParserTams.isQuery();
	}

	@Override
	public void close() throws SQLException {
		if (this.clientTams.isClosed()) {
			return;
		}
		cancel();
	}

	@Override
	public int getMaxFieldSize() throws SQLException {
		throw new SQLFeatureNotSupportedException("KubernetesStatement#getMaxFieldSize is not supported");
	}

	@Override
	public void setMaxFieldSize(int max) throws SQLException {
		throw new SQLFeatureNotSupportedException("KubernetesStatement#setMaxFieldSize is not supported");
	}

	@Override
	public int getMaxRows() throws SQLException {
		throw new SQLFeatureNotSupportedException("KubernetesStatement#getMaxRows is not supported");
	}

	@Override
	public void setMaxRows(int max) throws SQLException {
		setLargeMaxRows(max);
	}

	@Override
	public void setLargeMaxRows(long max) throws SQLException {
		checkClosed();
		if (max < 0) {
			throw new SQLException("Max rows must not be negative.");
		}
		this.maxRows = max;
	}

	@Override
	public void setEscapeProcessing(boolean enable) throws SQLException {
		throw new SQLFeatureNotSupportedException("KubernetesStatement#setEscapeProcessing is not supported");
	}

	@Override
	public int getQueryTimeout() throws SQLException {
		throw new SQLFeatureNotSupportedException("KubernetesStatement#getQueryTimeout is not supported");
	}

	@Override
	public void setQueryTimeout(int seconds) throws SQLException {
		throw new SQLFeatureNotSupportedException("KubernetesStatement#setQueryTimeout is not supported");
	}

	@Override
	public void cancel() throws SQLException {
		// 取消一个statement，如果整个statement被外层包裹可以使用（如：实现队列statement）
	}

	@Override
	public SQLWarning getWarnings() throws SQLException {
		return null;
	}

	@Override
	public void clearWarnings() throws SQLException {

	}

	@Override
	public void setCursorName(String name) throws SQLException {
		throw new SQLFeatureNotSupportedException("KubernetesStatement#setCursorName is not supported");
	}

	private TbSimple chekSqlAndInit(String sql) throws SQLException {
		String execMethod = MDC.get("execMethod");
		if (StringUtil.isNotNull(execMethod)) {
			return this.sqlParserTams.findTbInfos().get(0);
		}
		try {
			// TODO show desc都不支持
			this.sqlParserTams = new SqlParserAssit(this.kubernetesConnection.getCatalog(),
					this.kubernetesConnection.getSchema(), sql);
		} catch (Exception e) {
			throw new SQLException("语法错误，请检查是否符合mysql语法。", e);
		}
		if (this.sqlParserTams.getSqlNode() instanceof SqlCreateSchema
				|| this.sqlParserTams.getSqlNode() instanceof SqlDropSchema) {// 创库操作不需要做其它检查
			return null;
		}

		// 检查sql
		List<TbSimple> tbInfos = this.sqlParserTams.findTbInfos();
		if (tbInfos.size() > 1) {
			StringBuffer buff = new StringBuffer();
			for (TbSimple tbSimple : tbInfos) {
				buff.append(String.format("%s.%s,", tbSimple.getDb(), tbSimple.getTb()));
			}
			throw new SQLException(String.format("暂不支持多表操作:%s请确认", buff.toString()));
		}
		TbSimple tbSimple = tbInfos.get(0);
		ResourcesType resourcesType = ResourcesType.find(tbSimple.getTb());
		if (resourcesType == ResourcesType.UNKNOWN) {
			throw new SQLException("暂不支持此CRD的操作:" + tbSimple.getTb());
		}
		if (this.sqlParserTams.isQuery()) {
			List<String> colNames = this.sqlParserTams.findColNames();
			if (!colNames.contains("*")) {
				List<KubernetesResultCol> jdbcCols = resourcesType == ResourcesType.CRD
						? this.clientTams.getCrdColDef(tbSimple.getTb())
						: resourcesType.getJdbcColDef();
				StringBuffer buff = new StringBuffer();
				for (String colName : colNames) {
					if (!jdbcCols.contains(new KubernetesResultCol(colName))) {
						buff.append(String.format("%s,", colName));
					}
				}
				if (buff.length() > 0) {
					throw new SQLException(String.format("列没有定义:%s请确认", buff.toString()));
				}
			}
			return tbSimple;
		} else {
			if (resourcesType == ResourcesType.POD) {// pod不允许修改和新增
				if (this.sqlParserTams.getSqlNode() instanceof SqlInsert
						|| this.sqlParserTams.getSqlNode() instanceof SqlUpdate) {
					throw new SQLException("pod不允许新增和修改");
				}
			}
			return null;
		}
	}

	@Override
	public ResultSet getResultSet() throws SQLException {
		checkClosed();
		return this.kubernetesResultSet;
	}

	@Override
	public int getUpdateCount() throws SQLException {
		checkClosed();
		try {
			String resultColData = this.kubernetesResultSet.getKubernetesResultPo().getResultColData();
			return Integer.parseInt(resultColData);
		} catch (Throwable e) {
			throw new SQLException("Current result is not an update count.");
		}
	}

	@Override
	public boolean getMoreResults() throws SQLException {
		return this.sqlParserTams.isQuery();
	}

	@Override
	public void setFetchDirection(int direction) throws SQLException {
		throw new SQLFeatureNotSupportedException("KubernetesStatement#setFetchDirection is not supported");
	}

	@Override
	public int getFetchDirection() throws SQLException {
		throw new SQLFeatureNotSupportedException("KubernetesStatement#getFetchDirection is not supported");
	}

	@Override
	public void setFetchSize(int rows) throws SQLException {
		checkClosed();
		if (rows < 0) {
			throw new SQLException("Fetch size must not be negative.");
		}
		fetchSize = rows;
	}

	@Override
	public int getFetchSize() throws SQLException {
		checkClosed();
		return fetchSize;
	}

	@Override
	public int getResultSetConcurrency() throws SQLException {
		throw new SQLFeatureNotSupportedException("KubernetesStatement#getResultSetConcurrency is not supported");
	}

	@Override
	public int getResultSetType() throws SQLException {
		return ResultSet.TYPE_FORWARD_ONLY;
	}

	@Override
	public void addBatch(String sql) throws SQLException {
		throw new SQLFeatureNotSupportedException("KubernetesStatement#addBatch is not supported");
	}

	@Override
	public void clearBatch() throws SQLException {
		throw new SQLFeatureNotSupportedException("KubernetesStatement#clearBatch is not supported");
	}

	@Override
	public int[] executeBatch() throws SQLException {
		throw new SQLFeatureNotSupportedException("KubernetesStatement#executeBatch is not supported");
	}

	@Override
	public Connection getConnection() throws SQLException {
		return this.kubernetesConnection;
	}

	@Override
	public boolean getMoreResults(int current) throws SQLException {
		throw new SQLFeatureNotSupportedException("KubernetesStatement#getMoreResults is not supported");
	}

	@Override
	public ResultSet getGeneratedKeys() throws SQLException {
		throw new SQLFeatureNotSupportedException("KubernetesStatement#getGeneratedKeys is not supported");
	}

	@Override
	public int executeUpdate(String sql, int autoGeneratedKeys) throws SQLException {
		return (int) executeLargeUpdate(sql, autoGeneratedKeys);
	}

	@Override
	public int executeUpdate(String sql, int[] columnIndexes) throws SQLException {
		return (int) executeLargeUpdate(sql, columnIndexes);
	}

	@Override
	public int executeUpdate(String sql, String[] columnNames) throws SQLException {
		return (int) executeLargeUpdate(sql, columnNames);
	}

	@Override
	public boolean execute(String sql, int autoGeneratedKeys) throws SQLException {
		throw new SQLFeatureNotSupportedException("KubernetesStatement#execute is not supported");
	}

	@Override
	public boolean execute(String sql, int[] columnIndexes) throws SQLException {
		throw new SQLFeatureNotSupportedException("KubernetesStatement#execute is not supported");
	}

	@Override
	public boolean execute(String sql, String[] columnNames) throws SQLException {
		throw new SQLFeatureNotSupportedException("KubernetesStatement#execute is not supported");
	}

	@Override
	public int getResultSetHoldability() throws SQLException {
		throw new SQLFeatureNotSupportedException("KubernetesStatement#getResultSetHoldability is not supported");
	}

	@Override
	public boolean isClosed() throws SQLException {
		return this.clientTams.isClosed();
	}

	@Override
	public void setPoolable(boolean poolable) throws SQLException {
		throw new SQLFeatureNotSupportedException("KubernetesStatement#setPoolable is not supported");
	}

	@Override
	public boolean isPoolable() throws SQLException {
		throw new SQLFeatureNotSupportedException("KubernetesStatement#isPoolable is not supported");
	}

	@Override
	public void closeOnCompletion() throws SQLException {
		throw new SQLFeatureNotSupportedException("KubernetesStatement#closeOnCompletion is not supported");
	}

	@Override
	public boolean isCloseOnCompletion() throws SQLException {
		throw new SQLFeatureNotSupportedException("KubernetesStatement#isCloseOnCompletion is not supported");
	}

	@Override
	public <T> T unwrap(Class<T> iface) throws SQLException {
		throw new SQLFeatureNotSupportedException("KubernetesStatement#isCloseOnCompletion is not supported");
	}

	@Override
	public boolean isWrapperFor(Class<?> iface) throws SQLException {
		throw new SQLFeatureNotSupportedException("KubernetesStatement#isCloseOnCompletion is not supported");
	}

	protected void checkClosed() throws SQLException {

		if (clientTams.isClosed()) {
			throw new SQLException("This result set is already closed");
		}
	}

}
