package net.wicp.tams.common.kubernetes.constant;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;

import com.fasterxml.jackson.core.JsonPointer;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.JsonNodeFactory;
import com.fasterxml.jackson.databind.node.ObjectNode;

import io.fabric8.kubernetes.api.model.HasMetadata;
import net.wicp.tams.common.apiext.DateUtil;
import net.wicp.tams.common.apiext.StringUtil;
import net.wicp.tams.common.apiext.json.JSONUtil;
import net.wicp.tams.common.constant.DateFormatCase;
import net.wicp.tams.common.constant.dic.intf.IEnumCombobox;
import net.wicp.tams.common.exception.ExceptAll;
import net.wicp.tams.common.exception.ProjectExceptionRuntime;
import net.wicp.tams.common.kubernetes.apiserver.KubeClientTams;
import net.wicp.tams.common.kubernetes.beans.KubernetesResultCol;

/***
 * k8s的资源类型
 * 
 * @author andy.zhou
 * 
 */
public enum ResourcesType implements IEnumCombobox {
	SERVICE("网络", "Service", new String[] { "NAME", "TYPE", "CLUSTER-IP", "EXTERNAL-IP", "PORT(S)", "AGE" },
			new String[] { "/spec/clusterIP:clusterIP", "/spec/type:type", "/spec/ports:ports",
					"/spec/selector:selector", "/spec/sessionAffinity:sessionAffinity" },
			new String[] { "jdbc" }),

	DEPLOYMENT("部署", "Deployment", new String[] { "NAME", "DESIRED", "CURRENT", "UP-TO-DATE", "AVAILABLE", "AGE" },
			new String[] { "/spec/replicas:replicas", "template/spec/containers/0/image:image",
					"/spec/template/spec/containers/0/ports:ports",
					"/spec/template/spec/containers/0/resources:resources",
					"/spec/template/spec/imagePullSecrets:imagePullSecrets" },
			new String[] { "jdbc" }),

	POD("容器", "Pod", new String[] { "NAME", "READY", "STATUS", "RESTARTS", "AGE" },
			new String[] { "/spec/containers/0/image:image", "/spec/containers/0/ports:ports",
					"/spec/containers/0/resources:resources", "/status/phase:status", "/status/startTime:age" },
			new String[] { "jdbc" }),

	CRD("自定义资源", "crd", null, null, new String[] { "jdbc" }), // 还需要一个名称

	UNKNOWN("不支持的类型", "Unknown", new String[] {}, new String[] {}, new String[] {});

	private final String desc;
	private final String desc_en;
	private final String[] kubectlColNames;
	private final String[] jdbcMetaCols = new String[] { "name", "namespace" };

	public String[] getColValue(HasMetadata queryResByPage, String... colNames) {
		if (ArrayUtils.isEmpty(colNames)) {
			return new String[0];
		}
		List<KubernetesResultCol> colSort = getSortCol(colNames);
		String[] ret = getColValue(queryResByPage, colSort);
		return ret;
	}

	// 只有CRD才走这条路
	public String[] getColValue(HasMetadata queryResByPage, List<KubernetesResultCol> colSort) {
		String[] ret = new String[colSort.size()];
		ObjectMapper mapper = new ObjectMapper();
		JsonNode rootNode = mapper.valueToTree(queryResByPage);
		for (int i = 0; i < colSort.size(); i++) {
			JsonPointer jsonPointer = JsonPointer.compile(colSort.get(i).getJsonPath());
			JsonNode titleNode = rootNode.at(jsonPointer);
			if (!titleNode.isMissingNode()) {
				if (this == POD && "age".equalsIgnoreCase(colSort.get(i).getName())) {// 只有age做处理
					try {
						Date date = DateFormatCase.TZyyyyMMddHHmmss.getInstanc()
								.parse(titleNode.toString().replace("\"", ""));
						ret[i] = DateUtil.convertAgeFromTimeDif((System.currentTimeMillis() - date.getTime()) / 1000);
					} catch (Exception e) {
						ret[i] = titleNode.toString();
					}
				} else {
					ret[i] = titleNode.toString();
				}

//				if (titleNode.isObject()) {
//					ret[i] = titleNode.toString();
//				} else if (titleNode instanceof ObjectNode) {
//					ret[i] = titleNode.asText();
//				} else if (titleNode instanceof ArrayNode) {
//					ret[i] = titleNode.toString();
//				} else {
//					// 不区分类型isTextual
//					ret[i] = titleNode.toString();
//				}
			}
		}
		return ret;
	}

	/***
	 * 设置值
	 * 
	 * @param <T>
	 * @param oriObj
	 * @param crd
	 * @param classz
	 * @return
	 */
	public <T extends HasMetadata> T setColValue(HasMetadata oriObj, Map<String, Object> crd, Class<T> classz,
			boolean needMust) {
		ObjectMapper mapper = new ObjectMapper();
		ObjectNode rootNode = (ObjectNode) mapper.valueToTree(oriObj);
		List<KubernetesResultCol> jdbcColDefs = this.getJdbcColDef();
		for (KubernetesResultCol jdbcColDef : jdbcColDefs) {
			// JsonPointer jsonPointer = JsonPointer.compile(jdbcColDef.getJsonPath());
			if (crd.containsKey(jdbcColDef.getName())) {
				JSONUtil.setValueByPath(rootNode, jdbcColDef.getJsonPath(), crd.get(jdbcColDef.getName()));
			}
		}
		if (needMust) {
			if (this == DEPLOYMENT) {
				String labelValue = String.format("uuid-%s", crd.get("name"));
				ObjectNode lable = JsonNodeFactory.instance.objectNode();
				lable.put("app", labelValue);
				lable.put("from", "tams-jdbc");
				JSONUtil.setValueByPath(rootNode, "/spec/template/spec/containers/0/name", crd.get("name"));
				JSONUtil.setValueByPath(rootNode, "/spec/template/metadata/labels", lable);
				JSONUtil.setValueByPath(rootNode, "/spec/selector/matchLabels", lable);
			}
		}
		try {
			return mapper.treeToValue(rootNode, classz);
		} catch (Exception e) {
			throw new ProjectExceptionRuntime(ExceptAll.param_error);
		}
	}

	public List<KubernetesResultCol> getSortCol(String... colNames) {
		return getSortCol(this.getJdbcColDef(), colNames);
	}

	public List<KubernetesResultCol> getSortCol(List<KubernetesResultCol> oriCol, String... colNames) {
		if (ArrayUtils.contains(colNames, "*")) {
			return oriCol;
		}
		List<KubernetesResultCol> colSort = new ArrayList<KubernetesResultCol>();
		for (String colName : colNames) {
			int indexOf = oriCol.indexOf(new KubernetesResultCol(colName.toLowerCase()));
			if (indexOf < 0) {
				throw new ProjectExceptionRuntime(ExceptAll.Project_default, "列不存在:" + colName);
			}
			colSort.add(oriCol.get(indexOf));
		}
		return colSort;
	}

	public String[] getColValueAll(HasMetadata queryResByPage) {
		return getColValue(queryResByPage, this.getColNames());
	}

	private final String[] jdbcSpecCols;
	private final String[] groupSel;

	private ResourcesType(String desc, String desc_en, String[] kubectlColNames, String[] jdbcSpecCols,
			String[] groupSel) {
		this.desc = desc;
		this.desc_en = desc_en;
		this.kubectlColNames = kubectlColNames;
		this.jdbcSpecCols = jdbcSpecCols;
		this.groupSel = groupSel;
	}

	public static List<ResourcesType> getByGroup(String groupName) {
		List<ResourcesType> retlist = new ArrayList<ResourcesType>();
		for (ResourcesType resourcesType : ResourcesType.values()) {
			if (ArrayUtils.contains(resourcesType.getGroupSel(), groupName)) {
				retlist.add(resourcesType);
			}
		}
		return retlist;
	}

	public List<String> getJdbcCols() {
		List<KubernetesResultCol> jdbcColDef = getJdbcColDef();
		return jdbcColDef.stream().map((e) -> e.getName()).collect(Collectors.toList());
	}

	public List<KubernetesResultCol> getJdbcColDef() {
		List<KubernetesResultCol> ret = new ArrayList<KubernetesResultCol>();
		for (String metacol : this.jdbcMetaCols) {
			KubernetesResultCol temp = new KubernetesResultCol(metacol);
			temp.setJsonPath(String.format("/metadata/%s", metacol));
			ret.add(temp);
		}
		for (String col : this.jdbcSpecCols) {
			String[] cosEles = col.split(":");
			KubernetesResultCol temp = new KubernetesResultCol(cosEles.length > 1 ? cosEles[1] : cosEles[0]);
			temp.setJsonPath(cosEles[0]);
			ret.add(temp);
		}
		return ret;
	}

	public String getDesc() {
		return desc;
	}

	public String getName() {
		return this.name();
	}

	public String[] getColNames() {
		return kubectlColNames;
	}

	@Override
	public String getDesc_zh() {
		return this.desc;
	}

	@Override
	public String getDesc_en() {
		return this.desc_en;
	}

	/***
	 * kubectl get service 得到的默认字段
	 * 
	 * @param valueStr
	 * @return
	 */
	public String getKubectlColValue(String valueStr, String colName) {
		int indexOf = ArrayUtils.indexOf(this.kubectlColNames, colName);
		if (StringUtil.isNull(valueStr) || indexOf < 0) {
			return "";
		}
		String[] mens = valueStr.replaceAll("\\s+", "|").split("\\|");
		if ("FAILED".equals(valueStr)) {// 当没有这个任务时会返回
			return valueStr;
		} else {
			return mens[indexOf];
		}
	}

	public static ResourcesType find(String name) {
		if (StringUtils.isEmpty(name)) {
			return UNKNOWN;
		}
		for (ResourcesType ele : ResourcesType.values()) {
			if (name.equalsIgnoreCase(ele.name())) {
				return ele;
			}
		}
		boolean isCrd = KubeClientTams.cusCache.containsKey(name);
		return isCrd ? CRD : UNKNOWN;
	}

	public String[] getGroupSel() {
		return groupSel;
	}

}
