package net.wicp.tams.common.kubernetes.beans;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.ArrayUtils;

import lombok.AccessLevel;
import lombok.Data;
import lombok.Setter;
import net.wicp.tams.common.apiext.CollectionUtil;
import net.wicp.tams.common.apiext.StringUtil;
import net.wicp.tams.common.exception.ExceptAll;
import net.wicp.tams.common.exception.ProjectExceptionRuntime;
import net.wicp.tams.common.kubernetes.apiserver.KubeClientTams;
import net.wicp.tams.common.kubernetes.constant.ResourcesType;
import net.wicp.tams.common.kubernetes.driver.SqlParserAssit;

@Data
public class KubernetesResultPo {
	private List<KubernetesResultCol> cols;// 列相关信息
	@Setter(value = AccessLevel.PRIVATE)
	private String[] colNames;// cols的名字
	@Setter(value = AccessLevel.PRIVATE)
	private List<String[]> curData = new ArrayList<String[]>();// 当前页
	@Setter(value = AccessLevel.PRIVATE)
	private int curPage;
	@Setter(value = AccessLevel.PRIVATE)
	private int cursor = -1;// 当前所在位置
	private String errors;// 如果是错误则有信息，否则是成功

	private final ResourcesType resourcesType;// 只允许单表查询

	private final KubeClientTams kubeClientTams;

	private final SqlParserAssit sqlParserTams;

	private final boolean isQuery;

	// private final String crdName;// 只有自定义CRD时启作用

	public KubernetesResultPo(KubeClientTams kubeClientTams, ResourcesType resourcesType,
			SqlParserAssit sqlParserTams) {
		this.resourcesType = resourcesType;
		if (resourcesType == ResourcesType.CRD) {
			this.cols = resourcesType.getSortCol(
					kubeClientTams.getCrdColDef(sqlParserTams.findTbInfos().get(0).getTb()),
					sqlParserTams.findColNames().toArray(new String[sqlParserTams.findColNames().size()]));
		} else {
			this.cols = resourcesType
					.getSortCol(sqlParserTams.findColNames().toArray(new String[sqlParserTams.findColNames().size()]));
		}
		this.kubeClientTams = kubeClientTams;
		this.sqlParserTams = sqlParserTams;
		this.isQuery = sqlParserTams.isQuery();
		this.curData = kubeClientTams.statementsByPage(resourcesType, this.sqlParserTams, 0, 100);// 初始化数据
	}

	public KubernetesResultPo(String[] colNames, List<Map<String, String>> curData) {
		this.resourcesType = null;
		this.kubeClientTams = null;
		this.sqlParserTams = null;
		this.isQuery = false;
		this.initData(colNames, curData);
	}

	public void initData(String[] colNames, List<Map<String, String>> curData) {
		this.colNames = colNames;
		if (curData != null) {
			for (Map<String, String> map : curData) {
				String[] curDate = new String[colNames.length];
				for (int i = 0; i < colNames.length; i++) {
					if (map.containsKey(colNames[i])) {
						curDate[i] = map.get(colNames[i]);
					} else {
						curDate[i] = null;
					}
				}
				this.curData.add(curDate);
			}
		}
		this.cols = new ArrayList<KubernetesResultCol>();
		for (int i = 0; i < colNames.length; i++) {
			this.cols.add(new KubernetesResultCol(this.colNames[i], "varchar"));
		}
	}

	public void beforeFirst() {
		this.cursor = -1;
	}

	public void afterLast() {
		this.cursor = curData.size() - 1;
	}

	public boolean isSuc() {
		return StringUtil.isNull(this.errors);
	}

	public String[] getColNames() {
		if (!isSuc()) {// 非成功的结果
			return null;
		}
		if (colNames != null && colNames.length == cols.size()) {
			return colNames;
		} else {
			List<?> colFromObj = CollectionUtil.getColFromObj(cols, "name");
			this.colNames = colFromObj.toArray(new String[colFromObj.size()]);
			return this.colNames;
		}
	}

	// 得到当前行数据
	public String[] getDataCurRow() {
		if (!isSuc()) {// 非成功的结果
			return null;
		}
		if (cursor == -1) {
			throw new ProjectExceptionRuntime(ExceptAll.jdbc_sql_notOpenRs);
		}
		return curData.get(cursor);
	}

	// 只得到第一列数据，适合于简单查贸易
	public String[] getSimpleData() {
		if (!isSuc()) {// 非成功的结果
			return null;
		}
		String[] ret = new String[this.curData.size()];
		for (int i = 0; i < this.curData.size(); i++) {
			ret[i] = this.curData.get(i)[0];
		}
		return ret;
	}

	public String getResultColData() {
		this.cursor = 0;// 有且只有一行
		String dataColByCurRow = getDataColByCurRow("result");
		return dataColByCurRow;
	}

	public String getDataColByCurRow(String col) {
		if (!isSuc()) {// 非成功的结果
			return null;
		}
		int indexOf = ArrayUtils.indexOf(getColNames(), col);
		if (indexOf == -1) {
			throw new ProjectExceptionRuntime(ExceptAll.Project_default, "没有此列");
		}
		return getDataCurRow()[indexOf];
	}

	@Setter(value = AccessLevel.PRIVATE)
	private long maxCursor = 0;

	public boolean next(long maxRows) {
		if (!isSuc()) {// 非成功的结果
			return false;
		}
		if (++maxCursor > maxRows) {// 超过了最大行数
			return false;
		}
		if (cursor < curData.size() - 1) {// cursor从0开始，因为游标要下移一位，所以要-1
			this.cursor++;
			return true;
		} else {// 翻到下一页。
			if (!isQuery) {// 非查询类不需要翻页了
				return false;
			}
			// 正常结束且没拿到一条结果也算结束
			if (CollectionUtils.isEmpty(this.curData)||this.curData.size()<maxRows) {// 没有查到或查到的数据量小于每批的量时就应该结束
				return false;
			} else {
				int nextPage = this.getCurPage() + 1;
				List<String[]> result = kubeClientTams.statementsByPage(resourcesType, this.sqlParserTams, nextPage,
						maxRows);
				this.setCurPage(nextPage);
				this.setErrors("");
				this.setMaxCursor(0);
				this.setCurData(result);
				return CollectionUtils.isNotEmpty(result);
			}
		}
	}

}
