package net.wicp.tams.common.kubernetes.apiserver;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.regex.Pattern;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.collections.Predicate;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.apache.commons.lang3.tuple.Pair;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;

import io.fabric8.kubernetes.api.model.Container;
import io.fabric8.kubernetes.api.model.ContainerPort;
import io.fabric8.kubernetes.api.model.DeletionPropagation;
import io.fabric8.kubernetes.api.model.GenericKubernetesResource;
import io.fabric8.kubernetes.api.model.GenericKubernetesResourceList;
import io.fabric8.kubernetes.api.model.HasMetadata;
import io.fabric8.kubernetes.api.model.IntOrString;
import io.fabric8.kubernetes.api.model.KubernetesResourceList;
import io.fabric8.kubernetes.api.model.ListOptions;
import io.fabric8.kubernetes.api.model.ListOptionsBuilder;
import io.fabric8.kubernetes.api.model.Namespace;
import io.fabric8.kubernetes.api.model.NamespaceBuilder;
import io.fabric8.kubernetes.api.model.NamespaceList;
import io.fabric8.kubernetes.api.model.ObjectMeta;
import io.fabric8.kubernetes.api.model.OwnerReference;
import io.fabric8.kubernetes.api.model.PersistentVolumeClaimVolumeSource;
import io.fabric8.kubernetes.api.model.Pod;
import io.fabric8.kubernetes.api.model.PodList;
import io.fabric8.kubernetes.api.model.PodTemplateSpec;
import io.fabric8.kubernetes.api.model.Quantity;
import io.fabric8.kubernetes.api.model.ReplicationController;
import io.fabric8.kubernetes.api.model.ReplicationControllerSpec;
import io.fabric8.kubernetes.api.model.ResourceQuota;
import io.fabric8.kubernetes.api.model.ResourceQuotaBuilder;
import io.fabric8.kubernetes.api.model.Secret;
import io.fabric8.kubernetes.api.model.SecretBuilder;
import io.fabric8.kubernetes.api.model.SecretList;
import io.fabric8.kubernetes.api.model.Service;
import io.fabric8.kubernetes.api.model.ServiceAccount;
import io.fabric8.kubernetes.api.model.ServiceBuilder;
import io.fabric8.kubernetes.api.model.ServiceList;
import io.fabric8.kubernetes.api.model.ServicePort;
import io.fabric8.kubernetes.api.model.ServiceSpec;
import io.fabric8.kubernetes.api.model.Volume;
import io.fabric8.kubernetes.api.model.apiextensions.v1.CustomResourceDefinition;
import io.fabric8.kubernetes.api.model.apiextensions.v1.JSONSchemaProps;
import io.fabric8.kubernetes.api.model.apps.Deployment;
import io.fabric8.kubernetes.api.model.apps.DeploymentList;
import io.fabric8.kubernetes.api.model.batch.v1.Job;
import io.fabric8.kubernetes.api.model.batch.v1.JobList;
import io.fabric8.kubernetes.client.ConfigBuilder;
import io.fabric8.kubernetes.client.KubernetesClient;
import io.fabric8.kubernetes.client.KubernetesClientBuilder;
import io.fabric8.kubernetes.client.KubernetesClientException;
import io.fabric8.kubernetes.client.V1ApiextensionAPIGroupDSL;
import io.fabric8.kubernetes.client.VersionInfo;
import io.fabric8.kubernetes.client.dsl.FilterNested;
import io.fabric8.kubernetes.client.dsl.FilterWatchListDeletable;
import io.fabric8.kubernetes.client.dsl.MixedOperation;
import io.fabric8.kubernetes.client.dsl.NonNamespaceOperation;
import io.fabric8.kubernetes.client.dsl.PodResource;
import io.fabric8.kubernetes.client.dsl.Resource;
import io.fabric8.kubernetes.client.dsl.RollableScalableResource;
import io.fabric8.kubernetes.client.dsl.ScalableResource;
import io.fabric8.kubernetes.client.dsl.ServiceResource;
import io.fabric8.kubernetes.client.dsl.base.CustomResourceDefinitionContext;
import io.kubernetes.client.Exec;
import io.kubernetes.client.PodLogs;
import io.kubernetes.client.openapi.ApiClient;
import io.kubernetes.client.openapi.ApiException;
import io.kubernetes.client.openapi.Configuration;
import io.kubernetes.client.openapi.apis.AppsV1Api;
import io.kubernetes.client.openapi.apis.AppsV1Api.APIlistNamespacedDeploymentRequest;
import io.kubernetes.client.openapi.apis.AppsV1Api.APIreplaceNamespacedDeploymentRequest;
import io.kubernetes.client.openapi.apis.NetworkingV1Api;
import io.kubernetes.client.openapi.apis.NetworkingV1Api.APIcreateNamespacedIngressRequest;
import io.kubernetes.client.openapi.apis.RbacAuthorizationV1Api;
import io.kubernetes.client.openapi.models.V1ClusterRoleBinding;
import io.kubernetes.client.openapi.models.V1Deployment;
import io.kubernetes.client.openapi.models.V1DeploymentSpec;
import io.kubernetes.client.openapi.models.V1HTTPIngressPath;
import io.kubernetes.client.openapi.models.V1HTTPIngressRuleValue;
import io.kubernetes.client.openapi.models.V1Ingress;
import io.kubernetes.client.openapi.models.V1IngressBackend;
import io.kubernetes.client.openapi.models.V1IngressRule;
import io.kubernetes.client.openapi.models.V1IngressServiceBackend;
import io.kubernetes.client.openapi.models.V1IngressSpec;
import io.kubernetes.client.openapi.models.V1IngressTLS;
import io.kubernetes.client.openapi.models.V1ObjectMeta;
import io.kubernetes.client.openapi.models.V1OwnerReference;
import io.kubernetes.client.openapi.models.V1RoleBinding;
import io.kubernetes.client.openapi.models.V1RoleRef;
import io.kubernetes.client.openapi.models.V1ServiceBackendPort;
import io.kubernetes.client.openapi.models.V1Subject;
import io.kubernetes.client.util.Yaml;
import lombok.extern.slf4j.Slf4j;
import net.wicp.tams.common.Conf;
import net.wicp.tams.common.Result;
import net.wicp.tams.common.apiext.CollectionUtil;
import net.wicp.tams.common.apiext.FreemarkUtil;
import net.wicp.tams.common.apiext.IOUtil;
import net.wicp.tams.common.apiext.NumberUtil;
import net.wicp.tams.common.apiext.PwdUtil;
import net.wicp.tams.common.apiext.ReflectAssist;
import net.wicp.tams.common.apiext.StringUtil;
import net.wicp.tams.common.apiext.yaml.YamlAssist;
import net.wicp.tams.common.beans.SqlWherePo;
import net.wicp.tams.common.beans.StreamCopyBean;
import net.wicp.tams.common.beans.TbSimple;
import net.wicp.tams.common.callback.IProcess;
import net.wicp.tams.common.constant.dic.YesOrNo;
import net.wicp.tams.common.exception.ExceptAll;
import net.wicp.tams.common.exception.ProjectException;
import net.wicp.tams.common.exception.ProjectExceptionRuntime;
import net.wicp.tams.common.kubernetes.apiserver.crd.CrdDefConfig;
import net.wicp.tams.common.kubernetes.apiserver.crd.ICrdDef;
import net.wicp.tams.common.kubernetes.beans.KubernetesResultCol;
import net.wicp.tams.common.kubernetes.beans.PodPhase;
import net.wicp.tams.common.kubernetes.beans.QuotaBean;
import net.wicp.tams.common.kubernetes.constant.ResourcesType;
import net.wicp.tams.common.kubernetes.driver.SqlParserAssit;

/***
 * 
 * @author Andy.zhou
 *
 */
@Slf4j
public class KubeClientTams {
	private final KubernetesClient client;
	private final ApiClient apiClient;
	private String defaultNamespace = Conf.get("common.kubernetes.apiserver.namespace.default");
	private final String GROUP_LABE = "tams/catalog";
	private final Set<String> groups = new HashSet<String>();// 相当于jdbc的catalog,在大的公司，需要把namespace进行分组处理
	public static final Map<String, ICrdDef> cusCache = new HashMap<String, ICrdDef>();// 自定义资源缓存
	private static final Map<String, List<KubernetesResultCol>> crdFieldCache = new HashMap<String, List<KubernetesResultCol>>();

	{
		groups.add(Conf.get("common.kubernetes.jdbc.catalogDefault"));
	}

	static {
		// 判断是否在容器中
		File tokenFile = new File("/var/run/secrets/kubernetes.io/serviceaccount/token");
		if (tokenFile.exists()) {
			Conf.overProp("common.kubernetes.incluster", "yes");
		} else {
			Conf.overProp("common.kubernetes.incluster", "no");
		}
		log.info("====================是否在容器中运行:{}", Conf.get("common.kubernetes.incluster"));

		Map<String, String> crdconf = Conf.getPre("common.kubernetes.crd.", true);
		Set<String> confKeys = new HashSet<String>();
		for (String key : crdconf.keySet()) {
			String[] keyAry = key.split("\\.");
			switch (keyAry[0]) {
			case "enum":
				Object[] enumConstants;
				try {
					enumConstants = Class.forName(crdconf.get(key)).getEnumConstants();
					for (Object enumConstant : enumConstants) {
						ICrdDef temp = (ICrdDef) enumConstant;
						cusCache.put(temp.getShortNames()[0], temp);
					}
				} catch (ClassNotFoundException e) {
					log.error("不能处理类：" + crdconf.get(key), e);
				}
				break;
			case "class":
				ICrdDef value = ReflectAssist.newInst(crdconf.get(key));
				cusCache.put(value.getShortNames()[0], value);
				break;
			case "conf":
				confKeys.add(keyAry[1]);
				break;
			default:
				break;
			}
			// 只有配置文件
			for (String confKey : confKeys) {
				ICrdDef value = new CrdDefConfig(confKey);
				cusCache.put(confKey, value);
			}
		}

		// ServiceLoader
//		ServiceLoader<ICrdDef> serviceLoader = ServiceLoader.load(ICrdDef.class);
//		for (ICrdDef iCrdDef : serviceLoader) {
//			if (iCrdDef instanceof Enum) {
//				ICrdDef[] enumConstants = iCrdDef.getClass().getEnumConstants();
//				for (ICrdDef enumConstant : enumConstants) {
//					cusCache.put(iCrdDef.getShortNames()[0], enumConstant);
//				}
//			} else {
//				cusCache.put(iCrdDef.getShortNames()[0], iCrdDef);
//			}
//		}
	}

	/***
	 * 
	 * @param crdName 配置属性common.kubernetes.crd.[crdName]
	 * @return
	 */
	public List<KubernetesResultCol> getCrdColDef(String crdName) {
		if (!cusCache.containsKey(crdName)) {
			throw new ProjectExceptionRuntime(ExceptAll.Project_default, "不支持此CRD" + crdName);
		}
		if (!crdFieldCache.containsKey(crdName)) {
			List<KubernetesResultCol> crdColDef = this.getCrdColDef(cusCache.get(crdName));
			crdFieldCache.put(crdName, crdColDef);
		}
		return crdFieldCache.get(crdName);
	}

	private void apiClientInit(ApiClient apiClient) {
		Configuration.setDefaultApiClient(apiClient);
		this.apiClient.setReadTimeout(300000);// 5分钟
	}

	public KubernetesClient getClient() {
		return client;
	}

	/***
	 * 通过Conf的配置得到client
	 */
	public KubeClientTams() {
		String incluster = Conf.get("common.kubernetes.incluster");
		if (!YesOrNo.yes.name().equals(incluster)) {// 不在集群中，需要检查有没有k8s的config文件。
			String k8sConfig = IOUtil.mergeFolderAndFilePath(System.getProperty("user.home"), "/.kube/config");
			if (!new File(k8sConfig).exists()) {
				throw new ProjectExceptionRuntime(ExceptAll.Project_default, "没有在k8s容器中运行，需要有k8s的认证文件：~/.kube/config");
			}
		}
		this.client = initClient();
		this.apiClient = ApiClientManager.getApiClient();
		apiClientInit(apiClient);
	}

	/**
	 * 通过配置文件的内容和context得到client
	 * 
	 * @param kubeconfigstr kubeconfig内容
	 * @param confContext       启用哪个context,传空则为当前context
	 */
	public KubeClientTams(String kubeconfigstr, String confContext) {
		Validate.notEmpty(kubeconfigstr, "k8s的认证文件不能为空");
		this.client = getClientForKube(kubeconfigstr, confContext);
		this.apiClient = new ApiClientCreator(kubeconfigstr, confContext).createObject();
		apiClientInit(apiClient);
	}

	/***
	 * 配置文件路径得到client
	 * 
	 * @param pathTypePath 配置文件
	 */
	public KubeClientTams(String pathTypePath) {
		this(IOUtil.slurpPathTypePath(pathTypePath), null);
	}

	// 不需要认证的方式，如agent
	public KubeClientTams(URI mastUrl) {
		// String masterurl = mastUrl.getPath();
		this(String.format(confTemp, mastUrl.toString()), null);
	}

	public KubeClientTams(String host, int port, String webContext) {
		this(String.format(confTemp,
				StringUtil.isNotNull(webContext) ? String.format("http://%s:%s/%s", host, port, webContext)
						: String.format("http://%s:%s", host, port)),
				null);
	}

	private static String confTemp = "apiVersion: v1\r\n" + "clusters:\r\n" + "- cluster:\r\n" + "    server: %s\r\n"
			+ "  name: kubernetes\r\n" + "contexts:\r\n" + "- context:\r\n" + "    cluster: kubernetes\r\n"
			+ "    user: kubernetes-admin\r\n" + "    namespace: ummc-ummc\r\n"
			+ "  name: kubernetes-admin@kubernetes\r\n" + "current-context: kubernetes-admin@kubernetes\r\n"
			+ "kind: Config\r\n" + "preferences: {}\r\n" + "users:\r\n" + "- name: kubernetes-admin\r\n" + "  user:\r\n"
			+ "    client-certificate-data: \r\n" + "    client-key-data:";

	public KubeClientTams(KubernetesClient client, ApiClient apiClient) {
		this.client = client;
		this.apiClient = apiClient;
		apiClientInit(apiClient);
	}

	private KubernetesClient initClient() {
		KubernetesClient client = null;
		if (StringUtil.isNotNull(Conf.get("common.kubernetes.kubeconfig.path"))) {
			String kubeconfigstr = IOUtil.slurp(Conf.get("common.kubernetes.kubeconfig.path"));
			String context = IOUtil.slurp(Conf.get("common.kubernetes.kubeconfig.context"));
			client = getClientForKube(kubeconfigstr, context);
		} else if (StringUtil.isNotNull(Conf.get("common.kubernetes.apiserver.master.url"))) {
			ConfigBuilder config = new ConfigBuilder()
					.withMasterUrl(Conf.get("common.kubernetes.apiserver.master.url"));
			if (StringUtil.isNotNull(Conf.get("common.kubernetes.apiserver.master.username"))) {
				config.withUsername(Conf.get("common.kubernetes.apiserver.master.username"));
				config.withPassword(Conf.get("common.kubernetes.apiserver.master.password"));
			}
			config.withTrustCerts(true);
			client = new KubernetesClientBuilder().withConfig(config.build()).build();

		} else {
			client = new KubernetesClientBuilder().build();// 默认读./kube信息，支持在容器内执行。
		}
		return client;
	}

	private KubernetesClient getClientForKube(String kubeconfigstr, String context) {
		try {
			io.fabric8.kubernetes.client.Config config = StringUtil.isNull(context)
					? io.fabric8.kubernetes.client.Config.fromKubeconfig(kubeconfigstr)
					: io.fabric8.kubernetes.client.Config.fromKubeconfig(context, kubeconfigstr, null);
			this.defaultNamespace = config.getNamespace();
			KubernetesClient client = new KubernetesClientBuilder().withConfig(config).build();
			return client;
		} catch (Exception e) {
			throw new ProjectExceptionRuntime(ExceptAll.k8s_conn_error, "", e);
		}
	}

	private String getUseNamespaceName(String namespace) {
		String namespaceTrue = StringUtil.isNull(namespace) ? defaultNamespace : namespace;
		return namespaceTrue;
	}

	public Namespace getNamespace(String namespace, String group) {
		Namespace myns = this.client.namespaces().withName(namespace).get();
		if (StringUtil.isNotNull(group) && !Conf.get("common.kubernetes.jdbc.catalogDefault").equalsIgnoreCase(group)) {// 不是默认group
			if (myns.getMetadata().getLabels().containsKey(GROUP_LABE)
					&& group.equalsIgnoreCase(myns.getMetadata().getLabels().get(GROUP_LABE))) {
				return myns;
			} else {
				return null;
			}
		} else {
			return myns;
		}
	}

	/****
	 * 得到所有名称空间
	 * 
	 * @param group 名称空间可分组
	 * @return
	 */
	public List<String> getAllNamespace(String group) {
		List<String> namespaceName = new ArrayList<String>();
		String _continue = null;
		ListOptions listOptions = new ListOptionsBuilder().withLimit(10L).withContinue(_continue).build();
		NonNamespaceOperation<Namespace, NamespaceList, Resource<Namespace>> namespaces = this.client.namespaces();
		do {
			NamespaceList namespacelist = null;
			if (StringUtil.isNotNull(group)) {
				namespacelist = namespaces.withLabel(GROUP_LABE, group).list(listOptions);
			} else {
				namespacelist = namespaces.list(listOptions);
			}
			namespacelist.getItems().stream().filter(new java.util.function.Predicate<Namespace>() {
				@Override
				public boolean test(Namespace t) {
					return "Active".equals(t.getStatus().getPhase());
				}
			}).forEach(obj -> {
				namespaceName.add(obj.getMetadata().getName());
				Map<String, String> labels = obj.getMetadata().getLabels();
				if (labels.containsKey(GROUP_LABE)) {// 同时同步group
					this.groups.add(labels.get(GROUP_LABE));
				}
			});
			_continue = namespacelist.getMetadata().getContinue();
			listOptions.setContinue(_continue);
		} while (_continue != null);
		return namespaceName;
	}

	public List<String> getAllNamespace() {
		return getAllNamespace(null);
	}

	public ServiceList getServices(String namespace) {
		ServiceList myNsServices = this.client.services().inNamespace(getUseNamespaceName(namespace)).list();
		return myNsServices;
	}

	public ServiceList getServices() {
		return getServices(null);
	}

	public Service getService(String namespace, String serviceName) {
		Service myservice = this.client.services().inNamespace(getUseNamespaceName(namespace)).withName(serviceName)
				.get();
		return myservice;
	}

	/**
	 * 创建帐号对应的secret
	 * 
	 * @param namespace   帐号名称空间
	 * @param accountName
	 */
	private Secret createAccountSecret(String namespace, String accountName) {
		ObjectMeta metadata = new ObjectMeta();
		metadata.setName(String.format("%s-token", accountName));
		metadata.setNamespace(namespace);
		metadata.setAnnotations(CollectionUtil.newMapStr("kubernetes.io/service-account.name", accountName));
		Secret service = new SecretBuilder().withType("kubernetes.io/service-account-token").withMetadata(metadata)
				.build();
		Secret create = this.client.secrets().inNamespace(namespace).resource(service).create();
		return create;
	}

	// https://github.com/fabric8io/kubernetes-client/blob/master/kubernetes-examples/src/main/java/io/fabric8/kubernetes/examples/kubectl/equivalents/CustomResourceCreateDemoTypeless.java
	public GenericKubernetesResource createCusObject(ICrdDef crdVesion, String version, String namespace,
			InputStream fileInputStream) {
		try {
			CustomResourceDefinitionContext context = crdVesion.getCrd();
			NonNamespaceOperation<GenericKubernetesResource, GenericKubernetesResourceList, Resource<GenericKubernetesResource>> inNamespace = this.client
					.genericKubernetesResources(context).inNamespace(namespace);
			Resource<GenericKubernetesResource> dummyObject = inNamespace.load(fileInputStream);
			GenericKubernetesResource create = dummyObject.create();
			return create;
		} catch (Throwable e) {
			if (e.getMessage().contains("404 page not found")) {
				throw new ProjectExceptionRuntime(ExceptAll.k8s_api_pagenotfound, "没有布署api:" + e.getMessage());
			}
			if (e.getMessage().contains("already exists")) {
				throw new ProjectExceptionRuntime(ExceptAll.k8s_api_exists, "已存在此类型的同名资源:" + e.getMessage());
			}
			throw new ProjectExceptionRuntime(ExceptAll.k8s_api_other, "布署自定义的类型失败", e);
		}
	}

	// 未验证
	public void updateCrd(ICrdDef crdVesion, String namespace, String name, Map<String, Object> updatedSpec) {
		try {
			CustomResourceDefinitionContext context = crdVesion.getCrd();
			GenericKubernetesResource cr = this.client.genericKubernetesResources(context).inNamespace(namespace)
					.withName(name).get();
			@SuppressWarnings("unchecked")
			Map<String, Object> object = (Map<String, Object>) cr.getAdditionalProperties().get("spec");
			object.putAll(updatedSpec);
			this.client.genericKubernetesResources(context).resource(cr).update();
		} catch (Throwable e) {
			if (e.getMessage().contains("404 page not found")) {
				throw new RuntimeException("没有布署api:" + e.getMessage());
			}
			throw new RuntimeException("布署自定义的类型失败", e);
		}

	}

	public GenericKubernetesResource createCusObject(ICrdDef crdVesion, String namespace, InputStream fileInputStream) {
		return createCusObject(crdVesion, null, namespace, fileInputStream);
	}

	private GenericKubernetesResource createCusObject(ICrdDef crdVesion, String namespace,
			GenericKubernetesResource sorceCR) {
		try {
			CustomResourceDefinitionContext context = crdVesion.getCrd();
			NonNamespaceOperation<GenericKubernetesResource, GenericKubernetesResourceList, Resource<GenericKubernetesResource>> inNamespace = this.client
					.genericKubernetesResources(context).inNamespace(namespace);
			GenericKubernetesResource create = inNamespace.resource(sorceCR).create();
			return create;
		} catch (Throwable e) {
			if (e.getMessage().contains("404 page not found")) {
				throw new RuntimeException("没有布署api:" + e.getMessage());
			}
			if (e.getMessage().contains("already exists")) {
				throw new RuntimeException("已存在此类型的同名资源:" + e.getMessage());
			}
			throw new RuntimeException("布署自定义的类型失败", e);
		}
	}

	public GenericKubernetesResource createCusObject(ICrdDef crdVesion, String namespace, String name,
			Map<String, String> annotations, Map<String, String> labels, Map<String, Object> spec) {
		GenericKubernetesResource rs = new GenericKubernetesResource();
		rs.setKind(crdVesion.getKind());
		rs.setApiVersion(crdVesion.getApiVersion(0));
		ObjectMeta objectMeta = new ObjectMeta();
		objectMeta.setName(name);
		objectMeta.setNamespace(namespace);
		if (MapUtils.isNotEmpty(annotations)) {
			objectMeta.setAnnotations(annotations);
		}
		if (MapUtils.isNotEmpty(labels)) {
			objectMeta.setLabels(labels);
		}
		rs.setMetadata(objectMeta);
		Map<String, Object> additionalProperties = new HashMap<String, Object>();
		additionalProperties.put("spec", spec);
		rs.setAdditionalProperties(additionalProperties);
		return this.createCusObject(crdVesion, namespace, rs);
	}

//	ClusterRole有下面权限
//	- apiGroups:
//		  - apiextensions.k8s.io
//		  resources:
//		  - customresourcedefinitions
//		  verbs:
//		  - create
//		  - delete
//		  - get
//		  - list
//		  - patch
//		  - update
//		  - watch
	public List<KubernetesResultCol> getCrdColDef(ICrdDef crdVesion) {
		CustomResourceDefinition crdDef = client.apiextensions().v1().customResourceDefinitions()
				.withName(crdVesion.getCrdName()).get();
		List<KubernetesResultCol> retlist = new ArrayList<KubernetesResultCol>();
		if (crdDef == null) {
			return retlist;
		}
		LinkedHashMap<String, JSONSchemaProps> properties = (LinkedHashMap<String, JSONSchemaProps>) crdDef.getSpec()
				.getVersions().get(0).getSchema().getOpenAPIV3Schema().getProperties().get("spec").getProperties();
		KubernetesResultCol nameColDef = new KubernetesResultCol("name");
		nameColDef.setJsonPath("/metadata/name");
		KubernetesResultCol namespaceColDef = new KubernetesResultCol("namespace");
		namespaceColDef.setJsonPath("/metadata/namespace");
		retlist.add(nameColDef);
		retlist.add(namespaceColDef);
		// 保证顺序
		for (Map.Entry<String, JSONSchemaProps> entry : properties.entrySet()) {
			KubernetesResultCol temp = new KubernetesResultCol(entry.getKey(), entry.getValue().getType());
			temp.setJsonPath(String.format("/spec/%s", entry.getKey()));
			retlist.add(temp);
		}
		return retlist;
	}

	public boolean existCrd(ICrdDef crdVesion) {
		List<KubernetesResultCol> crdColDef = getCrdColDef(crdVesion);
		return CollectionUtils.isNotEmpty(crdColDef);
	}

	public void installCrd(InputStream is) {
		V1ApiextensionAPIGroupDSL v1 = client.apiextensions().v1();
		v1.customResourceDefinitions().load(is).create();
	}

	public void installCrd(String relaPath, Class<?> classz) {
		installCrd(IOUtil.fileToInputStream(relaPath, classz));
	}

	/****
	 * 查看日志
	 * 
	 * @param pod
	 * @return
	 */
	public InputStream viewLog(Pod pod) {
		PodLogs logs = new PodLogs(this.apiClient);
		// 5秒500行,不指定容器
		this.apiClient.setReadTimeout(300000);// 5分钟
		try {
			// InputStream is = logs.streamNamespacedPodLog(v1Pod);
			// 只查前500条日志
			InputStream is = logs.streamNamespacedPodLog(pod.getMetadata().getNamespace(), pod.getMetadata().getName(),
					pod.getSpec().getContainers().get(0).getName(), null, 500, false);
			// BufferedReader reader = new BufferedReader(new InputStreamReader(is));
			return is;
			// ByteStreams.copy(is, System.out);
			// String line;
			// while((line = reader.readLine()) != null) {
			// System.out.println("====="+line);
			// }
			// System.out.println(2222);
		} catch (Exception e) {
			log.error("获取日志失败", e);
			return null;
		}
	}

	/**
	 * 通过deploymentName随机查看其中一个pod的日志
	 * 
	 * @param namespace
	 * @param deploymentName
	 * @return
	 */
	public InputStream viewLog(String namespace, String deploymentName) {
		Pod pod = getPodByDeployment(namespace, deploymentName);
		return viewLog(pod);
	}

	/**
	 * 复制文件到pod
	 * 
	 * @param namespace
	 * @param deploymentName
	 * @param osFilepath
	 * @param podFilePath
	 * @return
	 */
	public Boolean copyFileToPod(String namespace, String deploymentName, String osFilepath, String podFilePath) {
		Pod pod = getPodByDeployment(namespace, deploymentName);
		Boolean upload = this.client.pods().inNamespace(pod.getMetadata().getNamespace())
				.withName(pod.getMetadata().getName()).file(podFilePath).upload(new File(osFilepath).toPath());
		return upload;
	}

	public Boolean copyFileToPod(String namespace, String deploymentName, InputStream inputStream, String podFilePath) {
		Pod pod = getPodByDeployment(namespace, deploymentName);
		Boolean upload = this.client.pods().inNamespace(pod.getMetadata().getNamespace())
				.withName(pod.getMetadata().getName()).file(podFilePath).upload(inputStream);
		return upload;
	}

	/***
	 * 复制文件从pod
	 * 
	 * @param namespace
	 * @param deploymentName
	 * @param podFilePath
	 * @return
	 */
	public InputStream copyFileFromPod(String namespace, String deploymentName, String podFilePath) {
		Pod pod = getPodByDeployment(namespace, deploymentName);
		InputStream inputStream = this.client.pods().inNamespace(pod.getMetadata().getNamespace())
				.withName(pod.getMetadata().getName()).file(podFilePath).read();
		return inputStream;
	}

	// kubectl exec po/flink-jobmanager-5d84d6b644-nbflx -n default --
	// /opt/flink/bin/sql-client.sh
	// 执行shell命令
	public Process execshell(String namespace, String podname, String... cmdargs) {
		Validate.notBlank(podname, "节点名必填");
		Validate.notEmpty(cmdargs, "命令参数不能为空");
//		final Options options = new Options();
//	    options.addOption(new Option("p", "pod", true, "The name of the pod"));
//	    options.addOption(new Option("n", "namespace", true, "The namespace of the pod"));
//	    CommandLineParser parser = new DefaultParser();
//	    CommandLine cmd = parser.parse(options, cmdOther);
		String[] args = null;
		if (cmdargs[0] != "sh") {// 兼容没有传 sh -c 的情况
			args = CollectionUtil.arrayMerge(String[].class, new String[] { "sh", "-c" }, cmdargs, false);
		} else {
			args = cmdargs;
		}
		Process proc = execPod(namespace, podname, args);
		return proc;
	}

	private Process execPod(String namespace, String podname, String[] args) {
		Exec exec = new Exec(this.apiClient);
		boolean tty = System.console() != null;
		Process proc;
		try {
			proc = exec.exec(StringUtil.hasNull(namespace, "default"), podname, args, true, tty);
		} catch (ApiException e) {
			throw new ProjectExceptionRuntime(ExceptAll.k8s_api_other,
					"执行shell失败：" + CollectionUtil.arrayJoin(args, " "));
		} catch (IOException e) {
			throw new ProjectExceptionRuntime(ExceptAll.k8s_api_other, "执行shell失败,网络或io方面问题");
		}
		return proc;
	}

	/**
	 * 执行flink sql程序
	 * 
	 * @param namespace
	 * @param podname
	 * @return
	 */
	public Process execFlinkSqlClient(String namespace, String podname, String shellcmd) {
		String[] paramsTrue;
		if (StringUtil.isNull(shellcmd)) {
			paramsTrue = new String[] { "sh", "-c", "/opt/flink/bin/sql-client.sh embedded" };
		} else {
			paramsTrue = new String[] { "sh", "-c", shellcmd };
		}
		return execshell(namespace, podname, paramsTrue);
	}

	/***
	 * 执行tams定制的sqlclient
	 * 
	 * @param namespace
	 * @param podname
	 * @param params       初始化文件的参数，如：tenantId:1,operate:2 表示租户为1，用户为2
	 * @param initFilePath 初始化文件地址
	 * @return
	 */
	public Process execFlinkSqlClientTams(String namespace, String podname, String params, String initFilePath) {
		String cmd = String.format("/opt/flink/bin/sql-client.sh embedded  -i   " + initFilePath + " -p %s", params);
		return execFlinkSqlClient(namespace, podname, cmd);
	}

	/***
	 * 登陆pod(没有成功)
	 * 
	 * @param namespace
	 * @param podname
	 * @return
	 */
	public Process execPod(String namespace, String podname) {
//		String[] paramsTrue= new String[] { "sh","-c","/bin/sh"};
		// "sh","-c","/bin/sh"
		String[] paramsTrue = new String[] { "sh", "-s", "/bin/bash" };
		return execPod(namespace, podname, paramsTrue);
	}

	public void execShellToSystemIo(String namespace, String podname, String... cmdargs) {
		Process proc = execshell(namespace, podname, cmdargs);
		execShellToCallback(proc, StreamCopyBean.procToSystem(proc), cmdargs);
	}

	/***
	 * 执行shell命令，并反回结果
	 * 
	 * @param namespace k8s
	 * @param podname   容器名
	 * @param needError 是否需要告警和错误信息
	 * @param cmd       执行的命令
	 * @return
	 */
	public String execShellToResult(String namespace, String podname, boolean needError, String cmd) {
		Process proc = execshell(namespace, podname, cmd);
		try {
			String retstr = IOUtil.slurp(proc.getInputStream());
			return retstr;
		} catch (IOException e) {
			log.error("执行命令失败:" + cmd, e);
			return null;
		}
		// 异步才用下面这种方式
//		StringBuffer buff=new StringBuffer();
//		execShellToCallback(proc, new IProcess() {
//			
//			@Override
//			public void doOutputStream(OutputStream outputStream) {
//			}
//			
//			@Override
//			public void doInputStream(InputStream inputStream) {
//				try {
//					buff.append(IOUtil.slurp(inputStream));
//				} catch (Exception e) {
//					log.error("执行命令失败："+cmd,e);
//				}
//			}
//			
//			@Override
//			public void doErrorStream(InputStream errorStream) {
//				if(needError) {
//					try {
//						buff.append(IOUtil.slurp(errorStream));
//					} catch (Exception e) {
//						log.error("执行命令失败："+cmd,e);
//					}
//				}
//			}
//		}, cmd);
//		return buff.toString();
	}

	// 反回空值即为失败
	public String execShellToResult(String namespace, String podname, String cmd) {
		return execShellToResult(namespace, podname, false, cmd);
	}

	// 直接调某pod的curl本机地址来得到信息
//	public String callRestServiceToResultForExec(Integer port, String namespace, String serviceName, String relaUrl,
//			String... params) {
////		Pod pod = this.getPodByService(namespace, serviceName);
////		String relaUrlTrue = StringUtil.packageUrlParams(relaUrl, params);
////		String retstr = this.execShellToResult(namespace, pod.getMetadata().getName(), "curl "
////				+ IOUtil.mergeFolderAndFilePath("http://localhost:" + (port == null ? 8080 : port), relaUrlTrue));
////		return retstr;
//		return callRestServiceToResultForExec(null,port, namespace, serviceName, relaUrl, params);
//	}

	/// 在指定pod上通过curl调用别的机器上的服务
	private String callRestServiceToResultForExec(Pod pod, Integer port, String namespace, String serviceName,
			String relaUrl, String modth, String... params) {
		Pod podTrue = pod == null ? this.getPodByService(namespace, serviceName) : pod;
		String relaUrlTrue = StringUtil.packageUrlParams(relaUrl, params);
		String serviceUrl = pod == null ? "localhost" : String.format("%s.%s", serviceName, namespace);
		int portTrue = port == null ? 80 : port;

		String modthPathStr = "";
		if ("PATCH".equalsIgnoreCase(modth)) {
			modthPathStr = "--request PATCH";
		}
		// 支持&等转义要加双引号
		String retstr = this.execShellToResult(podTrue.getMetadata().getNamespace(), podTrue.getMetadata().getName(),
				"curl  \"" + modthPathStr
						+ IOUtil.mergeFolderAndFilePath("http://" + serviceUrl + ":" + portTrue, relaUrlTrue) + "\"");
		return retstr;
	}

	public String callRestServiceToResultForExecGet(Pod pod, Integer port, String namespace, String serviceName,
			String relaUrl, String... params) {
		return callRestServiceToResultForExec(pod, port, namespace, serviceName, relaUrl, "GET", params);
	}

	public String callRestServiceToResultForExecPatch(Pod pod, Integer port, String namespace, String serviceName,
			String relaUrl, String... params) {
		return callRestServiceToResultForExec(pod, port, namespace, serviceName, relaUrl, "PATCH", params);
	}

	public void execShellToCallback(Process proc, IProcess callback, String... cmdargs) {

		// BufferedWriter write = new BufferedWriter(new
		// OutputStreamWriter(proc.getOutputStream()));
		// BufferedReader reader = new BufferedReader(new
		// InputStreamReader(proc.getInputStream()));
		Thread in = new Thread(new Runnable() {
			public void run() {
				try {
					callback.doOutputStream(proc.getOutputStream());
				} catch (Exception ex) {
					ex.printStackTrace();
				}
			}
		});
		in.start();

		Thread out = new Thread(new Runnable() {
			public void run() {
				try {
					callback.doInputStream(proc.getInputStream());
				} catch (Exception ex) {
					ex.printStackTrace();
				}
			}
		});
		out.start();

		Thread error = new Thread(new Runnable() {
			public void run() {
				try {
					callback.doErrorStream(proc.getErrorStream());
				} catch (Exception ex) {
					ex.printStackTrace();
				}
			}
		});
		error.start();

		try {
			proc.waitFor();
			// wait for any last output; no need to wait for input thread
			out.join();
			error.join();
		} catch (InterruptedException e) {
			throw new ProjectExceptionRuntime(ExceptAll.k8s_api_other, "启动线程有问题");
		}
		proc.destroy();
	}

	/***
	 * 删除自定义的资源
	 * 
	 * @param crdVesion
	 * @param version   版本
	 * @param namespace
	 * @param name
	 * @return
	 */
	public boolean deleteCusObject(ICrdDef crdVesion, String version, String namespace, String name) {
		try {
			CustomResourceDefinitionContext context = crdVesion.getCrd();
			boolean empty = this.client.genericKubernetesResources(context).inNamespace(namespace).withName(name)
					.delete().isEmpty();
			return !empty;
		} catch (KubernetesClientException e) {
			if (e.getCode() == 404) {
				throw new ProjectExceptionRuntime(ExceptAll.k8s_api_pagenotfound,
						"没有布署应用:" + name + ": " + e.getMessage());
			}
			throw new ProjectExceptionRuntime(ExceptAll.k8s_api_other, "布署自定义的类型失败");
		} catch (Throwable e) {
			if (e.getMessage().contains("already exists")) {
				throw new ProjectExceptionRuntime(ExceptAll.k8s_api_exists, "已存在此类型的同名资源:" + e.getMessage());
			}
			throw new ProjectExceptionRuntime(ExceptAll.k8s_api_other, "布署自定义的类型失败");
		}
	}

	public boolean deleteCusObject(ICrdDef crdVesion, String namespace, String name) {
		return deleteCusObject(crdVesion, null, namespace, name);
	}

	public boolean deleteCusObjectByLabel(ICrdDef crdVesion, String namespace, String... labels) {
		CustomResourceDefinitionContext context = crdVesion.getCrd();
		boolean empty = this.client.genericKubernetesResources(context).inNamespace(namespace)
				.withLabels(CollectionUtil.newMapStr(labels)).delete().isEmpty();
		return !empty;
	}

	/***
	 * 查询用户自定义的资源
	 * 
	 * @param crdVesion
	 * @param version   版本
	 * @param namespace
	 * @param name
	 * @return
	 */
	@SuppressWarnings({ "rawtypes", "unchecked" })
	private <T extends HasMetadata, L extends KubernetesResourceList<HasMetadata>, R extends Resource<T>> List<GenericKubernetesResource> selCusObject(
			ICrdDef crdVesion, String version, String namespace, String name) {
		CustomResourceDefinitionContext context = crdVesion.getCrd();
		List<GenericKubernetesResource> retlist = new ArrayList<GenericKubernetesResource>();
		if (StringUtil.isNotNull(namespace) && StringUtil.isNotNull(name)) {
			GenericKubernetesResource resource = this.client.genericKubernetesResources(context).inNamespace(namespace)
					.withName(name).get();
			if (resource != null) {
				retlist.add(resource);
			}
		} else if (StringUtil.isNotNull(namespace) && StringUtil.isNull(name)) {
			NonNamespaceOperation<T, L, Resource<T>> inNamespace = (NonNamespaceOperation) this.client
					.genericKubernetesResources(context).inNamespace(namespace);
			retlist = (List<GenericKubernetesResource>) queryAll(inNamespace);
		} else if (StringUtil.isNull(namespace) && StringUtil.isNotNull(name)) {
			GenericKubernetesResource resource = this.client.genericKubernetesResources(context).withName(name).get();
			if (resource != null) {
				retlist.add(resource);
			}
		} else if (StringUtil.isNull(namespace) && StringUtil.isNull(name)) {
			retlist = this.client.genericKubernetesResources(context).list().getItems();
		}
		return retlist;
	}

	public List<GenericKubernetesResource> selCusObject(ICrdDef crdVesion, String namespace, String name) {
		return selCusObject(crdVesion, crdVesion.getVersions()[0], namespace, name);
	}

	@SuppressWarnings({ "unchecked", "rawtypes" })
	public List<GenericKubernetesResource> selCusObjectByLabel(ICrdDef crdVesion, String namespace, String... labels) {
		FilterWatchListDeletable<GenericKubernetesResource, GenericKubernetesResourceList, Resource<GenericKubernetesResource>> withLabels = this.client
				.genericKubernetesResources(crdVesion.getCrd()).inNamespace(namespace)
				.withLabels(CollectionUtil.newMapStr(labels));
		List queryAll = queryAll((FilterWatchListDeletable) withLabels);
		return (List<GenericKubernetesResource>) queryAll;
	}

	// 定制，得到flink的crd
//	public static CustomResourceDefinitionContext getFlinkCRD(CrdVesion crdVesion) {
//		CustomResourceDefinitionContext context = new CustomResourceDefinitionContext.Builder()
//				.withGroup(crdVesion.getGroup())
//				// .withKind("Dummy")
//				.withName(crdVesion.getCrdName()).withPlural(crdVesion.getPlural()).withScope(crdVesion.getScope())
//				.withVersion(crdVesion.getVersions()[0]).build();
//		return context;
//	}

	public Service getService(String serviceName) {
		return getService(null, serviceName);
	}

	/***
	 * 通过名字得到RC
	 * 
	 * @param namespace
	 * @param rcName
	 * @return
	 */
	public ReplicationController getRc(String namespace, String rcName) {
		ReplicationController gotRc = this.client.replicationControllers().inNamespace(getUseNamespaceName(namespace))
				.withName(rcName).get();
		return gotRc;
	}

	public ReplicationController getRc(String rcName) {
		return getRc(null, rcName);
	}

	/***
	 * 得到service
	 * 
	 * @param namespace
	 * @param rcName
	 * @return
	 */
	public Service getSvc(String namespace, String rcName) {
		Service service = this.client.services().inNamespace(getUseNamespaceName(namespace)).withName(rcName).get();
		return service;
	}

	public boolean delService(String namespace, String serviceName) {
		boolean optResult = !this.client.services().inNamespace(getUseNamespaceName(namespace)).withName(serviceName)
				.delete().isEmpty();
		return optResult;
	}

	public boolean delService(String serviceName) {
		return delService(null, serviceName);
	}

	public Result createNamespace(String namespace, String group) {
		try {
			Map<String, String> labels = new HashMap<String, String>();
			if (StringUtil.isNotNull(group)) {
				labels.put(GROUP_LABE, group);
			}
			Namespace namespaceRet = this.client.namespaces().resource(new NamespaceBuilder().withNewMetadata()
					.withName(namespace).withLabels(labels).endMetadata().build()).create();
			if (StringUtil.isNotNull(group)) {
				this.groups.add(group);
			}
			return Result.getSuc().setRetObjs(namespaceRet);
		} catch (Exception e) {
			return Result.getError(e.getMessage());
		}
	}

	public boolean exitNamespace(String namespace, String group) {
		NonNamespaceOperation<Namespace, NamespaceList, Resource<Namespace>> namespaces = this.client.namespaces();
		if (StringUtil.isNotNull(group)) {
			List<String> allNamespace = getAllNamespace(group);
			return allNamespace.contains(namespace);
		} else {
			Namespace namespaceRes = namespaces.withName(namespace).get();
			return namespaceRes != null;
		}
	}

	/***
	 * 删除名称空间
	 * 
	 * @param namespace
	 * @param group
	 */
	public void delNamespace(String namespace, String group) {
		if (!exitNamespace(namespace, group)) {
			return;
		}
		this.client.namespaces().withName(namespace).delete().isEmpty();
	}

	public boolean delPodByName(String namespace, String podName) {
		return !this.client.pods().inNamespace(getUseNamespaceName(namespace)).withName(podName).delete().isEmpty();
	}

	public boolean delPodByName(String podName) {
		return delPodByName(null, podName);
	}

	public boolean delPodByLabel(String namespace, String key, String value) {
		return !this.client.pods().inNamespace(getUseNamespaceName(namespace)).withLabel(key, value).delete().isEmpty();
	}

	public boolean delPodByLabel(String key, String value) {
		return delPodByLabel(null, key, value);
	}

	/***
	 * 拿到deployment对应的某个pod
	 * 
	 * @param namespace      名称空间
	 * @param deploymentName
	 * @return
	 */
	public Pod getPodByDeployment(String namespace, String deploymentName, String... status) {
		List<Pod> items = getPodListByDeployment(namespace, deploymentName, status);
		return CollectionUtils.isEmpty(items) ? null : items.get(NumberUtil.random(items.size()));
	}

	/***
	 * 通过pod名拿到pod
	 * 
	 * @param namespace
	 * @param podName
	 * @return
	 */
	public Pod getPodByName(String namespace, String podName) {
		FilterNested<FilterWatchListDeletable<Pod, PodList, PodResource>> withLabels = this.client.pods()
				.inNamespace(namespace).withNewFilter();
		withLabels.withField("metadata.name", podName);
		FilterWatchListDeletable<Pod, PodList, PodResource> endFilter = withLabels.endFilter();
		PodList list = endFilter.list();
		return CollectionUtils.isEmpty(list.getItems()) ? null : list.getItems().get(0);
	}

	/***
	 * 当某个节点出现故障后需要强制迁移某个pod，
	 * 
	 * @param namespace 名称空间
	 * @param podName   pod名
	 */
	public void movePod(String namespace, String podName) {
		Pod podByName = getPodByName(namespace, podName);
		if (podByName == null) {
			return;
		}
		this.client.resource(podByName).withPropagationPolicy(DeletionPropagation.FOREGROUND).withGracePeriod(0)
				.delete();
		// List<StatusDetails> delete2 =
		// this.client.pods().withName(podName).withPropagationPolicy(DeletionPropagation.FOREGROUND).withGracePeriod(0).delete();
		List<Volume> volumes = podByName.getSpec().getVolumes();
		for (Volume volume : volumes) {
			PersistentVolumeClaimVolumeSource persistentVolumeClaim = volume.getPersistentVolumeClaim();
			if (persistentVolumeClaim != null) {
				String claimName = persistentVolumeClaim.getClaimName();
				if (StringUtils.isNoneBlank(claimName)) {
					this.client.persistentVolumeClaims().withName(claimName)
							.withPropagationPolicy(DeletionPropagation.FOREGROUND).withGracePeriod(0).delete();
				}
			}
		}
	}

	/**
	 * 查询命名空间配额
	 *
	 * @param namespace
	 */
	public QuotaBean queryResourcequota(String namespace, String quotasName) {
		ResourceQuota resourceQuota = this.client.resourceQuotas().inNamespace(namespace).withName(quotasName).get();
		if (resourceQuota == null) {
			return null;
		}
		QuotaBean ret = new QuotaBean();
		Map<String, Quantity> hard = resourceQuota.getStatus().getHard();
		Map<String, Quantity> used = resourceQuota.getStatus().getUsed();
		// CPU
		ret.setCpu(cpuInfo(hard.get("cpu")));
		ret.setCpuUsed(cpuInfo(used.get("cpu")));
		ret.setCpuFree(ret.getCpu() - ret.getCpuUsed());
		// 内存，单位为M
		ret.setMemory(memInfo(hard.get("memory")));
		ret.setMemoryUsed(memInfo(used.get("memory")));
		ret.setMemoryFree(ret.getMemory() - ret.getMemoryUsed());
		// pod
		ret.setPods(numInfo(hard.get("pods")));
		ret.setPodsUsed(numInfo(used.get("pods")));
		ret.setPodsFree(ret.getPods() - ret.getPodsUsed());
		// pvc
		ret.setPvcs(numInfo(hard.get("persistentvolumeclaims")));
		ret.setPvcsUsed(numInfo(used.get("persistentvolumeclaims")));
		ret.setPvcsFree(ret.getPvcs() - ret.getPvcsUsed());
		// 磁盘容量,单位为G
		ret.setStorage(memInfo(hard.get("requests.storage")) / 1024);
		ret.setStorageUsed(memInfo(used.get("requests.storage")) / 1024);
		ret.setStorageFree(ret.getStorage() - ret.getStorageUsed());
		return ret;
	}

	public List<Pod> getPodListByDeployment(String namespace, String deploymentName, String... status) {
		// 通过 deploymentName、namespace 获取matchLabels
		Deployment deployment = this.client.apps().deployments().inNamespace(namespace).withName(deploymentName).get();
		if (deployment == null) {
			return null;
		}
		Map<String, String> matchLabels = deployment.getSpec().getTemplate().getMetadata().getLabels();

		String[] podAry = ArrayUtils.isEmpty(status) ? new String[] { "Running" } : status;

		// 获取PodList
		List<Pod> items = this.client.pods().inNamespace(namespace).withLabels(matchLabels).list().getItems();
		CollectionUtils.filter(items, new Predicate() {
			@Override
			public boolean evaluate(Object object) {
				Pod pod = (Pod) object;
				return ArrayUtils.contains(podAry, pod.getStatus().getPhase());
			}
		});
		return items;
	}

	public Pod getPodByJob(String namespace, String jobName, String... status) {
		List<Pod> items = getPodListByJob(namespace, jobName, status);
		return CollectionUtils.isEmpty(items) ? null : items.get(NumberUtil.random(items.size()));
	}

	public List<Pod> getPodListByJob(String namespace, String jobName, String... status) {
		// 通过 deploymentName、namespace 获取matchLabels
		Job job = this.client.batch().v1().jobs().inNamespace(namespace).withName(jobName).get();
		if (job == null) {
			return null;
		}
		Map<String, String> matchLabels = job.getSpec().getTemplate().getMetadata().getLabels();
		String[] podAry = ArrayUtils.isEmpty(status) ? new String[] { "Running" } : status;
		// 获取PodList
		List<Pod> items = this.client.pods().inNamespace(namespace).withLabels(matchLabels).list().getItems();
		CollectionUtils.filter(items, new Predicate() {
			@Override
			public boolean evaluate(Object object) {
				Pod pod = (Pod) object;
				return ArrayUtils.contains(podAry, pod.getStatus().getPhase());
			}
		});
		return items;
	}

	/**
	 * 通过service拿到pod
	 * 
	 * @param namespace
	 * @param serviceName
	 * @return
	 */
	public Pod getPodByService(String namespace, String serviceName) {
		List<Pod> items = getPodListByService(namespace, serviceName);
		return CollectionUtils.isEmpty(items) ? null : items.get(NumberUtil.random(items.size()));
	}

	public List<Pod> getPodListByService(String namespace, String serviceName) {
		// 通过 deploymentName、namespace 获取matchLabels
		Map<String, String> matchLabels = this.client.services().inNamespace(namespace).withName(serviceName).get()
				.getSpec().getSelector();
		// 获取PodList
		List<Pod> items = this.client.pods().inNamespace(namespace).withLabels(matchLabels).list().getItems();
		CollectionUtils.filter(items, new Predicate() {
			@Override
			public boolean evaluate(Object object) {
				Pod pod = (Pod) object;
				return "Running".equals(pod.getStatus().getPhase());
			}
		});
		return items;
	}

	/****
	 * 查询pod，跟据过滤条件
	 * 
	 * @param namespace
	 * @param filtermap
	 * @param lable
	 * @param ids
	 * @return
	 */
	public List<Pod> getPodListByIds(String namespace, Map<String, String> filtermap, String lable, String... ids) {
		FilterNested<FilterWatchListDeletable<Pod, PodList, PodResource>> withLabels = this.client.pods()
				.inNamespace(namespace).withNewFilter();
		if (MapUtils.isNotEmpty(filtermap)) {
			withLabels = withLabels.withLabels(filtermap);
			if (ArrayUtils.isNotEmpty(ids)) {
				withLabels = withLabels.withLabelIn(lable, ids);
			}
		} else {
			if (ArrayUtils.isNotEmpty(ids)) {
				withLabels = withLabels.withLabelIn(lable, ids);
			}
		}
		FilterWatchListDeletable<Pod, PodList, PodResource> endFilter = withLabels.endFilter();
		PodList list = endFilter.list();
		return list.getItems();
	}

	/***
	 * 
	 * @param namespace
	 * @param filtermap
	 * @param lable
	 * @param ids
	 * @return
	 */
	public List<Job> getJobListByIds(String namespace, Map<String, String> filtermap, String lable, String... ids) {
		NonNamespaceOperation<Job, JobList, ScalableResource<Job>> withLabels = this.client.batch().v1().jobs()
				.inNamespace(namespace);
		FilterWatchListDeletable<Job, JobList, ScalableResource<Job>> queryListWatch = null;
		if (MapUtils.isNotEmpty(filtermap)) {
			queryListWatch = withLabels.withLabels(filtermap);
			if (ArrayUtils.isNotEmpty(ids)) {
				queryListWatch = queryListWatch.withLabelIn(lable, ids);
			}
		} else {
			if (ArrayUtils.isNotEmpty(ids)) {
				queryListWatch = withLabels.withLabelIn(lable, ids);
			}
		}
		return queryListWatch == null ? new ArrayList<Job>() : queryListWatch.list().getItems();
	}

	public List<Pair<String, PodPhase>> queryPodStatus(String namespace, Map<String, String> filtermap, String lable,
			String[] lableNames) {
//		PodList podList = this.client.pods().withLabels(labels).withLabel("app.kubernetes.io/instance", "rjzjh").withLabelIn("appname", ids).list();
		List<Pod> pods = getPodListByIds(namespace, filtermap, lable, lableNames);
		List<Pair<String, PodPhase>> retlist = new ArrayList<Pair<String, PodPhase>>();
		for (String lableName : lableNames) {
			PodPhase elePhase = PodPhase.noExit;
			for (Pod pod : pods) {
				String podname = pod.getMetadata().getName();
				if (podname.startsWith(lableName + "-")) {
					String[] groups = podname.split("-");
					String[] subgroups = ArrayUtils.subarray(groups, 0, groups.length - 2);
					// "duckula-init-agent-6c47b7bfc4-p2l9z"变为"duckula-init-agent-"
					// t-dtdt-saasproduct-06-d64bbfb68-t25dx
					// String podnamesub = podname.substring(0, podname.length()-16);
					if (CollectionUtil.arrayJoin(subgroups, "-").equals(lableName)) {// 防止
																						// "mall-rebate-tag"和"mall-rebate"类似的问题导致的判断不准确
						elePhase = PodPhase.valueOf(pod.getStatus().getPhase());
						break;
					}
				}
			}
			retlist.add(Pair.of(lableName, elePhase));
		}
		return retlist;
	}

	public List<Pair<String, PodPhase>> queryJobStatus(String namespace, Map<String, String> filtermap, String lable,
			String[] lableNames) {
//		PodList podList = this.client.pods().withLabels(labels).withLabel("app.kubernetes.io/instance", "rjzjh").withLabelIn("appname", ids).list();
		List<Job> jobList = getJobListByIds(namespace, filtermap, lable, lableNames);
		List<Pair<String, PodPhase>> retlist = new ArrayList<Pair<String, PodPhase>>();
		for (String lableName : lableNames) {
			PodPhase elePhase = PodPhase.noExit;
			for (Job job : jobList) {
				if (job.getMetadata().getName().equals(lableName)) {
					if (job.getStatus().getActive() != null && job.getStatus().getActive() == 1) {// 因为只有一个pod，所以写死为1
						elePhase = PodPhase.Running;
					} else if (job.getStatus().getSucceeded() != null && job.getStatus().getSucceeded() == 1) {// 因为只有一个pod，所以写死为1
						elePhase = PodPhase.Succeeded;
					} else {
						elePhase = PodPhase.Unknown;
					}
					break;
				}
			}
			retlist.add(Pair.of(lableName, elePhase));
		}
		return retlist;
	}

	/***
	 * 通过label得到pod
	 * 
	 * @param namespace
	 * @param labelmap
	 * @return
	 */
	public List<String> selPodNameByLabel(String namespace, Map<String, String> labelmap) {
		List<Pod> podList = getPodListByIds(namespace, labelmap, null);
		List<String> retlist = new ArrayList<String>();
		for (Pod pod : podList) {
			retlist.add(pod.getMetadata().getName());
		}
		return retlist;
	}

	/***
	 * 通过label得到service
	 * 
	 * @param namespace
	 * @param labelmap
	 * @return
	 */
	public List<String> selServiceByLabel(String namespace, Map<String, String> labelmap) {
		NonNamespaceOperation<Service, ServiceList, ServiceResource<Service>> serviceSel = this.client.services()
				.inNamespace(getUseNamespaceName(namespace));
		FilterWatchListDeletable<Service, ServiceList, ServiceResource<Service>> withLabel = null;
		for (String key : labelmap.keySet()) {
			if (withLabel == null) {
				withLabel = serviceSel.withLabel(key, labelmap.get(key));
			} else {
				withLabel = withLabel.withLabel(key, labelmap.get(key));
			}
		}
		ServiceList list = withLabel.list();
		List<String> retlist = new ArrayList<String>();
		for (Service service : list.getItems()) {
			retlist.add(service.getMetadata().getName());
		}
		return retlist;
	}

	public void createRc(String namespace, String filePath) throws ProjectException {
		FileInputStream inputstream;
		try {
			inputstream = new FileInputStream(filePath);
		} catch (FileNotFoundException e) {
			throw new ProjectException(ExceptAll.Project_default, "文件不存在");
		}
		createRc(namespace, inputstream);
	}

	/***
	 * 跟据文件创建RC TODO 测试
	 * 
	 * @param fileInputStream 输入流
	 * @throws ProjectException
	 */
	@SuppressWarnings("unchecked")
	public void createRc(String namespace, FileInputStream fileInputStream) throws ProjectException {
		ObjectMapper objmap = new ObjectMapper();
		ObjectNode jsonObject = YamlAssist.readYaml(fileInputStream);
		ReplicationController rc = new ReplicationController();
		if (jsonObject.has("metadata")) {
			ObjectNode metadataJson = (ObjectNode) jsonObject.get("metadata");
			if (!metadataJson.has("name")) {
				throw new ProjectException(ExceptAll.param_error, "metadata需要有name元素");
			}
			ObjectMeta metadata = new ObjectMeta();
			metadata.setName(metadataJson.get("name").asText());
			if (metadataJson.has("labels")) {
				ObjectNode labels = (ObjectNode) metadataJson.get("labels");
				Map<String, String> lables = new HashMap<String, String>();
				for (Iterator<String> iterator = labels.fieldNames(); iterator.hasNext();) {
					String key = iterator.next();
					lables.put(key, labels.get(key).asText());
				}
				metadata.setLabels(lables);
			}
			rc.setMetadata(metadata);
		}
		ObjectNode specJson = (ObjectNode) jsonObject.get("spec");
		ReplicationControllerSpec spec = rc.getSpec();
		if (specJson.has("replicas")) {
			spec.setReplicas(specJson.get("replicas").asInt());
		}
		if (specJson.has("selector")) {
			Map<String, String> selector;
			try {
				selector = objmap.readValue(objmap.writeValueAsString(specJson.get("selector")), Map.class);
			} catch (Exception e) {
				throw new ProjectExceptionRuntime(ExceptAll.param_error, e);
			}
			spec.setSelector(selector);
		}

		PodTemplateSpec newTemplate = spec.getTemplate();
		ObjectNode templateJson = (ObjectNode) specJson.get("template");
		if (templateJson.has("metadata")) {
			ObjectMeta withNewMetadata = newTemplate.getMetadata();
			ObjectNode jsonObject2 = (ObjectNode) templateJson.get("metadata").get("labels");
			Map<String, String> labels;
			try {
				labels = objmap.readValue(objmap.writeValueAsString(jsonObject2), Map.class);
			} catch (Exception e) {
				throw new ProjectExceptionRuntime(ExceptAll.param_error, e);
			}
			withNewMetadata.setLabels(labels);
		}
		// PodSpec innerSpec = newTemplate.getSpec();
		ObjectNode innerSpecJson = (ObjectNode) templateJson.get("spec");
		ArrayNode containersJson = (ArrayNode) innerSpecJson.get("containers");
		for (int i = 0; i < containersJson.size(); i++) {
			ObjectNode containerJson = (ObjectNode) containersJson.get(i);
			Container addNewContainer = new Container();
			addNewContainer.setName(containerJson.get("name").asText());
			addNewContainer.setImage(containerJson.get("image").asText());
			ArrayNode portAry = (ArrayNode) containerJson.get("ports");
			List<ContainerPort> ports = new ArrayList<ContainerPort>();
			for (int j = 0; j < portAry.size(); j++) {
				ContainerPort temport = new ContainerPort();
				temport.setContainerPort(portAry.get(i).get("containerPort").asInt());
				ports.add(temport);
			}
			addNewContainer.setPorts(ports);
		}
		rc = this.client.replicationControllers().inNamespace(getUseNamespaceName(namespace)).resource(rc).create();
	}

	/***
	 * 创建Service TODO 测试
	 * 
	 * @param namespace
	 * @param fileInputStream
	 * @throws ProjectException
	 */
	@SuppressWarnings("unchecked")
	public void createSvc(String namespace, FileInputStream fileInputStream) throws ProjectException {
		ObjectNode jsonObject = YamlAssist.readYaml(fileInputStream);
		Service createNew = new ServiceBuilder().build();
		ObjectNode metadata = (ObjectNode) jsonObject.get("metadata");

		ObjectMeta meta = new ObjectMeta();
		meta.setName(metadata.get("name").asText());
		createNew.setMetadata(meta);
		ObjectNode specjson = (ObjectNode) jsonObject.get("spec");
		ServiceSpec withNewSpec = createNew.getSpec();
		if (specjson.has("type")) {
			withNewSpec.setType(specjson.get("type").asText());
		}

		ObjectMapper objmap = new ObjectMapper();

		if (specjson.has("selector")) {

			Map<String, String> seleMap;
			try {
				seleMap = objmap.readValue(objmap.writeValueAsString(specjson.get("selector")), Map.class);
			} catch (Exception e) {
				throw new ProjectExceptionRuntime(ExceptAll.param_error, e);
			}

			withNewSpec.setSelector(seleMap);
		}
		if (specjson.has("ports")) {
			ArrayNode array = (ArrayNode) specjson.get("ports");
			List<ServicePort> ports = new ArrayList<>();
			for (int i = 0; i < array.size(); i++) {
				ObjectNode tempobj = (ObjectNode) array.get(i);
				ServicePort temp = new ServicePort();
				if (tempobj.has("name")) {
					temp.setName(tempobj.get("name").asText());
				}
				if (tempobj.has("nodePort")) {
					temp.setNodePort(tempobj.get("nodePort").asInt());
				}
				if (tempobj.has("port")) {
					temp.setPort(tempobj.get("port").asInt());
				}
				if (tempobj.has("protocol")) {
					temp.setProtocol(tempobj.get("protocol").asText());
				}
				if (tempobj.has("targetPort")) {
					IntOrString input = new IntOrString(tempobj.get("targetPort").asInt());
					temp.setTargetPort(input);
				}
				ports.add(temp);
			}
			withNewSpec.setPorts(ports);
		}
		Service service = new ServiceBuilder().withSpec(withNewSpec).build();
		service = this.client.services().inNamespace(getUseNamespaceName(namespace)).resource(service).create();
	}

	public void createSvc(String namespace, String filePath) throws ProjectException {
		FileInputStream inputstream;
		try {
			inputstream = new FileInputStream(filePath);
		} catch (FileNotFoundException e) {
			throw new ProjectException(ExceptAll.Project_default, "文件不存在");
		}
		createSvc(namespace, inputstream);
	}

	/***
	 * 创建命名空间配额
	 * 
	 * @param namespace
	 * @param minCpu
	 * @param minMemory
	 * @param maxCpu
	 * @param maxMemory
	 * @param pods
	 */
	public void createResourcequota(String namespace, int minCpu, int minMemory, int maxCpu, int maxMemory, int pods) {
		Map<String, Quantity> parma = new HashMap<>();
		parma.put("pods", new Quantity(String.valueOf(pods)));
		parma.put("requests.cpu", new Quantity(String.valueOf(minCpu)));
		parma.put("requests.memory", new Quantity(String.valueOf(minMemory)));
		parma.put("limits.cpu", new Quantity(String.valueOf(maxCpu)));
		parma.put("limits.memory", new Quantity(String.valueOf(maxMemory)));
		ResourceQuota quota = new ResourceQuotaBuilder().withNewMetadata().withName(getUseNamespaceName(namespace))
				.endMetadata().withNewSpec().addToHard(parma).endSpec().build();
		this.client.resourceQuotas().inNamespace(getUseNamespaceName(namespace)).resource(quota).create();
	}

	/**
	 * 删除命名空间配额
	 * 
	 * @param namespace
	 */
	public void delResourcequota(String namespace) {
		this.client.resourceQuotas().inNamespace(getUseNamespaceName(namespace)).delete();
	}

	public boolean delRc(String namespace, String rcName) {
		return !this.client.replicationControllers().inNamespace(getUseNamespaceName(namespace)).withName(rcName)
				.delete().isEmpty();
	}

	/***
	 * 删除svc
	 * 
	 * 
	 * @param namespace
	 * @param svcName
	 * @return
	 */
	public boolean delSvc(String namespace, String svcName) {
		return !this.client.services().inNamespace(getUseNamespaceName(namespace)).withField("metadata.name", svcName)
				.delete().isEmpty();
	}

	/***
	 * 更新容器的image TODO 测试
	 * 
	 * @param namespace
	 * @param rcName
	 * @param containerIndex 这个pod的第几个容器
	 * @param imageValue
	 */
	public void updateRcImage(String namespace, String rcName, int containerIndex, String imageValue) {
		ReplicationControllerSpec spec = this.client.replicationControllers()
				.inNamespace(getUseNamespaceName(namespace)).withName(rcName).edit().getSpec();
		Container container = spec.getTemplate().getSpec().getContainers().get(containerIndex);
		container.setImage(imageValue);
		RollableScalableResource<ReplicationController> withName = getClient().replicationControllers()
				.inNamespace((getUseNamespaceName(namespace))).withName(rcName);
		withName.updateImage(imageValue);
		withName.rolling();
//		getClient().replicationControllers().inNamespace((getUseNamespaceName(namespace))).withName(rcName) .rolling()
//				.updateImage(imageValue);
	}

	/**
	 * 更新副本
	 * 
	 * @param namespace
	 * @param deploymentName
	 * @param numberOfReplicas
	 * @throws ApiException
	 */
	public void scaleDeployment(String namespace, String deploymentName, int numberOfReplicas) throws ApiException {
		AppsV1Api appsV1Api = new AppsV1Api(this.apiClient);
		// 获取V1DeploymentList
		APIlistNamespacedDeploymentRequest listNamespacedDeployment = appsV1Api
				.listNamespacedDeployment(getUseNamespaceName(namespace));
		List<V1Deployment> appsV1DeploymentItems = listNamespacedDeployment.execute().getItems();
		Optional<V1Deployment> findedDeployment = appsV1DeploymentItems.stream()
				.filter((V1Deployment deployment) -> deployment.getMetadata().getName().equals(deploymentName))
				.findFirst();

		findedDeployment.ifPresent((V1Deployment deploy) -> {
			try {
				V1DeploymentSpec newSpec = deploy.getSpec().replicas(numberOfReplicas);
				V1Deployment newDeploy = deploy.spec(newSpec);
				APIreplaceNamespacedDeploymentRequest replaceNamespacedDeployment = appsV1Api
						.replaceNamespacedDeployment(deploymentName, getUseNamespaceName(namespace), newDeploy);
				replaceNamespacedDeployment.execute();
			} catch (ApiException ex) {
				log.warn("Scale the pod failed for Deployment:" + deploymentName, ex);
			}
		});
	}

	// protected abstract Object setImage(String imageValue);

	/***
	 * 滚动更新image TODO 测试
	 * 
	 * @param namespace
	 * @param rcName
	 * @param imageValue
	 * @return
	 */
	public Result updateImageRolling(String namespace, String rcName, String imageValue) {
		try {
			RollableScalableResource<ReplicationController> withName = this.client.replicationControllers()
					.inNamespace(namespace).withName(rcName);
			withName.updateImage(imageValue);
			withName.rolling();
			return Result.getSuc();
		} catch (Exception e) {
			return Result.getError(e.getMessage());
		}
	}

	/***
	 * 创建或更新RC
	 * 
	 * @param namespace  命名空间
	 * @param imageValue image值
	 * @param rcPath     rc的文件路径
	 * @return 创建或更新成功
	 */
	public Result createOrUpdateImageRolling(String namespace, String imageValue, String rcPath) {
		ObjectNode json = YamlAssist.readYaml(rcPath);
		String rcName = json.get("metadata").get("name").asText();
		ReplicationController queryRc = getRc(namespace, rcName);
		if (queryRc == null) {
			try {
				createRc(namespace, rcPath);
				return Result.getSuc();
			} catch (ProjectException e) {
				return Result.getError(e.getMessage());
			}
		} else {
			return updateImageRolling(namespace, rcName, imageValue);
		}
	}

	/***
	 * 得到deploymentName
	 */
	public Deployment getDeploymentByName(String namespace, String name) {
		Deployment deployment = this.client.apps().deployments().inNamespace(StringUtil.hasNull(namespace, "default"))
				.withName(name).get();
		return deployment;
	}

	/***
	 * 通过label查找惟一的deployment
	 * 
	 * @param namespace
	 * @param label
	 * @param value
	 * @return
	 */
	public Deployment getDeploymentByLabelOne(String namespace, String label, String value) {
		DeploymentList list = getDeploymentByLabel(namespace, label, value);
		if (list == null || list.getItems().size() != 1) {
			throw new ProjectExceptionRuntime(ExceptAll.param_notfit,
					"Deployment预期只有一个，现已查出:" + list.getItems().size());
		}
		return list.getItems().get(0);
	}

	public DeploymentList getDeploymentByLabel(String namespace, String label, String value) {
		DeploymentList deployments = this.client.apps().deployments()
				.inNamespace(StringUtil.hasNull(namespace, "default")).withLabel(label, value).list();
		return deployments;
	}

	public DeploymentList getDeploymentByLabel(String namespace, String label) {
		DeploymentList deployments = this.client.apps().deployments()
				.inNamespace(StringUtil.hasNull(namespace, "default")).withLabel(label).list();
		return deployments;
	}

	/***
	 * 通过freemark模板和参数部署deployment
	 * 
	 * @param namespace
	 * @param context
	 * @param params
	 * @return
	 */
	public V1Deployment installDeploymentByTemp(String namespace, String context, Map<String, Object> params) {
		try {
			String result = FreemarkUtil.getInst().doProcessByTemp(context, params);
			V1Deployment yamlSvc = (V1Deployment) Yaml.load(result);
			AppsV1Api appsV1Api = new AppsV1Api(this.apiClient);
			V1Deployment v1Deployment = appsV1Api.createNamespacedDeployment(namespace, yamlSvc).execute();
			return v1Deployment;
		} catch (ApiException e) {
			if ("Conflict".equals(e.getMessage())) {
				throw new ProjectExceptionRuntime(ExceptAll.k8s_deploy_conflict);
			} else {
				throw new ProjectExceptionRuntime(ExceptAll.k8s_api_other, e.getMessage());
			}
		} catch (Exception e) {
			throw new ProjectExceptionRuntime(ExceptAll.k8s_deploy_excetion, "部署task失败", e);
		}
	}

	// 安装资源
	public void installResource(ResourcesType resouceType, String namespace, Map<String, Object> crd) {
		if (resouceType == ResourcesType.UNKNOWN || resouceType == ResourcesType.CRD) {
			return;
		}
		HasMetadata po = null;
		switch (resouceType) {
		case DEPLOYMENT:
			po = new Deployment();
			po = resouceType.setColValue(po, crd, po.getClass(), true);
			this.client.apps().deployments().inNamespace(namespace).resource((Deployment) po).create();
			break;
		case SERVICE:
			po = new Service();
			po = resouceType.setColValue(po, crd, po.getClass(), true);
			this.client.services().inNamespace(namespace).resource((Service) po).create();
			break;

		default:
			break;
		}
	}

	// 安装资源
	public void updateResource(ResourcesType resouceType, String namespace, String name, Map<String, Object> crd) {
		if (resouceType == ResourcesType.UNKNOWN || resouceType == ResourcesType.CRD) {
			return;
		}
		switch (resouceType) {
		case DEPLOYMENT:
			Deployment deployment = this.client.apps().deployments().inNamespace(namespace).withName(name).get();
			deployment = resouceType.setColValue(deployment, crd, deployment.getClass(), false);
			this.client.apps().deployments().inNamespace(namespace).resource(deployment).update();
			break;
		case SERVICE:
			Service service = this.client.services().inNamespace(namespace).withName(name).get();
			service = resouceType.setColValue(service, crd, service.getClass(), false);
			this.client.services().inNamespace(namespace).resource(service).update();
			break;

		default:
			break;
		}
	}

	// 删除资源
	public void deleteResource(ResourcesType resouceType, String namespace, String name) {
		if (resouceType == ResourcesType.UNKNOWN || resouceType == ResourcesType.CRD) {
			return;
		}
		switch (resouceType) {
		case DEPLOYMENT:
			this.client.apps().deployments().inNamespace(namespace).withName(name).delete();
			break;
		case SERVICE:
			this.client.services().inNamespace(namespace).withName(name).delete();
			break;
		case POD:
			this.client.pods().inNamespace(namespace).withName(name).delete();
			break;
		default:
			break;
		}
	}

	/***
	 * 得到资源的owner
	 * 
	 * @param md 资源,如pod等
	 * @return null表示没有owner
	 */
	public OwnerReference getControllerOf(HasMetadata md) {
		List<OwnerReference> ownerReferences = md.getMetadata().getOwnerReferences();
		for (OwnerReference ownerReference : ownerReferences) {
			if (ownerReference.getController().equals(Boolean.TRUE)) {
				return ownerReference;
			}
		}
		return null;
	}

	///////////////////////////////////////////////////////////////////////// 为了创建config需要添加的方法////////////////////////////////////////////////////////////////
	/**
	 * 查找并创建SA，创建sa名称saName，绑定到集群角色cluster-admin
	 * 
	 * @param namespace
	 * @param saName
	 * @return
	 */
	public ServiceAccount findCreateSa(String namespace, String saName) {
		ServiceAccount serviceAccount = this.client.serviceAccounts().inNamespace(namespace).withName(saName).get();
		if (serviceAccount == null) {
			ServiceAccount sa = new ServiceAccount();
			ObjectMeta objectMeta = new ObjectMeta();
			objectMeta.setName(saName);
			sa.setMetadata(objectMeta);
			serviceAccount = this.client.serviceAccounts().inNamespace(namespace).resource(sa).create();// .create(sa);
		}
		return serviceAccount;
	}

	// 把同名称空间下的帐号赋于管理员角色
	/***
	 * 使用Secret类型是kubernetes.io/service-account-token，用于被serviceaccount引用。serviceaccout
	 * 创建时 Kubernetes 会默认创建对应的 secret，即使删除也会重建(适合于1.23及以下)。 查看：kubectl get
	 * RoleBinding
	 * 
	 * @param namespace
	 * @param bindname
	 * @param sas
	 * @return
	 */
	public Result bindAdminRoleForSa(String namespace, String bindname, String... sas) {
		RbacAuthorizationV1Api rolebind = new RbacAuthorizationV1Api(this.apiClient);
		V1RoleBinding body = new V1RoleBinding();
		body.setKind("RoleBinding");
		body.setApiVersion("rbac.authorization.k8s.io/v1");
		V1ObjectMeta metadata = new V1ObjectMeta();
		metadata.setName(bindname);
		metadata.setNamespace(namespace);
		body.setMetadata(metadata);
		V1RoleRef roleRef = new V1RoleRef();
		roleRef.setApiGroup("rbac.authorization.k8s.io");
		roleRef.setKind("ClusterRole");
		roleRef.setName("admin");
		body.setRoleRef(roleRef);
		// List<RbacV1Subject> subjects = new ArrayList<RbacV1Subject>();
		List<V1Subject> subjects = new ArrayList<V1Subject>();
		for (String sa : sas) {
			// RbacV1Subject temp = new RbacV1Subject();
			V1Subject temp = new V1Subject();
			temp.setKind("ServiceAccount");
			temp.setName(sa);
			temp.setNamespace(namespace);
			subjects.add(temp);
		}
		body.setSubjects(subjects);
		try {
			rolebind.createNamespacedRoleBinding(namespace, body).execute();
			return Result.getSuc();
		} catch (ApiException e) {
//			{
//				  "kind": "Status",
//				  "apiVersion": "v1",
//				  "metadata": {
//				    
//				  },
//				  "status": "Failure",
//				  "message": "rolebindings.rbac.authorization.k8s.io \"common-admin-binder\" already exists",
//				  "reason": "AlreadyExists",
//				  "details": {
//				    "name": "common-admin-binder",
//				    "group": "rbac.authorization.k8s.io",
//				    "kind": "rolebindings"
//				  },
//				  "code": 409
//				}
			if (e.getCode() == 409) {
				return Result.getSuc();
			} else {
				log.error("创建角色绑定关系失败:" + e.getResponseBody(), e);
				return Result.getError("创建角色绑定关系失败" + e.getMessage());
			}
		}
	}

	// 配置集群权限
	public Result bindClusterAdminRoleForSa(String namespace, String bindname, String... sas) {
		RbacAuthorizationV1Api rolebind = new RbacAuthorizationV1Api(this.apiClient);
		V1ClusterRoleBinding body = new V1ClusterRoleBinding();
		body.setKind("ClusterRoleBinding");
		body.setApiVersion("rbac.authorization.k8s.io/v1");
		V1ObjectMeta metadata = new V1ObjectMeta();
		metadata.setName(bindname);
		metadata.setNamespace(namespace);
		body.setMetadata(metadata);
		V1RoleRef roleRef = new V1RoleRef();
		roleRef.setApiGroup("rbac.authorization.k8s.io");
		roleRef.setKind("ClusterRole");
		roleRef.setName("cluster-admin");
		body.setRoleRef(roleRef);
//		List<RbacV1Subject> subjects = new ArrayList<RbacV1Subject>();
		List<V1Subject> subjects = new ArrayList<V1Subject>();
		for (String sa : sas) {
			V1Subject temp = new V1Subject();
			temp.setKind("ServiceAccount");
			temp.setName(sa);
			temp.setNamespace(namespace);
			subjects.add(temp);
		}
		body.setSubjects(subjects);
		try {
			rolebind.createClusterRoleBinding(body).execute();
			return Result.getSuc();
		} catch (ApiException e) {
//			{
//				  "kind": "Status",
//				  "apiVersion": "v1",
//				  "metadata": {
//				    
//				  },
//				  "status": "Failure",
//				  "message": "rolebindings.rbac.authorization.k8s.io \"common-admin-binder\" already exists",
//				  "reason": "AlreadyExists",
//				  "details": {
//				    "name": "common-admin-binder",
//				    "group": "rbac.authorization.k8s.io",
//				    "kind": "rolebindings"
//				  },
//				  "code": 409
//				}
			if (e.getCode() == 409) {
				return Result.getSuc();
			} else {
				log.error("创建角色绑定关系失败:" + e.getResponseBody(), e);
				return Result.getError("创建角色绑定关系失败" + e.getMessage());
			}
		}
	}

	/***
	 * 通过它可以拿到 data.ca.crt 和 data.token(base加密后的密文),通过 kubectl describe得到的token才是原文.
	 * 
	 * pod里反射的卷：kubectl exec -it demo -- cat
	 * /run/secrets/kubernetes.io/serviceaccount/token
	 * 
	 * 1.20（含 1.20）之前的版本，在创建 sa 时会自动创建一个 secret，然后这个会把这个 secret 通过投射卷挂载到 pod 里，该
	 * secret 里面包含的 token 是永久有效的。
	 * 
	 * 1.21~1.23 版本，在创建 sa 时也会自动创建 secret，但是在 pod 里并不会使用 secret 里的 token，而是由 kubelet
	 * 到 TokenRequest API 去申请一个 token，该 token 默认有效期为一年，但是 pod 每一个小时会更新一次 token。
	 * 
	 * 1.24 版本及以上，在创建 sa 时不再自动创建 secret 了，只保留由 kubelet 到 TokenRequest API 去申请 token。
	 * 
	 * @param namespace
	 * @param accountName
	 * @return
	 */
	public Secret getSecretAccount(String namespace, String accountName) {
		SecretList list = this.client.secrets().inNamespace(namespace).list();
		List<Secret> items = list.getItems();
		CollectionUtils.filter(items, new Predicate() {
			@Override
			public boolean evaluate(Object object) {
				Secret temp = (Secret) object;
				return MapUtils.isNotEmpty(temp.getMetadata().getAnnotations()) && accountName
						.equals(temp.getMetadata().getAnnotations().get("kubernetes.io/service-account.name"));
			}
		});
		Secret retObj = null;
		if (CollectionUtils.isEmpty(items)) {// 1.24以上就没有该secret了
			retObj = createAccountSecret(namespace, accountName);
		} else {
			retObj = items.get(0);
		}
		return retObj;
	}

	/***
	 * 通过帐号创建配置文件
	 * 
	 * @param namespace
	 * @param accountName
	 * @param isCluster   true:是集群管理员，false:名称空间管理员
	 * @return RetObjs[0]为config的内容
	 */
	public Result proKubeConfig(String clusterName, String namespace, String accountName, boolean isCluster) {
		ServiceAccount account = this.findCreateSa(namespace, accountName);// 找到或创建帐号
		// 集群资源，注意不要与其它名称空间注册的资源冲突。
		String bindName = isCluster
				? String.format("%s-%s-clusteradmin-binder", account.getMetadata().getName(), namespace)
				: String.format("%s-%s-admin-binder", account.getMetadata().getName(), namespace);// 按规则取一个绑定名
		Result result = isCluster ? this.bindClusterAdminRoleForSa(namespace, bindName, account.getMetadata().getName())
				: this.bindAdminRoleForSa(namespace, bindName, account.getMetadata().getName());
		if (!result.isSuc()) {
			return result;
		}
		Secret accountSecret = this.getSecretAccount(namespace, account.getMetadata().getName());// 找到这个帐号的secret,里面有CA和token
		String proKubeConfig = this.proKubeConfig(clusterName, accountSecret);
		Result suc = Result.getSuc();
		suc.setRetObjs(proKubeConfig);
		return suc;
	}

	/***
	 * 生成kubeconfig文件,测试通过
	 * 
	 * @param secret
	 * @param clusterName 集群的名称
	 * @return
	 */
	public String proKubeConfig(String clusterName, Secret secret) {
//		Config conifg = new Config();
//		List<NamedCluster> clusters = new ArrayList<NamedCluster>();
//		Cluster cluster = new Cluster();
//		cluster.setServer("https://192.168.21.9:1025");
//		cluster.setCertificateAuthorityData("");
//		clusters.add(new NamedCluster(cluster, "k8s-idc-dev"));
//		conifg.setClusters(clusters);
		try {
			Map<String, Object> params = new HashMap<String, Object>();
			params.put("cluster_certificate_authority_data", secret.getData().get("ca.crt"));
			params.put("cluster_server", this.client.getMasterUrl());
			// params.put("cluster_name",
			// this.client.getConfiguration().getCurrentContext().getContext().getCluster());//
			// 指定这个集群的名字,用连接的集群名字来定
			params.put("cluster_name", clusterName);// 指定这个集群的名字,用连接的集群名字来定
			params.put("user_name", secret.getMetadata().getAnnotations().get("kubernetes.io/service-account.name"));
			params.put("namespace", secret.getMetadata().getNamespace());
			params.put("context_name", clusterName);// 把当前配置的context当目标的context
			params.put("user_token", new String(PwdUtil.fromBase64(secret.getData().get("token"))));
			String slurp = IOUtil.slurp(IOUtil.fileToInputStream("/template/configSingle.yaml", KubeClientTams.class));
			String doProcessByTemp = FreemarkUtil.getInst().doProcessByTemp(slurp, params);
			return doProcessByTemp;
		} catch (IOException e) {
			log.error("生成config失败", e);
			return null;
		}
	}

	public VersionInfo getKubernetesVersion() {
		VersionInfo version = client.getKubernetesVersion();
		return version;
	}

	/**
	 * 创建k8s Ingress,兼容旧版本需要使用4.9.2版本
	 *
	 * @param namespace   命名空间
	 * @param ingressName ingress名称
	 * @param annotations ingress注解
	 * @param path        匹配的路径
	 * @param servicename 路由到的服务名称
	 * @param servicePort 路由到的服务端口
	 * @param host        注册的主机
	 * @param ref         主对象
	 * @param tls         启用的ssl secret
	 * @return 创建成功的ingress对象
	 */
	public V1Ingress createV1Ingress(String namespace, String ingressName, Map<String, String> annotations, String path,
			String servicename, Integer servicePort, String host, V1OwnerReference ref, V1IngressTLS tls) {
		V1Ingress ingress = new V1Ingress();
		V1ObjectMeta v1ObjectMeta = new V1ObjectMeta();
		v1ObjectMeta.setName(ingressName);
		v1ObjectMeta.setAnnotations(annotations);
		if (ref != null) {
			v1ObjectMeta.addOwnerReferencesItem(ref);
		}
		ingress.metadata(v1ObjectMeta);

		V1IngressSpec spec = new V1IngressSpec();
		V1IngressRule rulesItem = new V1IngressRule();
		if (StringUtil.isNotNull(host)) {
			rulesItem.setHost(host);
		}
		V1HTTPIngressRuleValue http = new V1HTTPIngressRuleValue();
		V1HTTPIngressPath ingressPath = new V1HTTPIngressPath();
		ingressPath.path(path);
		ingressPath.pathType("ImplementationSpecific");
		V1IngressBackend backend = new V1IngressBackend();
		V1IngressServiceBackend service = new V1IngressServiceBackend();
		service.setName(servicename);
		V1ServiceBackendPort v1ServiceBackendPort = new V1ServiceBackendPort();
		v1ServiceBackendPort.setNumber(servicePort);
		service.setPort(v1ServiceBackendPort);
		backend.setService(service);
		ingressPath.setBackend(backend);
		if (tls != null) {
			spec.addTlsItem(tls);
		}
		http.addPathsItem(ingressPath);
		rulesItem.setHttp(http);

		spec.addRulesItem(rulesItem);
		ingress.setSpec(spec);

		// 调用对应的API执行创建ingress的操作
		NetworkingV1Api api = new NetworkingV1Api(apiClient);
		V1Ingress v1Ingress = null;
		try {
			APIcreateNamespacedIngressRequest namespacedIngress = api.createNamespacedIngress(namespace, ingress);
			v1Ingress = namespacedIngress.execute();
		} catch (ApiException e) {
			log.error("创建ingress异常:" + e.getResponseBody(), e);
		} catch (Exception E) {
			log.error("创建ingress系统异常:", E);
		}
		return v1Ingress;
	}

	// 数字转换
	private static int numInfo(Quantity quantity) {
		int amount = Integer.parseInt(quantity.getAmount());
		if ("k".equalsIgnoreCase(quantity.getFormat())) {
			amount = amount * 1000;
		}
		return amount;
	}

	public static int numInfo(String quantityStr) {
		int indexOf = quantityStr.indexOf("k");
		Quantity params = indexOf < 0 ? new Quantity(quantityStr)
				: new Quantity(quantityStr.substring(0, indexOf), "k");
		return memInfo(params);
	}

	private static int cpuInfo(Quantity quantity) {
		int amount = Integer.parseInt(quantity.getAmount());
		if (!"m".equalsIgnoreCase(quantity.getFormat())) {
			amount = amount * 1000;
		}
		return amount;
	}

	public static int cpuInfo(String quantityStr) {
		int indexOf = quantityStr.indexOf("m");
		Quantity params = indexOf < 0 ? new Quantity(quantityStr)
				: new Quantity(quantityStr.substring(0, indexOf), "m");
		return memInfo(params);
	}

	private static int memInfo(Quantity quantity) {
		int amount = Integer.parseInt(quantity.getAmount());
		if ("Gi".equalsIgnoreCase(quantity.getFormat())) {
			amount = amount * 1024;
		} else if ("Ti".equalsIgnoreCase(quantity.getFormat())) {
			amount = amount * 1024 * 1024;
		}
		return amount;
	}

	public static int memInfo(String quantityStr) {
		return memInfo(new Quantity(quantityStr.substring(0, quantityStr.length() - 2),
				quantityStr.substring(quantityStr.length() - 2)));
	}

	@SuppressWarnings("unchecked")
	public <T extends HasMetadata, L extends KubernetesResourceList<HasMetadata>, R extends Resource<T>> List<T> queryResByPage(
			ResourcesType resourcesType, SqlParserAssit sqlParserTams, int pageSize, int pageNo) {
		long fetchNum = 10L;
		Validate.isTrue(pageSize >= fetchNum, "页面大小必须大于10");
		MixedOperation<T, L, R> operater = null;
		TbSimple tbSimple = sqlParserTams.findTbInfos().get(0);// 只支持一张表情况
		// ResourcesType resourcesType = ResourcesType.find(tbSimple.getTb());
		switch (resourcesType) {
		case POD:
			operater = (MixedOperation) this.client.pods();
			break;
		case SERVICE:
			operater = (MixedOperation) this.client.services();
			break;
		case DEPLOYMENT:
			operater = (MixedOperation) this.client.apps().deployments();
			break;
		case CRD:// 自定义资源走这条路
			ICrdDef crdVesion = cusCache.get(tbSimple.getTb());
			operater = (MixedOperation) this.client.genericKubernetesResources(crdVesion.getCrd());
			break;
		default:
			break;
		}
		if (operater == null) {
			throw new ProjectExceptionRuntime(ExceptAll.Project_default, "不支持的类型" + resourcesType.name());
		}
		NonNamespaceOperation<T, L, R> inNamespace = operater
				.inNamespace(StringUtil.isNull(tbSimple.getDb()) ? this.getDefaultNamespace() : tbSimple.getDb());
		List<T> retlist = new ArrayList<T>();
		List<SqlWherePo> queryWhere = sqlParserTams.queryWhere();
		if (CollectionUtils.isNotEmpty(queryWhere)) {
			List<SqlWherePo> queryWheres = sqlParserTams.queryWhere();
			SqlWherePo nameEqual = null;
			SqlWherePo label = null;
			for (SqlWherePo sqlWherePo : queryWheres) {
				if ("name".equalsIgnoreCase(sqlWherePo.getColName()) && "=".equals(sqlWherePo.getOperator())) {
					nameEqual = sqlWherePo;
					break;
				} else if ("label".equalsIgnoreCase(sqlWherePo.getColName()) && "=".equals(sqlWherePo.getOperator())) {
					label = sqlWherePo;
					break;
				}
			}
			if (nameEqual != null) {// 名称过滤第一位
				T t = inNamespace.withName(nameEqual.getValue()).get();
				retlist.add(t);
			} else if (label != null) {// label过滤第二位。
				Map<String, String> filters = new HashMap<String, String>();
				String[] lableArys = label.getValue().split(",");
				for (String lableAry : lableArys) {
					String[] lableVals = lableAry.split(":");
					filters.put(StringUtil.trimSpace(lableVals[0]), StringUtil.trimSpace(lableVals[1]));
				}
				FilterWatchListDeletable<T, L, R> withLabels = inNamespace.withLabels(filters);
				doQueryByPage(pageSize, pageNo, fetchNum, withLabels, retlist);
			} else {
				doQueryByPage(pageSize, pageNo, fetchNum, inNamespace, retlist);
			}
		} else {
			doQueryByPage(pageSize, pageNo, fetchNum, inNamespace, retlist);
		}
		return retlist;

	}

	@SuppressWarnings({ "unchecked" })
	private <T extends HasMetadata, L extends KubernetesResourceList<HasMetadata>, R extends Resource<T>> void doQueryByPage(
			int pageSize, int pageNo, long fetchNum, FilterWatchListDeletable<T, L, R> inNamespace, List<T> retlist) {
		String _continue = null;
		ListOptions listOptions = new ListOptionsBuilder().withLimit(fetchNum).withContinue(_continue).build();
		long hasQueryNum = 0;
		do {
			L list = inNamespace.list(listOptions);
			_continue = list.getMetadata().getContinue();
			if (hasQueryNum >= pageNo * pageSize) {
				retlist.addAll((List<T>) list.getItems());
			}
			hasQueryNum += fetchNum;
			listOptions.setContinue(_continue);
		} while (StringUtil.isNotNull(_continue) || hasQueryNum >= (pageNo + 1) * pageSize);
	}

	public <T extends HasMetadata, L extends KubernetesResourceList<HasMetadata>, R extends Resource<T>> List<T> queryAll(
			FilterWatchListDeletable<T, L, R> filter) {
		List<T> retlist = new ArrayList<T>();
		doQueryByPage(Integer.MAX_VALUE, 0, 0L, filter, retlist);
		return retlist;
	}

	///////////////////////////////////////////////////////////////////////
	private volatile boolean state = false;

	public synchronized void close() {
		if (!state) {
			this.client.close();
			this.state = true;
		}
	}

	public List<String[]> statementsByPage(ResourcesType resourcesType, SqlParserAssit sqlParserTams, int curPage,
			long maxSize) {
		List<HasMetadata> queryResByPages = queryResByPage(resourcesType, sqlParserTams, (int) maxSize, curPage);
		List<String[]> retlist = new ArrayList<String[]>();
		List<KubernetesResultCol> sortCols = resourcesType.getSortCol(
				resourcesType == ResourcesType.CRD ? this.getCrdColDef(sqlParserTams.findTbInfos().get(0).getTb())
						: resourcesType.getJdbcColDef(),
				sqlParserTams.findColNames().toArray(new String[sqlParserTams.findColNames().size()]));
		for (HasMetadata queryResByPage : queryResByPages) {
			String[] ele = resourcesType.getColValue(queryResByPage, sortCols);
			retlist.add(ele);
		}
		List<SqlWherePo> queryWhere = sqlParserTams.queryWhere();
		if (CollectionUtils.isNotEmpty(queryWhere)) {
			CollectionUtils.filter(queryWhere, new Predicate() {
				@Override
				public boolean evaluate(Object object) {
					SqlWherePo temp = (SqlWherePo) object;
					return !("name".equalsIgnoreCase(temp.getColName()) && "=".equals(temp.getOperator()))
							&& !("label".equalsIgnoreCase(temp.getColName()) && "=".equals(temp.getOperator()));
				}
			});

			// 其它硬过滤
			if (CollectionUtils.isNotEmpty(queryWhere)) {
				CollectionUtils.filter(retlist, new Predicate() {
					@Override
					public boolean evaluate(Object object) {
						String[] ele = (String[]) object;
						boolean need = true;
						for (SqlWherePo sqlWherePo : queryWhere) {
							int indexOf = sortCols.indexOf(new KubernetesResultCol(sqlWherePo.getColName()));
							if ("=".equals(sqlWherePo.getOperator())) {
								need = need & sqlWherePo.getValue().equalsIgnoreCase(ele[indexOf]);
							} else if ("like".equalsIgnoreCase(sqlWherePo.getOperator())) {
								Pattern javaPattern = Pattern
										.compile(sqlWherePo.getValue().replace("%", ".*").replace("_", ".?"));
								need = need & javaPattern.matcher(ele[indexOf]).matches();
							}
						}
						return need;
					}
				});
			}
		}
		return retlist;
	}

	public boolean isClosed() {
		return this.state;
	}

	public void setDefaultNamespace(String namespace) {
		this.defaultNamespace = namespace;
	}

	public String getDefaultNamespace() {
		return this.defaultNamespace;
	}

	public Set<String> getGroups() {
		return groups;
	}

}
